﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace monochrome
{
        public class CharSizeCalculator {

        public static LinkedList<string> SplitString(string s) {
            return TextUtils.CharsInString( s );
        }
        public CharSizeCalculator(Font font) {
            m_font = font;
            m_padding = MeasureFontPaddingWidth(m_font);
            m_sizes = new Dictionary<string,Size>();
        }
        
        public Size WidthOfCharPadded(string c) {
            Size s;
            if (m_sizes.TryGetValue(c, out s)) return s;
            s = TextRenderer.MeasureText(c.ToString(),m_font,new Size(int.MaxValue,int.MaxValue),TextLine.textFormatFlags);
            m_sizes.Add(c,s);
            return s;
        }

        public Size WidthOfChar(string strChar) {
            Size s = WidthOfCharPadded(strChar);
            s.Width = Math.Max(s.Width - m_padding,0);
            return s;
        }
        public Size WidthOfStringPadded(string s) {
            if (s.Length > 0) {
                // return SizeMerge(Get(s.Substring(0,s.Length - 1)),GetPadded(s[s.Length-1]));
                Size merge = new Size( 0, 0 );
                int delta = 0;
                foreach ( string c in SplitString(s) ) {
                    Size size = WidthOfCharPadded(c);
                    int newWidth = Math.Max(size.Width-m_padding,0);
                    delta = size.Width - newWidth;
                    size.Width = newWidth;
                    merge = SizeMerge(merge, size );
                }
                merge.Width += delta;
                return merge;
            } else {
                return new Size(0,0);
            }
        }
        public Size WidthOfString(string s) {
            Size acc = new Size(0,0);
            
            foreach(string c in SplitString(s)) acc = SizeMerge(acc,WidthOfChar(c));
            return acc;
        }
        public static Size SizeMerge(Size s1, Size s2) {
            return new Size(s1.Width + s2.Width, Math.Max(s1.Height,s2.Height));
        }

        public int Padding { get { return m_padding;} }
        public Font Font {get {return m_font;} }

        static Size __MeasureText(char c,Font font) {
            return __MeasureText(c.ToString(),font);
        }
        static Size __MeasureText(string text,Font font) {
            return TextRenderer.MeasureText(text,font,new Size(int.MaxValue,int.MaxValue),TextLine.textFormatFlags);
        }

        static int MeasureFontPaddingWidth(Font font) {
            Size val1 = __MeasureText("#",font);
            Size val2 = __MeasureText("##",font);
            return val1.Width - (val2.Width - val1.Width);
        }

        int m_padding;
        Font m_font;
        Dictionary<string,Size> m_sizes;
    };

    public class FontArray : IDisposable {
        public FontArray(Font fontBase) {
            Reinitialize(fontBase);
        }
        public void Reinitialize(Font fontBase) {
            Dispose();
            m_data = new Dictionary<FontStyle,CharSizeCalculator>();
            m_data.Add(0,new CharSizeCalculator((Font)fontBase.Clone()));
        }
        public CharSizeCalculator GetCalc(FontStyle flags) {
            CharSizeCalculator calc;
            if (!m_data.TryGetValue(flags,out calc)) {
                CharSizeCalculator baseCalc;
                if (!m_data.TryGetValue(0,out baseCalc)) throw new InvalidOperationException();
                Font baseFont = baseCalc.Font;
                Font newFont = new Font(baseFont,baseFont.Style ^ flags);
                calc = new CharSizeCalculator(newFont);
                m_data.Add(flags,calc);
            }
            return calc;
        }
        public Font GetFont(FontStyle flags) {
            return GetCalc(flags).Font;
        }
        public Font BaseFont {
            get { return GetFont(0); }
        }
        public void Dispose() {
            if (m_data != null) {
                foreach(CharSizeCalculator calc in m_data.Values) {
                    calc.Font.Dispose();
                }
                m_data = null;
            }
        }

        Dictionary<FontStyle,CharSizeCalculator> m_data;
    };

    public partial class ColorUtils {
        public static Color RandomColor(int seed,double saturation, double luminance) {
            return ColorFromRGB(RandomColorRGB( seed, saturation, luminance ));
        }

        public static bool LooksLikeBlackBackground(Color col) {
            return (int)col.R + (int)col.G + (int)col.B < 128;
        }

        public static Color ColorFromRGB(RGB rgb) {
            return ColorDesc.FromRGB( rgb ).PlatformColor;
        }
    }


    public partial class TextLine
    {
        public const TextFormatFlags textFormatFlags = TextFormatFlags.Left | TextFormatFlags.NoPrefix | TextFormatFlags.SingleLine;

        public void DrawSimple(Graphics target, FontArray fonts, int X, int Y, TextStyle.ColorResolver colorResolver) {
            int XWalk = X;
            foreach(TextLineElem elem in Elems) {
                CharSizeCalculator calc = fonts.GetCalc(elem.Style.FontStyle);

                Size s = calc.WidthOfStringPadded(elem.Text);

                Rectangle rect = new Rectangle(XWalk, Y, int.MaxValue, s.Height );

                ColorPair colors = elem.ResolveColor(colorResolver);

                TextRenderer.DrawText(target, elem.Text, calc.Font, rect, colors.Foreground, colors.Background,textFormatFlags);

                XWalk += Math.Max(0, s.Width - calc.Padding);
            }
        }

        public void DrawEx(Graphics target, FontArray fonts, int X, int Y, int widthLimit, TextStyle.ColorResolver colorResolver, LinkedList<WordSpecs> words) {
            foreach(LineSplitEntry line in SplitLines(words,widthLimit)) {
                //System.Drawing.Drawing2D.GraphicsState state = target.Save();
                //target.IntersectClip(new Rectangle(X,Y, widthLimit, line.Height));
                Substring(line.CharBase,line.CharCount).DrawSimple(target,fonts,X,Y + line.YOffset, colorResolver);
                //target.Restore(state);
            }
        }
        public LinkedList<WordSpecs> GetWords(FontArray fonts) {
            LinkedList<WordSpecs> output = new LinkedList<WordSpecs>();

            int baseChar = 0;
            int walkChar = 0;
            LinkedListNode<TextLineElem> walkElem = Elems.First;
            while(walkElem != null) {
                int wordWidth = 0, wordHeight = 0;
                int spacing = 0;
                bool inSpacing = false;
                int charsDelta = 0;
                string wordString = "";
                LinkedList<CharSpecs> curWordChars = new LinkedList<CharSpecs>();
                int lastPadding = 0;
                while(walkElem != null) {
                    CharSizeCalculator calc = fonts.GetCalc(walkElem.Value.Style.FontStyle);
                    lastPadding = calc.Padding;

                    int wordDelta = inSpacing ? ExtractSpacing(walkElem.Value.Text,walkChar) : ExtractWord(walkElem.Value.Text,walkChar);
                    string substring = walkElem.Value.Text.Substring(walkChar,wordDelta);
                    Size s = calc.WidthOfStringPadded(substring);
                    charsDelta += wordDelta;
                    int widthDelta = Math.Max(0,s.Width - calc.Padding);
                    if (inSpacing) spacing += widthDelta;
                    else {
                        wordWidth += widthDelta; wordString += substring;
                        foreach( string c in CharSizeCalculator.SplitString(substring) ) {
                            Size cs = calc.WidthOfCharPadded(c);
                            curWordChars.AddLast( new CharSpecs( Math.Max(0, cs.Width - calc.Padding), cs.Width , cs.Height ) );
                        }
                    }
                    wordHeight = Math.Max(wordHeight,s.Height);

                    walkChar += wordDelta;
                    if (walkChar == walkElem.Value.Text.Length) {
                        walkChar = 0; walkElem = walkElem.Next;
                    } else {
                        if (inSpacing) break;
                        else inSpacing = true;
                    }
                }

                WordSpecs ws = new WordSpecs();
                ws.BaseChar = baseChar;
                ws.Height = wordHeight;
                ws.Width = wordWidth;
                ws.Spacing = spacing;
                ws.WidthHT = wordWidth + lastPadding;
                ws.CharSpecs = ArrayHelper.Arrayify(curWordChars);
                
                output.AddLast(ws);

                baseChar += charsDelta;
            }

            return output;
        }

        public int PointToCharacter_SingleLine(int X,FontArray fonts) {
            int xWalk = 0;
            int charIndex = 0;
            foreach(TextLineElem elemWalk in Elems) {
                CharSizeCalculator calc = fonts.GetCalc(elemWalk.Style.FontStyle);
                foreach( string c in CharSizeCalculator.SplitString(elemWalk.Text)) {
                    int xDelta = calc.WidthOfChar(c).Width;
                    if (X < xWalk + xDelta + calc.Padding / 2) return charIndex;
                    xWalk += xDelta;
                    charIndex += c.Length;
                }
            }
            return charIndex;
        }
        public int PointToCharacter(Point pt,FontArray fonts,int targetWidth,LinkedList<WordSpecs> words) {
            foreach(LineSplitEntry entry in SplitLines(words,targetWidth)) {
                if (new Rectangle(0,entry.YOffset,targetWidth,entry.Height).Contains(pt)) {
                    return entry.CharBase + Substring(entry.CharBase,entry.CharCount).PointToCharacter_SingleLine(pt.X,fonts);
                }
            }
            return Length;
        }
    }

    public partial struct TextStyle
    {
        public FontStyle FontStyle {
            get {
                FontStyle fs = 0;
                if (Bold) fs ^= FontStyle.Bold;
                if (Italic) fs ^= FontStyle.Italic;
                if (Underline) fs ^= FontStyle.Underline;
                if (StrikeThrough) fs ^= FontStyle.Strikeout;
                return fs;
            }
        }

        public Font AlterFont(Font baseFont) {
            FontStyle fs = FontStyle;
            if (fs != 0) {
                return new Font(baseFont,baseFont.Style ^ FontStyle);
            } else {
                return baseFont;
            }
        }

        public ColorPair Resolve(ColorResolver resolver,object tag) {
            return resolver(this,tag);
        }

        public delegate ColorPair ColorResolver(TextStyle style, object tag);
    }

    public partial struct TextLineElem {
        public ColorPair ResolveColor(TextStyle.ColorResolver resolver) { return Style.Resolve(resolver,Tag); }
    }

    public struct ColorPair {
        public ColorPair(Color foreground) : this(foreground,Color.Transparent) {}
        public ColorPair(Color foreground, Color background) {Foreground = foreground; Background = background;}
        public Color Foreground, Background;
    };
}