using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows.Forms;

namespace monochrome
{
    

    public partial class ServerSetupDialog : SwitchableChildWindow
    {
        public ServerSetupDialog() : this(null) {}
        public ServerSetupDialog(MainWindow _owner) : base(_owner)
        {
            Text = "Server Setup";
            removedPresets = new LinkedList<string>();
            InitializeComponent();
            Show();

            foreach(string n in ServerConfigManager.ServerNames()) {
                presetList.Items.Add(n);
            }

            if (presetList.Items.Count > 0) {
                presetList.Text = ServerConfigManager.DefaultServer();
                presetList.Focus();
            } else {
                presetList.Text = ServerConfigManager.DefaultServerStub();
                textBoxHostName.Focus();
            }

            ReloadPreset();
        }

        void ReloadPreset() {
            {
                ServerParams preset;

                try {
                    preset = ServerConfigManager.GrabServerParamsByName(presetList.Text);
                } catch {
                    preset = ServerParams.Defaults;
                }
                setParams(preset,presetList.Text);
            }

            m_initialized = true;
            validateSetup();
        }

        private void ButtonCancel_Click(object sender, EventArgs e) {
            Close();
        }

        private void ButtonOK_Click(object sender, EventArgs e) {
            OnOK(true);
        }

        void setParams(ServerParams value,string name) {
            m_lastParams = value;
            m_lastParamsName = name;
            textBoxHostName.Text = value.hostName;
            textBoxPort.Text = Convert.ToString(value.hostPort);
            textBoxEncoding.Text = value.charEncodingName;
            textBoxNick.Text = value.nicks;
            textBoxNickPassword.Text = value.nickPassword;
            textBoxNickAccount.Text = value.nickAccount;
            textBoxAutoJoin.Text = value.autoJoinChannels;
            textBoxUserModes.Text = value.userModes;
            checkBoxAutoConnect.Checked = value.autoConnect;
            textBoxServerUser.Text = value.serverUser;
            textBoxServerPassword.Text = value.serverPassword;
            checkBoxSSL.Checked = value.useSSL;
            comboBoxNicksInUse.SelectedIndex = value.allowRandomNick ? 1 : 0;
            inviteHandlers.Text = value.inviteHandlers;
            extraStartupCommands.Text = value.extraStartupCommands;
        }
        ServerParams grabParams() {
            ServerParams svParams = m_lastParams;
            svParams.hostName = textBoxHostName.Text.Trim();
            try {
                svParams.hostPort = Convert.ToInt32(textBoxPort.Text);
            } catch(FormatException) {
                svParams.hostPort = 0;
            } catch(OverflowException) {
                svParams.hostPort = 0;
            }
            svParams.charEncodingName = textBoxEncoding.Text;
            svParams.nicks = textBoxNick.Text; 
            svParams.nickPassword = textBoxNickPassword.Text;
            svParams.nickAccount = textBoxNickAccount.Text;
            svParams.autoJoinChannels = textBoxAutoJoin.Text;
            svParams.userModes = textBoxUserModes.Text;
            svParams.autoConnect = checkBoxAutoConnect.Checked;
            svParams.serverUser = textBoxServerUser.Text;
            svParams.serverPassword = textBoxServerPassword.Text;
            svParams.useSSL = checkBoxSSL.Checked;
            svParams.allowRandomNick = comboBoxNicksInUse.SelectedIndex > 0;
            svParams.inviteHandlers = inviteHandlers.Text;
            svParams.extraStartupCommands = extraStartupCommands.Text;
            
            return svParams;
        }

        void validateSetup() {
            bool isValid = this.isValid();
            ButtonOK.Enabled = isValid;
            buttonSave.Enabled = removedPresets.Count > 0 || (isValid && hasChanged);
        }

        bool hasChanged {
            get {
                return (m_lastParamsName != presetList.Text || m_lastParams != grabParams());
            }
        }

        static bool isValidPresetName(string presetName) {
            if (presetName.Length == 0) return false;
            if (presetName.IndexOfAny(System.IO.Path.GetInvalidFileNameChars()) >= 0) return false;
            return true;
        }

        bool isValid() {
            if (!m_initialized)  return false;
            try {
                if (!isValidPresetName(presetList.Text)) return false;
                return grabParams().isValid();
            } catch {
                return false;
            }
        }

        void OnOK(bool connect) {
            ServerConfigManager.RemovePresets ( removedPresets );

            if (!isValidPresetName(presetList.Text)) return;
            

            ServerParams svParams;
            try {
                svParams = grabParams();
            } catch(Exception ex) {
                new PopupMessage(ownerMainWnd,ex.Message,"Error");
                return;
            }
            if (!svParams.isValid()) {
                return;
            }
            
            try {
                if (hasChanged) {
                    ServerConfigManager.StoreServerParamsByName(presetList.Text,svParams);
                    m_lastParams = svParams;
                    m_lastParamsName = presetList.Text;
                }

                
                ServerConfigManager.SetDefaultServer(presetList.Text);
            } catch(Exception ex) {
                Debug.WriteLine(ex.Message);
            }
            if (!presetList.Items.Contains(presetList.Text)) presetList.Items.Add(presetList.Text);

            if (connect) {
                string netName = presetList.Text;

                Close();

                if (ownerMainWnd.FindExistingNetInstance(netName) != null) {
                    (new ConnectionExistsDialog(ownerMainWnd,netName,svParams)).SwitchTo();
                } else {
                    (new ServerWindow(ownerMainWnd,svParams,netName)).SwitchTo();
                }
            } else {
                validateSetup();
            }
        }

        void KeyPressFilter(object sender, KeyPressEventArgs e) {
            if (e.KeyChar == '\r') {
                e.Handled = true;
                if (ButtonOK.Enabled) {
                    OnOK(true);
                }
            } else if (e.KeyChar == '') {
                e.Handled = true;
                Close();
            }
        }

        void OnDispose() {
            ownerMainWnd.SetupDialogClosing();
            if (m_hostNameToolTip != null)
            {
                m_hostNameToolTip.Dispose();
                m_hostNameToolTip = null;
            }
        }


        private void OnDataChanged(object sender, EventArgs e) {
            validateSetup();
        }

        private void presetList_SelectedIndexChanged(object sender, EventArgs e) {
            ReloadPreset();
        }

        LinkedList<string> removedPresets;


        private void removeToolStripMenuItem_Click(object sender, EventArgs e) {
            int index = presetList.SelectedIndex;
            if (index >= 0) {
                removedPresets.AddLast(presetList.SelectedItem.ToString());
                presetList.Items.RemoveAt(index);
                validateSetup();
            }
        }

        private void presetList_TextChanged(object sender, EventArgs e) {
            validateSetup();
        }

        private void buttonSave_Click(object sender, EventArgs e) {
            OnOK(false);
        }


        bool m_initialized;
        ServerParams m_lastParams; string m_lastParamsName;

        static bool isValidPort(string text) {
            try {
                int port = Convert.ToInt32(text);
                return port > 0 && port <= 0xFFFF;
            } catch(FormatException) {return false;}
            catch(OverflowException) {return false;}
        }

        private void textBoxPort_Validating(object sender, CancelEventArgs e) {
            if (!isValidPort(textBoxPort.Text)) {
                MessageBox.Show(this,"Please enter a valid port number.","Invalid Input");
                e.Cancel = true;
            }
        }

        private void checkBoxSSL_CheckedChanged(object sender, EventArgs e)
        {
            textBoxPort.Text = checkBoxSSL.Checked ? "6697" : "6667";
            OnDataChanged(sender, e);
        }

        private void textBoxHostName_TextChanged(object sender, EventArgs e)
        {
            string host = textBoxHostName.Text.ToLower();
            if (host == "freenode.net" || host.EndsWith(".freenode.net") ) {
                hostNameTip("This is not the freenode you're looking for.");
            } else {
                hostNameTip(null);
            }
            OnDataChanged(sender, e);
        }

        private void hostNameTip(string str)
        {
            if ( str == null )
            {
                if (m_hostNameToolTip != null) m_hostNameToolTip.Hide(textBoxHostName);
            }
            else
            {
                if (m_hostNameToolTip==null) m_hostNameToolTip = new ToolTip();
                m_hostNameToolTip.Show( str, textBoxHostName );
            }
        }

        private ToolTip m_hostNameToolTip;
    }
}
