using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Xml;
using System.IO;


namespace monochrome
{
    public partial class Preferences : SwitchableChildWindow
    {
        void addColorItem(object tag, string label, Color initVal, bool enabled) {
            ListViewItem item = new ListViewItem();
            item.Text = label;
            item.Tag = tag;
            item.Checked = enabled;
            ListViewItem.ListViewSubItem subitem = new ListViewItem.ListViewSubItem();
            item.UseItemStyleForSubItems = false;
            subitem.BackColor = initVal;
            item.SubItems.Add(subitem);
            listViewColors.Items.Add(item);
        }

        public Preferences(MainWindow _ownerWnd) : base(_ownerWnd) {
            this.Text = "Preferences";
            m_dataBackup = m_data = PreferencesManager.Current;
            InitializeComponent();
            defaultMessagesPicker.SelectedIndex = 0;
            if (m_data.fullName != null) textBox_fullName.Text = m_data.fullName;
            buttonFont.Font = m_data.Font;
            buttonFont.Text = m_data.fontName;
            buttonFontLists.Font = m_data.ListFont;
            buttonFontLists.Text = m_data.listFontName;
            portRangeBox.Text = m_data.xferPortRange;
            checkBoxXferAskForLocation.Checked = m_data.xferAskForFileLocation;
            checkBoxXferHideOutgoing.Checked = m_data.xferHideOutgoing;
            textBoxDefaultXferDir.Text = m_data.xferDefaultDirectory;
            textBoxAutoAcceptByChannel.Text = m_data.xferAAChannels;
            textBoxAutoAcceptByUser.Text = m_data.xferAAUsers;
            textBoxIgnore.Text = m_data.ignore;
            textBoxHalfIgnore.Text = m_data.halfIgnore;

            checkBoxShowColors.Checked = m_data.displayColors;
            checkBoxShowFormatting.Checked = m_data.displayFormatting;

            checkBoxBeep.Checked = m_data.beepOnTrigger;

            checkBoxLogging.Checked = m_data.useLogging;
            textBoxLoggingDirectory.Text = m_data.loggingPath;
            textBoxTimestampFormat.Text = m_data.timestampFormat;
            textBoxLoggingTimestampFormat.Text = m_data.loggingTimestampFormat;

            checkBoxSplitLogs.Checked = m_data.splitLogsByDays;

            textBoxTriggerWords.Text = m_data.triggerWords;
            checkBoxTriggersCaseSensitive.Checked = m_data.triggerWordsCaseSensitive;
            checkBoxUseIdent.Checked = m_data.useIdent;
            textBoxIdentName.Text = m_data.identName;

            textBoxSplitLines.Text = (m_data.splitLines > 0) ? m_data.splitLines.ToString() : "";
            textBoxBacklogLimit.Text = m_data.lineCountLimit.ToString();
            checkBoxAutoUnaway.Checked = m_data.unAwayOnActivity;
            checkBoxAwayOnMinimize.Checked = m_data.awayOnMinimize;
            checkBoxAwayOnLock.Checked = m_data.awayOnLock;
            checkBoxTriggerLog.Checked = m_data.useTriggerLog;
            checkBoxMessageLog.Checked = m_data.useMessageLog;
            checkBoxFlash.Checked = m_data.flashOnTrigger;
            checkBoxNickColors.Checked = m_data.useNickColors;

            hideMessageTypes.Text = m_data.hideMessageTypes;
            outgoingMessageSecurity.Checked = m_data.outgoingMsgSecurity;
            

            radioButtonURLOpen.Checked = !m_data.defaultURLCopy;
            radioButtonURLCopy.Checked = m_data.defaultURLCopy;

            addColorItem(0,"Text",m_data.ColorText,!m_data.colorText.IsEmpty);
            addColorItem(1,"Background",m_data.ColorBackground,!m_data.colorBackground.IsEmpty);
            addColorItem(2,"Highlight",m_data.ColorHighlight,!m_data.colorHighlight.IsEmpty);
            addColorItem(3,"Channel highlight (activity)",  m_data.ColorChanHighlight1,!m_data.colorChanHighlight1.IsEmpty);
            addColorItem(4,"Channel highlight (talk)",      m_data.ColorChanHighlight2,!m_data.colorChanHighlight2.IsEmpty);
            addColorItem(5,"Channel highlight (trigger)",   m_data.ColorChanHighlight3,!m_data.colorChanHighlight3.IsEmpty);
            addColorItem(6,"User joins",                    m_data.ColorUserJoin,!m_data.colorUserJoin.IsEmpty);
            addColorItem(7,"User parts",                    m_data.ColorUserPart,!m_data.colorUserPart.IsEmpty);

            textBoxAliases.Text = m_data.aliases;

            m_initialized = true;
            onDataChanged();
        }
        enum Messages {
            Quit = 0,
            Part,
            Away,
            Kick
        }
        Messages editedMessage { get { return (Messages) defaultMessagesPicker.SelectedIndex; } }
        void setMessage(Messages which, string content) {
            switch(which) {
                case Messages.Quit:
                    m_data.defaultQuitMessages = content;
                    onDataChanged();
                    break;
                case Messages.Part:
                    m_data.defaultPartMessages = content;
                    onDataChanged();
                    break;
                case Messages.Away:
                    m_data.defaultAwayMessages = content;
                    onDataChanged();
                    break;
                case Messages.Kick:
                    m_data.defaultKickMessages = content;
                    onDataChanged();
                    break;
            }
        }


        string getMessage(Messages which) {
            string msg = _getMessage(which);
            if (msg == null) msg = "";
            return msg;
        }
        string _getMessage(Messages which) {
            switch(which) {
                case Messages.Quit:     return m_data.defaultQuitMessages;
                case Messages.Part:     return m_data.defaultPartMessages;
                case Messages.Away:     return m_data.defaultAwayMessages;
                case Messages.Kick:     return m_data.defaultKickMessages;
                default:                return null;
            }
        }


        private void defaultMessagesPicker_SelectedIndexChanged(object sender, EventArgs e) {
            defaultMessages_content.Text = getMessage(editedMessage);
        }
        private void defaultMessages_content_TextChanged(object sender, EventArgs e) {
            setMessage(editedMessage,defaultMessages_content.Text);
        }

        private void buttonCancel_Click(object sender, EventArgs e) {
            Close();
        }

        bool _Apply() {
            if (m_data == m_dataBackup) return true;
            if (!m_data.Valid) return false;

            try {
                PreferencesManager.Set(m_data);
                m_dataBackup = m_data;
                onDataChanged();
                return true;
            } catch(Exception e) {
                new PopupMessage(ownerMainWnd,"Could not save preferences: " + e.Message,"Error While Saving Preferences");
                return false;
            }
        }
        void Apply() {
            if (m_data.Valid) _Apply();
        }
        private void buttonOK_Click(object sender, EventArgs e) {
            if (m_data.Valid) {
                if (_Apply()) {
                    Close();
                }
            }
        }


 

        private void Preferences_FormClosed(object sender, FormClosedEventArgs e) {
            if (ownerMainWnd != null) {
                ownerMainWnd.preferencesClosed();
            }
        }

        private void textBox_fullName_TextChanged(object sender, EventArgs e) {
            m_data.fullName = textBox_fullName.Text;
            onDataChanged();
        }


        private void buttonFontLists_Click(object sender, EventArgs e) {
            try {
                fontDialog.Font = m_data.ListFont;
                if (fontDialog.ShowDialog(this) == DialogResult.OK) {
                    buttonFontLists.Font = fontDialog.Font;
                    buttonFontLists.Text = fontDialog.Font.Name;
                    m_data.ListFont = fontDialog.Font;
                    onDataChanged();
                }
            } catch (Exception ex) {
                new PopupMessage(ownerMainWnd, ex.Message, "Font Selection Error");
            }
        }
        private void buttonFont_Click(object sender, EventArgs e) {
            try {
                fontDialog.Font = m_data.Font;
                if (fontDialog.ShowDialog(this) == DialogResult.OK) {
                    buttonFont.Font = fontDialog.Font;
                    buttonFont.Text = fontDialog.Font.Name;
                    m_data.Font = fontDialog.Font;
                    onDataChanged();
                }
            } catch (Exception ex) {
                new PopupMessage(ownerMainWnd, ex.Message, "Font Selection Error");
            }
        }

        private void portRangeBox_TextChanged(object sender, EventArgs e) {
            m_data.xferPortRange = portRangeBox.Text;
            onDataChanged();
        }


        private void checkBoxXferAskForLocation_CheckedChanged(object sender, EventArgs e) {
            m_data.xferAskForFileLocation = checkBoxXferAskForLocation.Checked;
            onDataChanged();
        }

        private void buttonDefaultXferDir_Click(object sender, EventArgs e) {
            using(FolderBrowserDialog dlg = new FolderBrowserDialog()) {
                dlg.SelectedPath = textBoxDefaultXferDir.Text;
                dlg.ShowNewFolderButton = true;
                dlg.Description = "Select default incoming file directory.";
                if (dlg.ShowDialog(this) == DialogResult.OK) {
                    textBoxDefaultXferDir.Text = dlg.SelectedPath;
                }
            }
        }

        private void textBoxDefaultXferDir_TextChanged(object sender, EventArgs e) {
            m_data.xferDefaultDirectory = textBoxDefaultXferDir.Text;
            m_data.xferDefaultDirectoryToken = "";
            onDataChanged();
        }

        private void textBoxAutoAcceptByUser_TextChanged(object sender, EventArgs e) {
            m_data.xferAAUsers = textBoxAutoAcceptByUser.Text;
            onDataChanged();
        }

        private void textBoxAutoAcceptByChannel_TextChanged(object sender, EventArgs e) {
            m_data.xferAAChannels = textBoxAutoAcceptByChannel.Text;
            onDataChanged();
        }

        private void textBoxHalfIgnore_TextChanged(object sender, EventArgs e) {
            m_data.halfIgnore = textBoxHalfIgnore.Text;
            onDataChanged();
        }

        private void textBoxIgnore_TextChanged(object sender, EventArgs e) {
            m_data.ignore = textBoxIgnore.Text;
            onDataChanged();
        }

        private void checkBoxShowColors_CheckedChanged(object sender, EventArgs e) {
            m_data.displayColors = checkBoxShowColors.Checked;
            onDataChanged();
        }

        private void checkBoxShowFormatting_CheckedChanged(object sender, EventArgs e) {
            m_data.displayFormatting = checkBoxShowFormatting.Checked;
            onDataChanged();
        }

        private void buttonApply_Click(object sender, EventArgs e) {
            Apply();
        }

        private void checkBoxBeep_CheckedChanged(object sender, EventArgs e) {
            m_data.beepOnTrigger = checkBoxBeep.Checked;
            onDataChanged();
        }

        bool chooseLoggingDirectory() {
            using(FolderBrowserDialog dialog = new FolderBrowserDialog()) {
                dialog.ShowNewFolderButton = true;
                dialog.Description = "Select Logging Directory";
                dialog.SelectedPath = m_data.loggingPath;
                if (dialog.ShowDialog(this) == DialogResult.OK) {
                    m_data.loggingPathToken = "";
                    textBoxLoggingDirectory.Text = m_data.loggingPath = dialog.SelectedPath;
                    onDataChanged();
                    return true;
                } else {
                    return false;
                }
            }
        }

        private void checkBoxLogging_CheckedChanged(object sender, EventArgs e) {
            bool state = checkBoxLogging.Checked;
            if (state && m_data.loggingPath == "") {
                if (!chooseLoggingDirectory()) {
                    checkBoxLogging.Checked = state = false;
                }
            }   
            m_data.useLogging = state;
            onDataChanged();
        }

        private void buttonLoggingDirectory_Click(object sender, EventArgs e) {
            chooseLoggingDirectory();
        }

        void handleTimestampPreview(Label preview, string value) {
            string previewString;
            try {
                previewString = DateTime.Now.ToString(value);
            } catch(Exception ex) {
                previewString = ex.Message;
            }
            preview.Text = previewString;
        }

        private void textBoxTimestampFormat_TextChanged(object sender, EventArgs e) {
            string value = textBoxTimestampFormat.Text;
            m_data.timestampFormat = value;
            onDataChanged();
            handleTimestampPreview(labelTimestampPreview,value);
        }

        private void textBoxLoggingTimestampFormat_TextChanged(object sender, EventArgs e) {
            string value = textBoxLoggingTimestampFormat.Text;
            m_data.loggingTimestampFormat = value;
            onDataChanged();
            handleTimestampPreview(labelLoggingTimestampPreview,value);
        }

        private void timerTimestampUpdate_Tick(object sender, EventArgs e) {
            handleTimestampPreview(labelTimestampPreview,m_data.timestampFormat);
            handleTimestampPreview(labelLoggingTimestampPreview,m_data.loggingTimestampFormat);
        }

        private void checkBoxSplitLogs_CheckedChanged(object sender, EventArgs e) {
            m_data.splitLogsByDays = checkBoxSplitLogs.Checked;            
            onDataChanged();
        }

        void OnColorItemChanged(ListViewItem item) {
            var val = ColorDesc.FromPlatformColor( item.Checked ? item.SubItems[1].BackColor : Color.Empty );
            switch((int)item.Tag) {
                case 0: m_data.colorText = val; break;
                case 1: m_data.colorBackground = val; break;
                case 2: m_data.colorHighlight = val; break;
                case 3: m_data.colorChanHighlight1 = val; break;
                case 4: m_data.colorChanHighlight2 = val; break;
                case 5: m_data.colorChanHighlight3 = val; break;
                case 6: m_data.colorUserJoin = val; break;
                case 7: m_data.colorUserPart = val; break;
            }
            onDataChanged();
        }

        private void listViewColors_ItemChecked(object sender, ItemCheckedEventArgs e) {
            OnColorItemChanged(e.Item);
        }

        ListViewItem resolveColorEntry(Point pt) {
            //urf
            foreach(ListViewItem item in listViewColors.Items) {
                if (item.SubItems[1].Bounds.Contains(pt)) return item;
            }
            return null;
        }

        private void listViewColors_MouseDown(object sender, MouseEventArgs e) {
            try {
                ListViewItem item = resolveColorEntry(e.Location);
                if (item != null && item.Checked) {
                    using(ColorDialog dialog = new ColorDialog()) {
                        dialog.Color = item.SubItems[1].BackColor;
                        if (dialog.ShowDialog(this) == DialogResult.OK) {
                            item.SubItems[1].BackColor = dialog.Color;
                            OnColorItemChanged(item);
                        }
                    }
                }
            } catch(Exception ex) {
                new PopupMessage(ownerMainWnd,ex.Message,"Color Selection Error");
            }

        }

        private void textBoxTriggerWords_TextChanged(object sender, EventArgs e) {
            m_data.triggerWords = textBoxTriggerWords.Text;
            onDataChanged();
        }

        private void checkBoxTriggersCaseSensitive_CheckedChanged(object sender, EventArgs e)
        {
            m_data.triggerWordsCaseSensitive = checkBoxTriggersCaseSensitive.Checked;
            onDataChanged();
        }

        private void checkBoxUseIdent_CheckedChanged(object sender, EventArgs e) {
            m_data.useIdent = checkBoxUseIdent.Checked;
            onDataChanged();
        }

        private void textBoxIdentName_TextChanged(object sender, EventArgs e) {
            string name = textBoxIdentName.Text;
            if (IRCUtils.Nicks.IsValid(name)) {
                m_data.identName = name;
                onDataChanged();
            }
        }

        private void textBoxIdentName_Validating(object sender, CancelEventArgs e) {
            if (!IRCUtils.Nicks.IsValid(textBoxIdentName.Text)) {
                e.Cancel = true;
                MessageBox.Show("Not a valid user name.");
            }
        }
        void onDataChanged() {
            if (m_initialized) {
                bool valid = m_data.Valid;
                buttonOK.Enabled = valid;
                buttonApply.Enabled = valid && (m_data != m_dataBackup);
            }
        }

        bool m_initialized;
        PreferencesData m_data, m_dataBackup;

        private void textBoxSplitLines_Validating(object sender, CancelEventArgs e) {
            string val = textBoxSplitLines.Text;
            if (val.Length > 0) {
                try {
                    Convert.ToInt32(textBoxSplitLines.Text);
                } catch {
                    MessageBox.Show(this,"Please enter a number.","Invalid Input");
                    e.Cancel = true;
                }
            }
        }

        private void textBoxSplitLines_TextChanged(object sender, EventArgs e) {
            int oldVal = m_data.splitLines;
            try {
                m_data.splitLines = Convert.ToInt32(textBoxSplitLines.Text);
            } catch {
                m_data.splitLines = 0;
            }
            if (oldVal != m_data.splitLines) onDataChanged();
        }

        private void checkBoxAutoUnaway_CheckedChanged(object sender, EventArgs e) {
            m_data.unAwayOnActivity = checkBoxAutoUnaway.Checked;
            onDataChanged();
        }

        private void checkBoxTriggerLog_CheckedChanged(object sender, EventArgs e) {
            m_data.useTriggerLog = checkBoxTriggerLog.Checked;
            onDataChanged();
        }

        private void checkBoxAwayOnMinimize_CheckedChanged(object sender, EventArgs e) {
            m_data.awayOnMinimize = checkBoxAwayOnMinimize.Checked;
            onDataChanged();
        }

        private void checkBoxFlash_CheckedChanged(object sender, EventArgs e) {
            m_data.flashOnTrigger = checkBoxFlash.Checked;
            onDataChanged();
        }

        private void checkBoxAwayOnLock_CheckedChanged(object sender, EventArgs e) {
            m_data.awayOnLock = checkBoxAwayOnLock.Checked;
            onDataChanged();
        }

        private void checkBoxMessageLog_CheckedChanged(object sender, EventArgs e) {
            m_data.useMessageLog = checkBoxMessageLog.Checked;
            onDataChanged();
        }

        private void checkBoxNickColors_CheckedChanged(object sender, EventArgs e) {
            m_data.useNickColors = checkBoxNickColors.Checked;
            onDataChanged();
        }

        private void outgoingMessageSecurity_CheckedChanged(object sender, EventArgs e)
        {
            m_data.outgoingMsgSecurity = outgoingMessageSecurity.Checked;
            onDataChanged();
        }

        private void checkBoxHideOutgoing_CheckedChanged(object sender, EventArgs e)
        {
            m_data.xferHideOutgoing = checkBoxXferHideOutgoing.Checked;
            onDataChanged();
        }

        private void radioButtonURLCopy_CheckedChanged(object sender, EventArgs e) {
            m_data.defaultURLCopy = radioButtonURLCopy.Checked;
            onDataChanged();
        }

        private void textBoxNumber_Validating(object sender, CancelEventArgs e)
        {
            if (!e.Cancel) {
                string val = ((TextBox)sender).Text;
                if (val.Length > 0) {
                    try {
                        Convert.ToInt32(val);
                    } catch {
                        MessageBox.Show(this,"Please enter a number.","Invalid Input");
                        e.Cancel = true;
                    }
                }
            }
        }

        private void textBoxBacklogLimit_TextChanged(object sender, EventArgs e)
        {
            int oldVal = m_data.lineCountLimit;
            try {
                m_data.lineCountLimit = Convert.ToInt32(textBoxBacklogLimit.Text);
            } catch {
                m_data.lineCountLimit = 0;
            }
            if (oldVal != m_data.lineCountLimit) onDataChanged();
        }

        private void Preferences_Load(object sender, EventArgs e)
        {

        }

        private void hideMessageTypes_TextChanged(object sender, EventArgs e) {
            m_data.hideMessageTypes = hideMessageTypes.Text;
            onDataChanged();
        }

        private void textBoxAliases_TextChanged(object sender, EventArgs e)
        {
            m_data.aliases = textBoxAliases.Text;
            onDataChanged();
        }

    }

    public class PreferencesNotifyEntry : IComponent {
        public PreferencesNotifyEntry(IContainer container) : this() {container.Add(this);}
        public PreferencesNotifyEntry() {
            PreferencesManager.Notify += _OnPreferencesChanged;
        }
        void IDisposable.Dispose() {
            PreferencesManager.Notify -= _OnPreferencesChanged;
            if (disposed != null) disposed(this,EventArgs.Empty);
        }

        ISite IComponent.Site { get {return m_site;} set {m_site = value;} }
        
        event EventHandler IComponent.Disposed {
            add {disposed += value;}
            remove {disposed -= value;}
        }

        public delegate void PreferencesChangedHandler(PreferencesData data);

        public event PreferencesChangedHandler PreferencesChanged {
            add {
                _PreferencesChanged += value;
            }
            remove {
                _PreferencesChanged -= value;
            }
        }

        PreferencesChangedHandler _PreferencesChanged;

        EventHandler disposed;

        void _OnPreferencesChanged(PreferencesData data) {
            if (_PreferencesChanged != null) _PreferencesChanged(data);
        }
        ISite m_site;
    }

    partial struct PreferencesData
    {
        public void ApplyColors(Control ctrl) {
            ctrl.ForeColor = ColorText;
            ctrl.BackColor = ColorBackground;
        }
        public void SetPlatformDefaults()
        {
            using (Font defFont = new Font(FontFamily.GenericMonospace, SystemFonts.DefaultFont.Size)) {
                Font = defFont;
                ListFont = defFont;
            }
        }
        public Font ListFont {
            get {
                try {
                    return new Font(listFontName, Single.Parse(listFontSize.Replace(',', '.'), System.Globalization.CultureInfo.InvariantCulture), (FontStyle)Convert.ToInt32(listFontStyle));
                } catch {
                    return SystemFonts.DefaultFont;
                }
            }
            set {
                listFontName = value.Name;
                listFontSize = value.Size.ToString(System.Globalization.CultureInfo.InvariantCulture);
                listFontStyle = ((int)value.Style).ToString();
            }
        }
        public Font Font {
            get {
                try {
                    return new Font(fontName,Single.Parse(fontSize.Replace(',','.'),System.Globalization.CultureInfo.InvariantCulture),(FontStyle) Convert.ToInt32( fontStyle ));
                } catch {
                    return SystemFonts.DefaultFont;
                }
            }
            set {
                fontName = value.Name;
                fontSize = value.Size.ToString(System.Globalization.CultureInfo.InvariantCulture);
                fontStyle = ((int)value.Style).ToString();
            }
        }
    }

    partial class PreferencesManager
    {
        private static string preferencesFilePathCache = null;
        private static string makePreferencesFilePath()
        {
            const string fileName = ConfigFileName;

            string path = System.IO.Path.Combine(Application.StartupPath, fileName);
            if (System.IO.File.Exists(path)) return path;


            string directory =
                System.IO.Path.Combine(System.Environment.GetFolderPath(System.Environment.SpecialFolder.ApplicationData), "Monochrome");

            System.IO.Directory.CreateDirectory(directory);

            return System.IO.Path.Combine(directory, fileName);
        }
        public static string PreferencesFilePath {
            get {
                if (preferencesFilePathCache == null) {
                    preferencesFilePathCache = makePreferencesFilePath();
                }
                return preferencesFilePathCache;
            }
        }
        public static XmlDocument GetPreferencesXML() {
            XmlDocument doc = new XmlDocument();
            try { doc.Load(PreferencesFilePath); } catch { return new XmlDocument(); }
            return doc;
        }
        public static void SetPreferencesXML(XmlDocument doc) {
            Debug.WriteLine("Rewriting " + ConfigFileName + " ...");
            string filePath = PreferencesFilePath;
            string tempPath = filePath + ".tmp";
            doc.Save(tempPath);
            if (System.IO.File.Exists(filePath)) System.IO.File.Delete(filePath);
            System.IO.File.Move(tempPath, filePath);
        }
        private static FileStream profileLock = null;
        public static bool LockProfile() {
            if (profileLock != null) return true;
            string lockPath = PreferencesFilePath + ".lock";
            try {
                profileLock = File.Create(lockPath, 1, FileOptions.DeleteOnClose);
                return true;
            } catch {
                return false;
            }
        }
    };

}