﻿using System;
using System.Net;
using System.Net.Security;
using System.Net.Sockets;
using System.IO;
using System.Security.Cryptography.X509Certificates;
using System.Security.Authentication;
using System.Threading;
using System.Threading.Tasks;

namespace monochrome
{
    struct NetStreamArg {
        public string remoteHost;
        public int remotePort;
        public bool useSSL;
        public bool verifySSL;
    };

    class NetStream : IDisposable
    {
        TcpClient client;
        Stream stream;

        public void Close() {
            if (client != null) {
                try {client.Close();} catch {}
                client = null;
            }
            if (stream != null) {
                try {stream.Close();} catch {}
                stream = null;
            }
        }

        public NetStream()
        {
        }
        private bool RemoteCertificateValidationCallbackImpl(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors) {
            return sslPolicyErrors == SslPolicyErrors.None;
        }
        private bool RemoteCertificateValidationCallbackDummy(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            return true;
        }
        public async Task Accept(IPAddress address, int port, CancellationToken token) {
            var listener = new TcpListener(address, port);
            try {
                listener.Start(1);
                var task = listener.AcceptTcpClientAsync();
                await Task.WhenAny( task, Task.Delay(-1, token) );
                token.ThrowIfCancellationRequested();
                client = task.Result;
                client.NoDelay = true;
                stream = client.GetStream();
            } finally {
                listener.Stop();
            }
        }
        public async Task Open(NetStreamArg arg, CancellationToken token ) {
            IPHostEntry serverHostEntry = await ResolveHostNameInner(arg.remoteHost);
            token.ThrowIfCancellationRequested();
            if (serverHostEntry.AddressList.Length == 0) throw new Exception("DNS failure - empty address list");

            Exception rethrow_me = null;
            foreach( IPAddress serverAddress in serverHostEntry.AddressList)
            {
                try {
                    await openEndPoint(new IPEndPoint(serverAddress, arg.remotePort), arg);
                    return;
                } catch(Exception e) {
                    if (rethrow_me == null) rethrow_me = e;
                }
            }
            throw rethrow_me;
        }

        private async Task openEndPoint(  IPEndPoint endPoint, NetStreamArg arg )
        {
            try {
                client = new TcpClient(endPoint.AddressFamily);
                
                await client.ConnectAsync(endPoint.Address, endPoint.Port);
                client.NoDelay = true;
                stream = client.GetStream();
                if (arg.useSSL) {
                    RemoteCertificateValidationCallback cb;
                    if ( arg.verifySSL ) cb = new RemoteCertificateValidationCallback(RemoteCertificateValidationCallbackImpl);
                    else cb = new RemoteCertificateValidationCallback( RemoteCertificateValidationCallbackDummy);
                    SslStream sslStream = new SslStream(stream,false,cb);
                    stream = sslStream;
                    await sslStream.AuthenticateAsClientAsync( arg.remoteHost );
                }
            } catch {
                this.Close();
                throw;
            }
        }

        /*
        public int Available
        {
            get { return client.Available; }
        }
        public bool Poll( int ms )
        {
            return client.Client.Poll(ms * 1000,SelectMode.SelectError | SelectMode.SelectRead);
        }

        public void Send(byte[] buffer,int offset, int count) {
            stream.Write(buffer,offset,count);
        }
        public int Read(byte[] buffer, int offset, int count) {
            return stream.Read(buffer, offset, count);
        }
        */
        public async Task SendAsync(byte[] buffer,int offset, int count, System.Threading.CancellationToken token) {
            await stream.WriteAsync(buffer, offset, count, token);
        }
        public async Task<int> ReadAsync(byte[] buffer, int offset, int count, System.Threading.CancellationToken token) {
            return await stream.ReadAsync( buffer, offset, count, token );
        }

        void IDisposable.Dispose()
        {
            Close();
        }


        public async static Task<string> ResolveHostName( string inHostName )
        {
            var hostEntry = await ResolveHostNameInner( inHostName );

            string message = "";
            string addressList = ArrayHelper.MakeListString(hostEntry.AddressList,", ","<empty list>");
            if (addressList == inHostName && hostEntry.HostName != null) {
                message += hostEntry.HostName;
            } else {
                message += addressList;
                if (hostEntry.HostName != null && hostEntry.HostName != inHostName) message += "; Host Name: " + hostEntry.HostName;
            }
            return message;
        }

        async static Task<IPHostEntry> ResolveHostNameInner( string hostName )
        {
            IPAddress address;
            if (IPAddress.TryParse(hostName,out address)) {
                var ip = new IPHostEntry();
                ip.AddressList = new IPAddress[]{address};
                return ip;
            } else {
                return await System.Net.Dns.GetHostEntryAsync( hostName );
            }
        }
        public async static Task<string> ResolveHostNameForDCC( string hostName )
        {
            var host = await ResolveHostNameInner( hostName );

            foreach(System.Net.IPAddress address in host.AddressList) {
                if (address.AddressFamily == System.Net.Sockets.AddressFamily.InterNetwork) {
                    return IRCUtils.Formatting.formatDCCIP(  address.ToString() );
                }
            }
            foreach(System.Net.IPAddress address in host.AddressList) {
                if (address.AddressFamily == System.Net.Sockets.AddressFamily.InterNetworkV6) {
                    return address.ToString();
                }
            }
            foreach(System.Net.IPAddress address in host.AddressList) {
                return address.ToString();
            }
            throw new Exception("Could not resolve own address");

        }
    }
}
