using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.Win32;
using System.IO;
using System.IO.Pipes;
using System.Threading;
using System.Threading.Tasks;

namespace monochrome
{
    public partial class MainWindow : Form
    {


        private CancellationTokenSource m_aborter = new CancellationTokenSource();
        private static string pipeServerName() {
            var hash = PlatformMethods.StringMD5(PreferencesManager.PreferencesFilePath);
            return "monochrome-multiinstance-sync-" + BitConverter.ToString(hash);
        }

        async Task multiInstanceSyncWorker() {
            try {
                var abort = m_aborter.Token;
                var name = pipeServerName();
                using (var server = new NamedPipeServerStream(pipeServerName(), PipeDirection.In, 1, PipeTransmissionMode.Byte, PipeOptions.Asynchronous)) {
                    var aborter = Task.Delay(-1, abort);
                    for (; ; ) {
                        await Task.WhenAny(
                             Task.Factory.FromAsync(server.BeginWaitForConnection, server.EndWaitForConnection, TaskCreationOptions.None),
                             aborter
                             );
                        if (abort.IsCancellationRequested) return;
                        server.Disconnect();
                        Activate();
                    }
                }
            } catch {

            }
        }

        public MainWindow() {
            _ = multiInstanceSyncWorker();
            
            m_childWindows = new Dictionary<SwitchableChildWindow, TreeNode>();
            m_sessionLocked = false;
            InitializeComponent();

            float dpiX, dpiY;
            using (Graphics g = CreateGraphics()) {
                dpiX = g.DpiX;
                dpiY = g.DpiY;
            }

            this.ClientSize = new System.Drawing.Size( (int)Math.Round( 800 * dpiX / 96 ), (int)Math.Round( 500 * dpiY / 96 ) );
            SystemEvents.SessionSwitch += new SessionSwitchEventHandler(onSessionSwitch);

            Win32Utils.Forms.SetExplorerTheme(windowTree);

            //windowTree.Font = PreferencesManager.Current.Font;

            OnPreferencesChanged(PreferencesManager.Current);
        }

        void ApplyColorsRecur(TreeNodeCollection nodes) {
            foreach (TreeNode node in nodes) {
                node.ForeColor = HighlightToColor(((SwitchableChildWindow)node.Tag).HighlightLevel);
                ApplyColorsRecur(node.Nodes);
            }
        }

        void OnPreferencesChanged(PreferencesData newData)
        {
            newData.ApplyColors(windowTree);

            windowTree.Font = newData.ListFont;

            ApplyColorsRecur(windowTree.Nodes);

            if (!newData.colorText.IsEmpty) {
                windowTree.DrawMode = TreeViewDrawMode.OwnerDrawText;
            } else {
                windowTree.DrawMode = TreeViewDrawMode.Normal;
            }
        }

        int autoConnectComparison(KeyValuePair<string, ServerParams> item1, KeyValuePair<string, ServerParams> item2) {
            return string.Compare(item1.Key, item2.Key);
        }

        void ProcessAutoConnect() {
            if (m_autoConnectDone) return;
            m_autoConnectDone = true;

            // hold shift to suppress autoconnect
            if (ModifierKeys == Keys.Shift) return;

            try {
                List<KeyValuePair<string, ServerParams>> entries = new List<KeyValuePair<string, ServerParams>>(ServerConfigManager.GrabAutoConnectEntries());
                entries.Sort(new Comparison<KeyValuePair<string, ServerParams>>(autoConnectComparison));
                foreach (KeyValuePair<string, ServerParams> entry in entries) {
                    new ServerWindow(this, entry.Value, entry.Key);
                }
            } catch (Exception ex) {
                new PopupMessage(this, "Could not process connect-on-startup server list: " + ex, "Error");
            }
            SwitchToDefaultChild();
        }


        public void SwitchToChildDelta(int delta) {
            prevNext(delta);
        }

        public void SwitchToDefaultChild() {
            if (m_activeSCW == null) {
                TreeNode n = FirstNode();
                if (n != null) ((SwitchableChildWindow)n.Tag).SwitchTo();
            }
        }
        public void SwitchToChild(SwitchableChildWindow child) {
            if (m_activeSCW != child) {
                SwitchableChildWindow before = m_activeSCW;
                m_activeSCW = child;
                if (before != null) {
                    MenuStrip strip = before.MainMenuStrip;
                    if (strip != null) ToolStripManager.RevertMerge(mainMenu, strip);
                }

                if (m_activeSCW != null) {
                    m_activeSCW.Show();
                    m_activeSCW.BringToFront();
                    m_activeSCW.FlashReset();
                    windowActivated(m_activeSCW);

                    MenuStrip strip = m_activeSCW.MainMenuStrip;
                    if (strip != null) {
                        strip.AllowMerge = true; mainMenu.AllowMerge = true;
                        ToolStripManager.Merge(strip, mainMenu);
                    }
                }
                if (before != null) before.Hide();
                updateServerMenu();
            }
            if (m_activeSCW != null) m_activeSCW.SetDefaultFocus();

            updateWindowText();
        }

        public bool SwitchToChild(int index) {
            /*if (index >= 0 && index < windowList.Controls.Count) {
                ((Form)windowList.Controls[index].Tag).BringToFront();
                return true;
            } else {
                return false;
            }*/
            return false;
        }
        public void MoveChildDelta(int delta) {
            SwitchableChildWindow child = ActiveSCW as SwitchableChildWindow;
            if (child != null) MoveChildDelta(child, delta);
        }
        public void MoveChildDelta(SwitchableChildWindow child, int delta) {
            TreeNode node;
            if (m_childWindows.TryGetValue(child, out node)) {
                SwitchableChildWindow oldActive = ActiveSCW as SwitchableChildWindow;
                bool ignoreSwitchNotificationsBackup = m_ignoreSwitchNotifications;
                try {
                    m_ignoreSwitchNotifications = true;
                    TreeNodeCollection collection;
                    if (node.Parent != null) collection = node.Parent.Nodes;
                    else collection = node.TreeView.Nodes;
                    int index = Math.Max(Math.Min(node.Index + delta, collection.Count - 1), 0);
                    collection.Remove(node);
                    collection.Insert(index, node);
                    node.TreeView.SelectedNode = node;
                    rebuildWindowMenu();
                } finally {
                    m_ignoreSwitchNotifications = ignoreSwitchNotificationsBackup;
                }
                if (oldActive != null) oldActive.SwitchTo();
            }
        }

        public bool MoveChildToIndex(SwitchableChildWindow child, int target) {
            return false;
            /*if (target >= this.windowList.Controls.Count) return false;
            int oldIndex = getWindowIndex(child);
            windowList.Controls.SetChildIndex(windowList.Controls[oldIndex],target);
            return true;*/
        }

        private void onCommandConnect(object sender, EventArgs e) {
            if (m_setupDialog == null) m_setupDialog = new ServerSetupDialog(this);
            m_setupDialog.SwitchTo();
        }

        public void SetupDialogClosing() {
            m_setupDialog = null;
        }

        bool isAwayInternal {
            get {
                if (PreferencesManager.Current.awayOnMinimize) {
                    if (WindowState == FormWindowState.Minimized) return true;
                }
                if (PreferencesManager.Current.awayOnLock && m_sessionLocked) return true;
                return awayToolStripMenuItem.Checked;
            }
        }
        void ApplyAway() {
            ServerConnection.IsAway = isAwayInternal;
        }
        public void OnManualAway(bool bState) {
            awayToolStripMenuItem.Checked = bState;
        }


        static int truncat(double val) {
            return Convert.ToInt32(Math.Round(val));
        }
        static int fixDPI(int size, float dpi) {
            return truncat(size * dpi / 96.0);
        }

        static Color HighlightToColor(int level) {
            switch (level) {
                case 0: return Color.Empty;
                case 1: return PreferencesManager.Current.ColorChanHighlight1;
                case 2: return PreferencesManager.Current.ColorChanHighlight2;
                case 3: return PreferencesManager.Current.ColorChanHighlight3;
                default: return Color.Pink;
            }
        }


        public void windowActivated(SwitchableChildWindow window) {
            TreeNode node;
            if (m_childWindows.TryGetValue(window, out node)) {
                windowTree.SelectedNode = node;
            }
        }

        void rebuildWindowMenu() {
            {
                bool haveChildren = m_childWindows.Count > 0;
                previousToolStripMenuItem.Enabled = haveChildren;
                nextToolStripMenuItem.Enabled = haveChildren;
                moveUpToolStripMenuItem.Enabled = haveChildren;
                moveDownToolStripMenuItem.Enabled = haveChildren;
                toolStripSeparator2.Visible = haveChildren;
            }
            while (windowToolStripMenuItem.DropDownItems.Count > 0) {
                ToolStripItem item = windowToolStripMenuItem.DropDownItems[windowToolStripMenuItem.DropDownItems.Count - 1];
                if (item is ToolStripSeparator) break;
                windowToolStripMenuItem.DropDownItems.RemoveAt(windowToolStripMenuItem.DropDownItems.Count - 1);
            }
            int numWalk = 0;
            for (TreeNode nodeWalk = FirstNode(); nodeWalk != null; nodeWalk = NextNode(nodeWalk)) {
                SwitchableChildWindow child = (SwitchableChildWindow)nodeWalk.Tag;
                ToolStripMenuItem item = new ToolStripMenuItem(child.GetButtonText());
                try {
                    item.Tag = child;
                    item.Click += new EventHandler(windowMenuItemClick);
                    int num = ++numWalk;
                    if (num <= 10) {
                        if (num == 10) num = 0;
                        //ShortcutKeys = ((System.Windows.Forms.Keys)((System.Windows.Forms.Keys.Alt | System.Windows.Forms.Keys.W)))
                        item.ShortcutKeys = (Keys)((Keys)(Keys.D0 + num) | Keys.Alt);
                        item.ShortcutKeyDisplayString = "Alt+" + num;
                    }
                    windowToolStripMenuItem.DropDownItems.Add(item);
                } catch {
                    item.Dispose();
                    throw;
                }
            }
        }

        void updateWindowText() {
            if (m_activeSCW == null) Text = "Monochrome";
            else Text = "Monochrome - " + m_activeSCW.Text;
        }

        public Control SCWParent {
            get {
                return this.panel1;
            }
        }

        public void windowAdded(SwitchableChildWindow window, SwitchableChildWindow parent) {
            TreeNode node = new TreeNode(window.GetButtonText());
            
            node.ContextMenuStrip = window.windowListContextMenu;
            node.Tag = window;
            node.ForeColor = HighlightToColor(window.HighlightLevel);

            m_childWindows.Add(window, node);

            TreeNode parentNode;
            if (parent != null && m_childWindows.TryGetValue(parent, out parentNode)) {
                parentNode.Nodes.Add(node);
                parentNode.Expand();
            } else {
                windowTree.Nodes.Add(node);
            }

            fixTreeWidth( node );

            rebuildWindowMenu();

            if (this.ActiveSCW == window) windowActivated(window);
        }
        private void fixTreeWidth(TreeNode node) {
            int preferred = windowTree.GetPreferredSize(new Size(0, 0)).Width;
            String text = node.Text;
            if (text == "") return;

            var w = TextRenderer.MeasureText(text, windowTree.Font).Width;

            w += node.Level * windowTree.Indent;

            int delta = w - windowTree.ClientSize.Width;
            if (delta > 0) {
                windowTree.Parent.Width += delta;
            }
        }
        public void windowRenamed(SwitchableChildWindow window) {
            TreeNode node;
            if (m_childWindows.TryGetValue(window, out node)) {
                node.Text = window.GetButtonText();
                rebuildWindowMenu();
                fixTreeWidth(node);
            }
            updateWindowText();
        }
        public void windowRemoved(SwitchableChildWindow window) {
            if (window == ActiveSCW) {
                SwitchToChild(null);
            }
            TreeNode node;
            if (m_childWindows.TryGetValue(window, out node)) {
                if (!windowTree.IsDisposed) node.Remove();
                m_childWindows.Remove(window);
                rebuildWindowMenu();
            }

            if (!windowTree.IsDisposed && m_activeSCW == null) SwitchToDefaultChild();
        }
        public void onWindowTriggered(SwitchableChildWindow window) {
            PreferencesData prefs = PreferencesManager.Current;
            if (prefs.beepOnTrigger) {
                System.Media.SystemSounds.Beep.Play();
            }
            if (prefs.flashOnTrigger && ActiveForm != this) {
                Win32Utils.Forms.FlashForm(this);
            }
        }
        public void setHighlightLevel(SwitchableChildWindow window, int level) {
            TreeNode node;
            if (m_childWindows.TryGetValue(window, out node)) {
                node.ForeColor = HighlightToColor(level);
            }
        }



        delegate void onSessionSwitchInternalFunc(bool state);

        void onSessionSwitchInternal(bool state) {
            try {
                if (InvokeRequired) {//doesn't seem needed though
                    BeginInvoke(new onSessionSwitchInternalFunc(onSessionSwitchInternal), state);
                } else {
                    m_sessionLocked = !state;
                    ApplyAway();
                }
            } catch {
            }
        }

        void onSessionSwitch(object sender, Microsoft.Win32.SessionSwitchEventArgs e) {
            switch (e.Reason) {
                case SessionSwitchReason.ConsoleConnect:
                case SessionSwitchReason.RemoteConnect:
                case SessionSwitchReason.SessionLogon:
                case SessionSwitchReason.SessionUnlock:
                    onSessionSwitchInternal(true);
                    break;
                case SessionSwitchReason.ConsoleDisconnect:
                case SessionSwitchReason.RemoteDisconnect:
                case SessionSwitchReason.SessionLock:
                case SessionSwitchReason.SessionLogoff:
                    onSessionSwitchInternal(false);
                    break;
            }
        }

        private void awayToolStripMenuItem_Click(object sender, EventArgs e) {
            awayToolStripMenuItem.Checked = !awayToolStripMenuItem.Checked;
            ApplyAway();
        }

        protected override void OnClosed(EventArgs e) {
            m_aborter.Cancel();

            SystemEvents.SessionSwitch -= new SessionSwitchEventHandler(onSessionSwitch);
            {
                LinkedList<SwitchableChildWindow> servers = new LinkedList<SwitchableChildWindow>();
                foreach (SwitchableChildWindow item in m_childWindows.Keys) {
                    if (item is ServerWindow) servers.AddLast(item);
                }
                foreach (SwitchableChildWindow item in servers) {
                    item.Close();
                }
            }
            
            base.OnClosed(e);
        }

        void windowMenuItemClick(object sender, EventArgs e) {
            if (sender is ToolStripMenuItem) {
                SwitchToChild((SwitchableChildWindow)((ToolStripMenuItem)sender).Tag);
            }
        }
        private void exitToolStripMenuItem_Click(object sender, EventArgs e) {
            Close();
        }


        private void preferencesToolStripMenuItem_Click(object sender, EventArgs e) {
            if (m_preferencesWnd == null) {
                m_preferencesWnd = new Preferences(this);
            }
            m_preferencesWnd.SwitchTo();
        }

        public void preferencesClosed() { m_preferencesWnd = null; }

        public bool canExternalUpdatePreferences { get { return m_preferencesWnd == null; } }

        public void externalUpdatePreferences(PreferencesData data) {
            if (canExternalUpdatePreferences) {
                PreferencesManager.Set(data);
            }
        }

        private void serverCloseToolStripMenuItem_Click(object sender, EventArgs e) {
            IRCWindow active = ActiveSCW as IRCWindow;
            if (active != null) {
                ServerWindow server = active.OwningServer;
                if (server != null) server.Close();
            }
        }
        string getActiveServerName() {
            IRCWindow active = ActiveSCW as IRCWindow;
            if (active == null) return null;
            if (active is ServerWindow) return active.Text;
            ServerWindow server = active.OwningServer;
            if (server == null) return null;
            return server.Text;
        }
        private void updateServerMenu() {
            string name = getActiveServerName();
            if (name == null) {
                serverCloseToolStripMenuItem.Enabled = false;
                serverCloseToolStripMenuItem.Text = "Close";
            } else {
                serverCloseToolStripMenuItem.Enabled = true;
                serverCloseToolStripMenuItem.Text = "Close: " + name;
            }
        }
        private void aboutToolStripMenuItem_Click(object sender, EventArgs e) {
            string msg = "Monochrome " + Globals.VersionFull + "\r\n" + Globals.Copyright;
            msg += "\r\n\r\n";
            msg += monochrome.Properties.Resources.LICENSE;
            new PopupMessage(this, msg, "About Monochrome");
        }

        private void previousToolStripMenuItem_Click(object sender, EventArgs e) {
            prevNext(-1);
        }

        private void nextToolStripMenuItem_Click(object sender, EventArgs e) {
            prevNext(1);
        }
        void prevNext(int delta) {
            TreeNode node = windowTree.SelectedNode;
            while (delta != 0 && node != null) {
                if (delta < 0) {
                    node = PrevNode(node);
                    if (node == null) node = LastNode();
                    delta++;
                } else {
                    node = NextNode(node);
                    if (node == null) node = FirstNode();
                    delta--;
                }
            }
            if (node != null) {
                SwitchToChild((SwitchableChildWindow)node.Tag);
            }
        }
        delegate void voidFunc();

        private void windowTree_DrawNode(object sender, DrawTreeNodeEventArgs e) {
            PreferencesData prefs = PreferencesManager.Current;
            Color bkColor;
            if ((e.State & TreeNodeStates.Selected) != 0) {
                bkColor = SystemColors.Highlight;
            } else {
                bkColor = prefs.ColorBackground;
            }
            Brush bkBrush = new SolidBrush(bkColor);
            e.Graphics.FillRectangle(bkBrush, e.Bounds);
            bkBrush.Dispose();
            Font nodeFont = e.Node.NodeFont;
            if (nodeFont == null) nodeFont = ((TreeView)sender).Font;

            Color txColor = e.Node.ForeColor;
            if (txColor.IsEmpty) txColor = prefs.ColorText;

            Brush txBrush = new SolidBrush(txColor);
            try {
                e.Graphics.DrawString(e.Node.Text, nodeFont, txBrush, e.Bounds.Left, e.Bounds.Top);
            } catch { }
            
            txBrush.Dispose();
        }

        private void windowTree_AfterSelect(object sender, TreeViewEventArgs e) {
            if (!m_ignoreSwitchNotifications) {
                TreeNode node = windowTree.SelectedNode;
                if (node != null) {
                    //node.ForeColor = HighlightToColor(0);
                    SwitchableChildWindow child = ((SwitchableChildWindow) node.Tag);
                    if (child != this.ActiveSCW) {
                        if (e.Action == TreeViewAction.ByMouse) {
                            BeginInvoke(
                                new voidFunc(delegate() { SwitchToChild(child); })
                                );
                        } else {
                            SwitchToChild(child);
                            windowTree.Focus();
                        }
                    }
                }
            }
        }

        TreeNode NextNode(TreeNode node) {
            if (node.Nodes.Count > 0) {
                return node.Nodes[0];
            }
            for(;;) {
                TreeNode parent = node.Parent;
                if (parent == null) {
                    int index = windowTree.Nodes.IndexOf(node);
                    if (index < 0) return null;
                    if (index + 1 < windowTree.Nodes.Count) return windowTree.Nodes[index+1];
                    else return null;
                } else {
                    int index = parent.Nodes.IndexOf(node);
                    if (index < 0) return null;
                    if (index + 1 < parent.Nodes.Count) return parent.Nodes[index+1];
                    node = parent;
                }
            }
        }

        static TreeNode _LastChild(TreeNode node) {
            for(;;) {
                if (node.Nodes.Count == 0) return node;
                node = node.Nodes[node.Nodes.Count-1];
            }
        }

        TreeNode PrevNode(TreeNode node) {
            for(;;) {
                TreeNode parent = node.Parent;
                if (parent == null) {
                    int index = windowTree.Nodes.IndexOf(node);
                    if (index < 0) return null;
                    if (index == 0) return null;
                    return _LastChild(windowTree.Nodes[index-1]);
                } else {
                    int index = parent.Nodes.IndexOf(node);
                    if (index < 0) return null;
                    if (index == 0) return parent;
                    return _LastChild(parent.Nodes[index-1]);
                }
            }
        }

        TreeNode FirstNode() {
            if (windowTree.Nodes.Count > 0) return windowTree.Nodes[0];
            else return null;
        }
        TreeNode LastNode() {
            if (windowTree.Nodes.Count > 0) {
                return _LastChild(windowTree.Nodes[windowTree.Nodes.Count-1]);
            } else {
                return null;
            }
        }

        public ServerWindow FindExistingNetInstance(string name) {
            foreach(SwitchableChildWindow child in m_childWindows.Keys) {
                ServerWindow server = child as ServerWindow;
                if (server != null) {
                    if (server.Connection.NetName == name) return server;
                }
            }
            return null;
        }
        public void CloseExistingConnections(string name) {
            LinkedList<ServerWindow> toClose = new LinkedList<ServerWindow>();
            foreach(SwitchableChildWindow child in m_childWindows.Keys) {
                ServerWindow server = child as ServerWindow;
                if (server != null) {
                    if (server.Connection.NetName == name) toClose.AddLast(server);
                }
            }
            foreach(ServerWindow window in toClose) window.Close();
        }

        bool haveServerWindows {
            get {
                foreach(SwitchableChildWindow child in m_childWindows.Keys) {
                    if (child is ServerWindow) return true;
                }
                return false;
            }
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            if (!m_overrideNoCloseQuery && !e.Cancel && haveServerWindows) {
                if (MessageBox.Show(this,"Are you sure?","Close Monochrome",MessageBoxButtons.YesNo) == DialogResult.No)
                    e.Cancel = true;
            }
            base.OnClosing(e);
        }
        public void OnOwnActivity() {
            if (PreferencesManager.Current.unAwayOnActivity) {
                awayToolStripMenuItem.Checked = false;
                ApplyAway();
            }
        }

        public void TriggerLogClosed(TriggerLog window) {
            if (window == m_triggerLog) {
                m_triggerLog = null;
            }
            if (window == m_messageLog) {
                m_messageLog = null;
            }
        }

        void LogEvent(ServerWindow source, string context, TextLine line, ref TriggerLog log, string logLabel, bool highlight) {
            if (log == null) {
                try {
                    SwitchableChildWindow oldActive = ActiveSCW as SwitchableChildWindow;
                    log = new TriggerLog(this,logLabel);
                    if (oldActive != null) oldActive.SwitchTo();
                } catch(Exception e) {
                    try { source.PrintOutputV2( TextLine.Simple( "Couldn't spawn + " + logLabel + " window: " + e.Message) );  } catch {}
                    return;
                }
            }
            log.AddLine( TextLine.ContextPrefix(source.Connection.NetName) + TextLine.ContextPrefix(context) + line);
        }

        public void LogActivity(ServerWindow source,string context, TextLine line,int level) {
            PreferencesData prefs = PreferencesManager.Current;
            if (prefs.useTriggerLog && level >= 3) LogEvent(source,context,line,ref m_triggerLog,"Trigger Log",false);
            if (prefs.useMessageLog && level >= 2) LogEvent(source,context,line,ref m_messageLog,"Message Log",level >= 3);
        }

        private void MainWindow_Layout(object sender, LayoutEventArgs e) {
            ApplyAway();
        }

        bool m_sessionLocked;
        bool m_ignoreSwitchNotifications;
        Preferences m_preferencesWnd;
        Dictionary<SwitchableChildWindow,TreeNode> m_childWindows;
        TriggerLog m_triggerLog, m_messageLog;
        ServerSetupDialog m_setupDialog;
        bool m_autoConnectDone;
        SwitchableChildWindow m_activeSCW;
        public SwitchableChildWindow ActiveSCW { get { return m_activeSCW; } }


        private void moveUpToolStripMenuItem_Click(object sender, EventArgs e) {
            MoveChildDelta(-1);
        }

        private void moveDownToolStripMenuItem_Click(object sender, EventArgs e) {
            MoveChildDelta(1);
        }
        
        static Control FindFocusedControlRecur(Control.ControlCollection collection) {
            foreach(Control walk in collection) {
                if (walk.Focused) {
                    return walk;
                }
            }
            foreach(Control walk in collection) {
                Control focused = FindFocusedControlRecur(walk.Controls);
                if (focused != null) return focused;
            }
            return null;
        }

        private void reverseTextToolStripMenuItem_Click(object sender, EventArgs e) {
            try {
                SwitchableChildWindow form = ActiveSCW;
                if (form != null) {
                    {
                        TextBoxBase ctrl = FindFocusedControlRecur(form.Controls) as TextBoxBase;
                        if (ctrl != null) {
                            if (ctrl.Enabled && !ctrl.ReadOnly) {
                                int selStart = ctrl.SelectionStart;
                                int selLen = ctrl.SelectionLength;
                                ctrl.Text = TextUtils.Reverse(ctrl.Text);
                                if (selStart >= 0 && selLen >= 0) {
                                    ctrl.SelectionStart = ctrl.Text.Length - (selStart + selLen);
                                    ctrl.SelectionLength = selLen;
                                }
                            }
                            return;
                        }
                    }

                    {
                        ComboBox ctrl = FindFocusedControlRecur(form.Controls) as ComboBox;
                        if (ctrl != null) {
                            if (ctrl.Enabled) {
                                int selStart = ctrl.SelectionStart;
                                int selLen = ctrl.SelectionLength;
                                ctrl.Text = TextUtils.Reverse(ctrl.Text);
                                if (selStart >= 0 && selLen >= 0) {
                                    ctrl.SelectionStart = ctrl.Text.Length - (selStart + selLen);
                                    ctrl.SelectionLength = selLen;
                                }
                            }
                            return;
                        }
                    }

                }
            } catch {}
        }


        private void MainWindow_Shown(object sender, EventArgs e)
        {
            ProcessAutoConnect();           
        }

        private static void PerformSelectAll(Control ctrl) {
            {
                TextBoxBase tb = ctrl as TextBoxBase;
                if (tb != null) {
                    tb.SelectAll();
                    return;
                }
            }
            {
                ComboBox cb = ctrl as ComboBox;
                if (cb != null) {
                    cb.SelectAll();
                    return;
                }
            }
            {
                ListView lv = ctrl as ListView;
                if (lv != null) {
                    if (lv.MultiSelect) for(int walk = 0; walk < lv.Items.Count; ++walk) lv.SelectedIndices.Add(walk);
                    return;
                }
            }
        }

        private void selectAllToolStripMenuItem_Click(object sender, EventArgs e) {
            try {
                SwitchableChildWindow form = ActiveSCW;
                if (form != null) {
                    Control ctrl = FindFocusedControlRecur(form.Controls);
                    if (ctrl != null) {
                        PerformSelectAll(ctrl);
                    }
                }
            } catch {}
        }

        private void closeWindowShortcutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            SwitchableChildWindow form = ActiveSCW;
            if (form != null) form.CloseRequest();
        }

        void CloseForced() {
            bool before = m_overrideNoCloseQuery;
            try {
                m_overrideNoCloseQuery = true;
                Close();
            } finally {
                m_overrideNoCloseQuery = before;
            }
        }

        bool m_overrideNoCloseQuery;

        private const int WM_QUERYENDSESSION = 0x0011;
        private const int WM_ENDSESSION = 0x0016;

        private static bool charIsDelim(char c) {
            return c == ' ' || c == ';' || c == ',' || c == ':' || c == '.';
        }

        private void deleteWordToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try {
                SwitchableChildWindow form = ActiveSCW;
                if (form != null) {
                    {
                        TextBoxBase ctrl = FindFocusedControlRecur(form.Controls) as TextBoxBase;
                        if (ctrl != null) {
                            if (ctrl.Enabled && !ctrl.ReadOnly) {
                                int selStart = ctrl.SelectionStart;
                                int selLength = ctrl.SelectionLength;
                                if (selLength > 0) {
                                    ctrl.SelectedText = "";
                                } else
                                {
                                    int selEnd = selLength + selStart;
                                    int i = selStart;
                                    String text = ctrl.Text;
                                    while( i > 0 && charIsDelim(text[i-1])) --i;
                                    while( i > 0 && !charIsDelim(text[i-1])) --i;
                                    ctrl.SelectionStart = i;
                                    ctrl.SelectionLength = selEnd - i;
                                    ctrl.SelectedText = "";
                                }
                            }
                            return;
                        }
                    }

                    {
                        ComboBox ctrl = FindFocusedControlRecur(form.Controls) as ComboBox;
                        if (ctrl != null) {
                            if (ctrl.Enabled) {
                                int selStart = ctrl.SelectionStart;
                                int selLength = ctrl.SelectionLength;
                                if (selLength > 0) {
                                    ctrl.SelectedText = "";
                                } else
                                {
                                    int selEnd = selLength + selStart;
                                    int i = selStart;
                                    String text = ctrl.Text;
                                    while( i > 0 && charIsDelim(text[i-1])) --i;
                                    while( i > 0 && !charIsDelim(text[i-1])) --i;
                                    ctrl.SelectionStart = i;
                                    ctrl.SelectionLength = selEnd - i;
                                    ctrl.SelectedText = "";
                                }
                            }
                            return;
                        }
                    }

                }
            } catch {}
        }

        protected override void WndProc(ref Message m) { 
            if(m.Msg==WM_QUERYENDSESSION) { 
                CloseForced();
            } 
            base.WndProc(ref m); 
        } 

        public static bool TriggerRunningInstance() {
            bool rv = false;
            try {
                using (NamedPipeClientStream client = new NamedPipeClientStream(".",pipeServerName(), PipeDirection.Out, PipeOptions.Asynchronous)) {
                    client.Connect();
                    rv = true;
                }
            } catch { }
            return rv;
        }
    }
}