using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using IRCUtils;

namespace monochrome {

    public class SwitchableChildWindow : UserControl {

        public virtual void CloseRequest() {
            Close();
        }

        public void Close() {
            if (SCWClosed != null) SCWClosed(this, null);
            Dispose();
        }

        public MainWindow SCWParent {get {return m_owner;} }

        protected SwitchableChildWindow () { // for designer only
            InitializeComponent();
        }
        protected SwitchableChildWindow (MainWindow _owner, SwitchableChildWindow parent = null) {
            m_owner = _owner;
            Parent = _owner.SCWParent;
            Dock = DockStyle.Fill;
            InitializeComponent();
            AddToOwner(parent);
            

            // This is no longer relevant without MDI involved??? Disabled 2015-06-25

            // workaround clowntarded bugs with AutoScrollPosition in ChannelView
            // HAVE to show the window when creating or else scrolling bugs will h1en when text appears
            // ShowInBackground(); 
        }

        protected void ShowInBackground() {
            this.SendToBack();
            this.Show();
        }
        

        public void SwitchTo() {
            if (addedToOwner) ownerMainWnd.SwitchToChild(this);
        }

        protected void AddToOwner(SwitchableChildWindow parent) {
            if (m_owner != null && !addedToOwner) {
                m_owner.windowAdded(this, parent);
                addedToOwner = true;
            }
        }

        public virtual string GetButtonText() {return Text;}

        public virtual SwitchableChildWindow ParentSCW { get {return null; } }

        public virtual void Flash(int level) {
            if (level >= 3) {
                ownerMainWnd.onWindowTriggered(this);
            }
            if (!IsActive && level > highlightLevel) {
                highlightLevel = level;
                ownerMainWnd.setHighlightLevel(this,level);
            }
        }
        public void FlashReset() {
            highlightLevel = 0;
            ownerMainWnd.setHighlightLevel(this,0);
        }
        public int HighlightLevel {get { return highlightLevel;} }
        
        public MainWindow ownerMainWnd {get {return m_owner;}}

        bool IsActive {
            get {
                if (m_owner == null) return false;
                SwitchableChildWindow active = m_owner.ActiveSCW;
                if (active == null) return false;
                return this.Equals(active);
            }
        }
        void RemoveFromOwner() {
            if (m_owner != null && addedToOwner) {
                m_owner.windowRemoved(this);
                addedToOwner = false;
            }
        }
        protected override void Dispose(bool disposing) {
            RemoveFromOwner();
            base.Dispose(disposing);
        }

        protected override void OnTextChanged(EventArgs e) {
            if (m_owner != null) m_owner.windowRenamed(this);
        }


        public virtual void SetDefaultFocus() {}

        MainWindow m_owner;
        private ToolStripMenuItem closeToolStripMenuItem;
        int highlightLevel;
        bool addedToOwner;

        public MenuStrip MainMenuStrip;
        
        public event FormClosedEventHandler SCWClosed;


        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            this.windowListContextMenu = new System.Windows.Forms.ContextMenuStrip(this.components);
            this.closeToolStripMenuItem = new System.Windows.Forms.ToolStripMenuItem();
            this.windowListContextMenu.SuspendLayout();
            this.SuspendLayout();
            // 
            // windowListContextMenu
            // 
            this.windowListContextMenu.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this.closeToolStripMenuItem});
            this.windowListContextMenu.Name = "windowListContextMenu";
            this.windowListContextMenu.Size = new System.Drawing.Size(104, 26);
            // 
            // closeToolStripMenuItem
            // 
            this.closeToolStripMenuItem.Name = "closeToolStripMenuItem";
            this.closeToolStripMenuItem.Size = new System.Drawing.Size(103, 22);
            this.closeToolStripMenuItem.Text = "Close";
            this.closeToolStripMenuItem.Click += new System.EventHandler(this.closeToolStripMenuItem_Click);
            // 
            // SwitchableChildWindow
            // 
            this.Size = new System.Drawing.Size(284, 264);
            this.Visible = false;
            this.windowListContextMenu.ResumeLayout(false);
            this.ResumeLayout(false);

        }

        public ContextMenuStrip windowListContextMenu;
        private System.ComponentModel.IContainer components;

        private void closeToolStripMenuItem_Click(object sender, EventArgs e) {
            Close();
        }
    };

    public class IRCWindow : SwitchableChildWindow {
        protected IRCWindow() {} // for designer only
        protected IRCWindow(MainWindow _mainWindow, SwitchableChildWindow _parent = null) : base(_mainWindow, _parent) {
            InitPrefs();
        }
        protected IRCWindow(ServerWindow _ownerServer) : this(_ownerServer.ownerMainWnd, _ownerServer) {m_ownerServer = _ownerServer;}

        void InitPrefs() {
            PreferencesManager.Notify += OnPreferencesChanged;
        }

        public void Clear() {
            ChannelView view = GetContentBox();
            if (view != null) view.Clear();
        }

        void OnPreferencesChanged(PreferencesData newData) {
            using ( var f = newData.Font ) {
                Font = newData.Font;
            }
        }

        public ServerWindow OwningServer {
            get {
                if (this is ServerWindow) return (ServerWindow) this;
                else return m_ownerServer;
            }
        }

        public void PrintOutputV2(TextLine line) {
            Flash(1);
            ChannelView view = GetContentBox();
            if (view != null) view.addLine(line);
        }


        public void OnTrigger() {Flash(3);}
        public void OnMessage() {Flash(2);}

        protected virtual ChannelView GetContentBox() {return null;}

        protected override void Dispose(bool disposing) {
            PreferencesManager.Notify -= OnPreferencesChanged;
            base.Dispose(disposing);
        }

        void ScrollPages(int delta) {
            ChannelView view = GetContentBox();
            if (view != null) {
                view.ScrollDelta( view.TextPageHeight * delta );
            }
        }

        protected bool OnEditBoxKey(Keys code) {
            switch(code) {
                case Keys.PageDown:
                    ScrollPages(1);
                    return true;
                case Keys.PageUp:
                    ScrollPages(-1);
                    return true;
                default:
                    return false;
            }
        }
        protected override void WndProc(ref Message m) {
            switch(m.Msg) {
                default:
                    base.WndProc(ref m);
                    break;
                case Win32Utils.Messages.WM_MOUSEHWHEEL:
                case Win32Utils.Messages.WM_MOUSEWHEEL:
                    {
                        ChannelView view = GetContentBox();
                        if (view != null && !inRelayMessage) {
                            try {
                                inRelayMessage = true;
                                Win32Utils.Forms.RelayMessage(view,m);
                            } finally {
                                inRelayMessage = false;
                            }
                        } else {
                            base.WndProc(ref m);
                        }
                    }
                    break;
            }
        }

        bool inRelayMessage;
        private ServerWindow m_ownerServer;

        protected string Context;

        public override string GetButtonText() { return Context; }
        protected IAutoCompleteSource ACSource {
            get { return new ACSourceImpl(this); }
        }

        private class ACSourceImpl : IAutoCompleteSource {
            public ACSourceImpl(IRCWindow wnd) {
                m_wnd = new WeakReference<IRCWindow>(wnd);
            }
            private WeakReference<IRCWindow> m_wnd;
            LinkedList<string> IAutoCompleteSource.GetACStrings() {
                IRCWindow wnd;
                if (!m_wnd.TryGetTarget(out wnd)) return null;
                return wnd.OwningServer.Connection.GetACStringsFor(wnd.Context);
            }
            string IAutoCompleteSource.GetACChanTypes() {
                IRCWindow wnd;
                if (!m_wnd.TryGetTarget(out wnd)) return "";
                return wnd.OwningServer.Connection.GetACChanTypes();
            }
        }

    }
}
