using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.Net.Sockets;
using System.IO;
using System.Threading.Tasks;

namespace monochrome
{
    sealed class FileWriterImpl : IDCCFileWriter {
        public FileWriterImpl(FileStream stream ) {
            m_stream = stream;
        }

        async Task IDCCFileWriter.Seek(ulong position) {
            await Task.Run( () => { m_stream.Seek((long) position, SeekOrigin.Begin); } );
        }
        async Task IDCCFileWriter.SetLength(ulong length) {
            await Task.Run( () => { m_stream.SetLength((long) length); } );
        }
        async Task<ulong> IDCCFileWriter.GetLength() {
            return await Task.Run( () => { return (ulong) m_stream.Length; } );
        }
        async Task IDCCFileWriter.Write(byte[] buffer, uint length) {
            await Task.Run( () => { m_stream.Write(buffer, 0, (int) length); } );
        }

        void IDisposable.Dispose() {
            m_stream.Dispose();
        }
        private FileStream m_stream;

    }


    public partial class DCCReceiveWindow : DCCWindowBase, IDCCReceiveDelegate {

        // IDCCReceiveDelegate
        public UInt64 FileSize { get { return m_fileSize; } }


        async Task<IDCCFileWriter> IDCCReceiveDelegate.CreateFile() {
            return await Task.Run( () => {
                return new FileWriterImpl(File.Create(m_targetPath));
            });
        }

        async Task<IDCCFileWriter> IDCCReceiveDelegate.AppendFile() {
            return await Task.Run( () => {
                return new FileWriterImpl(File.Open(m_targetPath, FileMode.Append));
            });
        }

        public void SetProgress( UInt64 bytesTransferred ) {
            if (bytesTransferred > m_fileSize) {
                bytesTransferred = m_fileSize;
            }
            progressBar.Value = Math.Min( progressBar.Maximum,  (int) ( (UInt64)progressBar.Maximum * bytesTransferred / m_fileSize ) );
        }

        public String MainText
        {
            get { return Text; }
            set { Text = value; }
        }

        public String StatusText
        {
            get { return this.statusBox.Text; }
            set { this.statusBox.Text = value; }
        }
        public String SaveDialogFileName
        {
            get { return saveFileDialog.FileName; }
            set {
                saveFileDialog.DefaultExt = System.IO.Path.GetExtension( value );
                saveFileDialog.FileName = value;
            }
        }

        public void ToggleRunning(bool bRunning) {
            if (!m_closed) buttonCancel.Text = bRunning ? "Cancel" : "Close";
        }

        public bool CanAcceptResume {
            get
            {
                if (!m_accepted) return false;
                if (!checkBoxAcceptResume.Checked) return false;
                return true;
            }
        }

        public void SetPercentDone( UInt32 pcDone )
        {
            SetTitlePercentage( pcDone );
        }

        public DCCReceiveWindow(ServerWindow owningServer, DCCDescription desc, UInt64 _fileSize,PreferencesData _prefs,bool _autoAcceptTrigger) : base(owningServer,desc) {

            m_prefs = _prefs;
            m_fileName = desc.FileName;
            m_fileSize = _fileSize;

            InitializeComponent();

            m_receive = new DCCReceive( owningServer.Connection, desc, this, _prefs );

            labelFileName.Text = m_fileName;
            labelFileSize.Text = (m_fileSize != UInt64.MaxValue) ? m_fileSize.ToString() + " bytes" : "unknown";
            labelNick.Text = desc.Nick;


            if (_autoAcceptTrigger) {
                OnAccept(true);
            }
        }
        
        void OnDispose() {
            m_closed = true;
            m_receive.Dispose();
            m_receive = null;
        }

        private void buttonCancel_Click(object sender, EventArgs e) {
            Close();
        }


        bool grabTargetPath(out string targetPath,bool wasAA) {
            if (!m_prefs.xferAskForFileLocation) {
                string outDir = m_prefs.xferDefaultDirectory;
                if (outDir.Length > 0 && Directory.Exists(outDir)) {
                    string path;
                    path = Path.Combine(outDir,m_fileName);
                    if (!File.Exists(path)) {
                        targetPath = path;
                        return true;
                    }
                    int ext = 1;
                    while(ext < 100) {
                        path = Path.Combine(outDir, Path.GetFileNameWithoutExtension(m_fileName) + "-" + ext.ToString() + Path.GetExtension(m_fileName));
                        if (!File.Exists(path)) {
                            targetPath = path;
                            return true;
                        }
                        ++ext;
                    }
                }
            }

            if (wasAA) {
                targetPath = null;
                return false;
            }

            if (saveFileDialog.ShowDialog(this) == DialogResult.OK) {
                string outFn = saveFileDialog.FileName;
                if (!Path.HasExtension(outFn)) {
                    outFn += Path.GetExtension(m_fileName);
                }
                targetPath = outFn;
                return true;
            } else {
                targetPath = null;
                return false;
            }
        }

        void onAccepted(string targetPath) {
            m_targetPath = targetPath;
            m_accepted = true;
            buttonAccept.Enabled = false;
            buttonAcceptResume.Enabled = false;
        }

        void OnAccept(bool wasAA) {
            string targetPath;
            if (grabTargetPath(out targetPath,wasAA)) {
                onAccepted(targetPath);
                StartTransfer();
            }
        }

        private void buttonAccept_Click(object sender, EventArgs e) {
            OnAccept(false);
        }
        private void buttonAcceptResume_Click(object sender, EventArgs e) {
            if (resumeFileDialog.ShowDialog(this) == DialogResult.OK) {
                try {
                    string targetPath = resumeFileDialog.FileName;
                    long length = 0;
                    using(FileStream f = File.Open(targetPath,FileMode.Append)) {
                        length = f.Length;
                    }
                    onAccepted(targetPath);
                    m_receive.InitiateResume((UInt64)length);
                } catch(Exception ex) {
                    new PopupMessage(ownerMainWnd,ex.Message,"Error Initializing DCC Resume");
                }
            }
        }
        void StartTransfer() {
            if (m_closed) return;
            if (m_receive.Running) throw new ArgumentException();
            m_receive.Start( );
        }

        
        public void OnFinished(Exception e) {
            if (m_closed) return;
            if (e == null) {
                m_succeeded = true;
                buttonOpen.Enabled = true;
                buttonBrowse.Enabled = true;
                progressBar.Value = progressBar.Maximum;
            }
            Flash(3);
        }
        private void buttonOpen_Click(object sender, EventArgs e) {
            if (m_succeeded && m_targetPath != null) {
                try {
                    System.Diagnostics.Process.Start(m_targetPath);
                } catch(Exception ex) {
                    new PopupMessage(ownerMainWnd,"Could not open downloaded file: " + ex.Message,"Error");
                }
            }
        }

        private void buttonBrowse_Click(object sender, EventArgs e) {
            if (m_succeeded && m_targetPath != null) {
                try {
                    //System.Diagnostics.Process.Start(Path.GetDirectoryName(m_targetPath));
                    //System.Diagnostics.Process.Start("explorer.exe /e,/select,\"" + m_targetPath + "\"");
                    Win32Utils.Shell.OpenContainingFolder(m_targetPath);
                } catch(Exception ex) {
                    new PopupMessage(ownerMainWnd,"Could not browse for downloaded file: " + ex.Message,"Error");
                }
            }
        }

        string m_fileName;
        UInt64 m_fileSize;
        bool m_succeeded;
        string m_targetPath;

        bool m_accepted;

        PreferencesData m_prefs;
        bool m_closed;


        DCCReceive m_receive;

    }
}
