using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using IRCUtils;

namespace monochrome
{
    public partial class ChannelWindow : IRCWindow, IChannelWindow
    {
        private ChannelStatus m_status;

        private ServerConnection Connection
        {
            get { return OwningServer.Connection; }
        }

        public ChannelWindow(ServerWindow _owner,string _channel, ChannelStatus status) : base(_owner)
        {
            m_status = status;
            m_status.PropertyChanged += StatusPropertyChanged;
            m_status.UserPrefixChanged += StatusUserPrefixChanged;
            lastSortDirection = new int[]{1,1,1};
            sortHistory = new LinkedList<int>(new int[]{1,0,2});
            m_listWindows = new Dictionary<char,BanListWindow>();
            m_nickMap = new Dictionary<string,ListViewItem>(Nicks.Comparer);
            ChannelName = _channel;
            InitializeComponent();
            editBox.ACSource = this.ACSource;
            Win32Utils.Forms.SetExplorerTheme(nameList);
            Text = ChannelName;
            Font = PreferencesManager.Current.Font;
            applySort();

            OnPreferencesChanged(PreferencesManager.Current);
            channelToolStripMenuItem.Text = "Channel: " + ChannelName;
            toggleNameListSelectionAvailable(false);
            nameList_ItemSelectionChanged();

            StatusPropertyChangedEx(null);
        }


        private void StatusUserPrefixChanged(string nick, string newPrefix) {
            ListViewItem item;
            if (findUser(nick, out item)) {
                item.SubItems[1].Text = newPrefix;
                nameList.Sort();
            }
        }

        private void StatusPropertyChangedEx( string name ) {
            bool didSetTopic = false;
            if (name == null || name == "Topic") {
                topicBox.Text = m_status.Topic;
                didSetTopic = true;
            }
            if (name == null || name == "CanSetTopic") {
                bool readOnly = !m_status.CanSetTopic;
                topicBox.ReadOnly = readOnly;
                if ( readOnly && ! didSetTopic ) {
                    topicBox.Text = m_status.Topic;
                }

            }
            if ( name == null || name == "ChannelMode" ) {
                Text = ChannelName + " (" + m_status.ChannelMode + ")";
            }
            if ( name == null || name == "HaveOp" ) {
                bool op = m_status.HaveOp;
                opCommandsToolStripMenuItem.Enabled = op;
                foreach (ToolStripItem item in opCommandsToolStripMenuItem.DropDownItems) {
                    item.Enabled = op;
                }
            }
            if ( name == null || name == "UserNicks" ) {
                NameListChanged();
            }
        }
        private void StatusPropertyChanged(object sender, PropertyChangedEventArgs e) {
            StatusPropertyChangedEx( e.PropertyName );
        }

        void OnPreferencesChanged(PreferencesData newData) {
            using (var f = newData.ListFont) {
                nameList.Font = f;
            }   
            newData.ApplyColors(contentBox);
            newData.ApplyColors(editBox);
            newData.ApplyColors(nameList);
            newData.ApplyColors(topicBox);
        }

        public void HandleDisconnection() {
            OnUnsubscribe();
        }

        void clearNameList() {
            nameList.Items.Clear();
            m_nickMap.Clear();
        }

        void fixColumnSizes() {
            columnName.AutoResize(ColumnHeaderAutoResizeStyle.ColumnContent);
            columnUserHost.AutoResize(ColumnHeaderAutoResizeStyle.ColumnContent);
            //nameList.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);
        }

        private void NameListChanged() {
            var hosts = new Dictionary<string,string>(Nicks.Comparer);
            foreach(ListViewItem item in nameList.Items) {
                ChannelUserData data = (ChannelUserData) item.Tag;
                hosts.Add(data.Nick,data.UserHost);
            }
            clearNameList();
            foreach( string nick in m_status.UserNicks ) {
                var prefix = m_status.PrefixOfNick( nick );
                string userHost;
                if (!hosts.TryGetValue(nick,out userHost)) userHost = "";
                
                addNick(nick,prefix,userHost);
            }
            fixColumnSizes();
            applySort();
        }
        public ICollection<string> NameList {
            get { return m_nickMap.Keys; }
        }
        public void UserJoined(string userString,bool isSelf) {
            if (!isSelf) {
                string name = Nicks.Extract(userString);
                addNick(name, "", Nicks.ExtractUserHost(userString));
                fixColumnSizes();
            }
            if (isSelf) {
                OnSubscribe();
            }
        }
        void OnSubscribe() {
            
            showBanList.Enabled = listAvailable('b');
            showBanExemptionList.Enabled = listAvailable('e');
            showQuietList.Enabled = listAvailable('q');
            showInviteList.Enabled = listAvailable('I');
        }
        bool listAvailable(char type) {
            string[] chanModes = Connection.InstanceData.ModeSpecs.ChanModes;
            if (chanModes == null) return false;
            return chanModes[0].IndexOf(type) >= 0;
        }
        public void UserQuit(string source) {
            removeNick(Nicks.Extract(source));
        }
        void addNick(string name,string status,string userHost) {
            ListViewItem item = new ListViewItem(new string[]{name,status,Nicks.ExtractHostmask(userHost)});
            ChannelUserData userData = new ChannelUserData();
            userData.UserHost = userHost;
            userData.Nick = name;
            item.Tag = userData;
            nameList.Items.Add(item);
            m_nickMap.Remove(name);//for safety
            m_nickMap.Add(name,item);
        }
        bool removeNick(string nick) {
            ListViewItem item = resolveNick(nick);
            if (item != null) {
                m_nickMap.Remove(nick);
                nameList.Items.Remove(item);
                return true;
            } else {
                return false;
            }
        }

        ListViewItem resolveNickEx(string who) {
            return resolveNick(Nicks.Extract(who));
        }
        ListViewItem resolveNick(string nick) {
            ListViewItem item;
            if (!m_nickMap.TryGetValue(nick,out item)) item = null;
            return item;
        }
        public void SetUserAway(string nick, bool isAway, string reasonOptional) {
            ListViewItem item = resolveNick(nick);
            if (item != null) {
                Color newColor = isAway ? SystemColors.GrayText : Color.Empty;
                //don't set the color if it's same as existing one (it will repaint anyway)
                if (item.ForeColor != newColor) item.ForeColor = newColor;
            }
        }

        void onUserActive(string source) {
            SetUserAway(Nicks.Extract(source),false,null);
        }

        public bool HasNick(string nick) {
            return resolveNick(nick) != null;
        }

        void OnUnsubscribe() {
            clearNameList();

            LinkedList<SwitchableChildWindow> toClose = new LinkedList<SwitchableChildWindow>();
            foreach(SwitchableChildWindow form in m_listWindows.Values) toClose.AddLast(form);
            
            foreach(SwitchableChildWindow form in toClose) form.Close();

            showBanList.Enabled = false;
            showBanExemptionList.Enabled = false;
            showQuietList.Enabled = false;
            showInviteList.Enabled = false;
        }

        public bool isUserPresent(string nick) {
            return resolveNick(nick) != null;
        }

        public void UserParted(string nick,bool isSelf) {
            removeNick(nick);
            if (isSelf) OnUnsubscribe();
        }

        public void HandleNickChange(string oldNick,string newNick,bool queryClash) {
            ListViewItem item;
            if (findUser(oldNick,out item)) {
                ChannelUserData data = (ChannelUserData) item.Tag;
                data.Nick = newNick;
                item.Text = newNick;
                m_nickMap.Remove(oldNick); m_nickMap.Remove(newNick);
                m_nickMap.Add(newNick,item);
                nameList.Sort();
                fixColumnSizes();
            }
        }

        bool findUser(string user, out ListViewItem result) {
            result = resolveNickEx(user);
            return result != null;
        }

        public void HandleMode(IRCEvent ev,ServerModeSpecs modeSpecs) {
            LinkedList<Modes.Mode> modes = Modes.Parse(ev,modeSpecs);
            foreach(Modes.Mode mode in modes) {

                if (mode.param != null) {
                    BanListWindow wnd;
                    if (m_listWindows.TryGetValue(mode.action,out wnd)) {
                        if (mode.sign) {
                            BanDescription desc = new BanDescription();
                            desc.Mask = mode.param;
                            desc.ByWhom = ev.Source;
                            desc.Date = DateTime.UtcNow;
                            wnd.OnEntryAdded(desc);
                        } else {
                            wnd.OnEntryRemoved(mode.param);
                        }
                    }
                }
            }
        }


        public void runCommand(string command) {
            Connection.runChannelCommand( command, ChannelName, this );
        }

        protected override ChannelView GetContentBox() {return contentBox;}

        void OnDispose() {
            if ( Connection != null ) {
                Connection.WindowClosed(ChannelName, this);
            }
            m_status.PropertyChanged -= StatusPropertyChanged;
            m_status.UserPrefixChanged -= StatusUserPrefixChanged;
        }

        private void topicBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (e.KeyChar == '\r' && m_status.CanSetTopic) {
                string newTopic = topicBox.Text;
                if (newTopic != m_status.Topic) {
                    Connection.addCommand(new IRCCommand("TOPIC",new string[]{ChannelName,newTopic}));
                }
                e.Handled = true;
            } else if (e.KeyChar == (char)27) {
                topicBox.Text = m_status.Topic;
                editBox.Focus();
                e.Handled = true;
            }
        }

        class SortByStatus : IComparer {
            public SortByStatus(int _direction,string _svPrefixes) {direction = _direction; svPrefixes = _svPrefixes;}

            int rate(object item) {
                string label = ((ListViewItem)item).SubItems[1].Text;
                for(int walk = 0; walk < svPrefixes.Length; ++walk) {
                    if (label.IndexOf(svPrefixes[walk]) >= 0) return walk;
                }
                return svPrefixes.Length;
            }

            public int Compare(object v1, object v2) {
                return direction * Comparer<int>.Default.Compare(rate(v1),rate(v2));
            }
            int direction;
            string svPrefixes;
        };

        class SortByHost : IComparer {
            public SortByHost(int _direction) {direction = _direction;}
            static string extractHost(object item) {
                return Nicks.ExtractHostmask(((ChannelUserData)((ListViewItem)item).Tag).UserHost);
            }
            public int Compare(object v1, object v2) {
                return direction * string.Compare( extractHost(v1), extractHost(v2) );
            }
            int direction;
        };

        class UserListSorter : IComparer {
            public UserListSorter(int _column, int _direction) {column = _column; direction = _direction;}
            public UserListSorter() : this(0,1) {}
            public int Compare(object v1, object v2) {
                return direction * String.Compare(((ListViewItem)v1).SubItems[column].Text, ((ListViewItem)v2).SubItems[column].Text);
            }            
            int column, direction;
        };

        class SortByName : UserListSorter {
            public SortByName(int _direction) : base(0,_direction) {}
        }

        class SortChain : IComparer {
            public SortChain() {m_chain = new LinkedList<IComparer>();}

            public void AddChain(IComparer comparer) {m_chain.AddLast(comparer);}

            public int Compare(object v1, object v2) {
                foreach(IComparer comparer in m_chain) {
                    int state = comparer.Compare(v1,v2);
                    if (state != 0) return state;
                }
                return 0;
            }

            private LinkedList<IComparer> m_chain;
        };

        void applySort(int column) {
            sortHistory.Remove(column);
            sortHistory.AddFirst(column);
            applySort();
        }
        void applySort() {
            SortChain chain = new SortChain();
            IComparer[] comparers = new IComparer[] { new SortByName(lastSortDirection[0]), new SortByStatus(lastSortDirection[1],PrefixList), new SortByHost(lastSortDirection[2]) };
            foreach(int walk in sortHistory) {
                chain.AddChain(comparers[walk]);
            }
            nameList.ListViewItemSorter = chain;
        }

        private void nameList_ColumnClick(object sender, ColumnClickEventArgs e) {
            int column = e.Column;
            if (column >= 0 && column < lastSortDirection.Length) {
                int direction = 1;
                if (lastSortColumn == column && lastSortDirection[column] > 0) direction = -1;
                lastSortColumn = column;
                lastSortDirection[column] = direction;
                applySort(column);
            }
        }

        private void openQueryWindowToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_openQuery();
        }
        
        private void nameList_DoubleClick(object sender, EventArgs e) {
            nameList_openQuery();
        }

        private void sendFileToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_command("/SENDFILE");
        }

        private void voiceToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_mode("+v");
        }
        
        private void deVoiceToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_mode("-v");
        }

        private void opToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_mode("+o");
        }

        private void deOpToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_mode("-o");
        }
        void nameList_command(string command) {
            foreach(ListViewItem item in nameList.SelectedItems) {
                runCommand(command + " " + ((ChannelUserData)item.Tag).Nick);
            }
        }
        private void kickToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_command("/KICK");
        }
        private void kickBanToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_command("/KICKBAN");
        }
        void nameList_mode(string mode) {
            LinkedList<string> modes = new LinkedList<string>();
            foreach(ListViewItem item in nameList.SelectedItems) {
                modes.AddLast(mode + " " + ((ChannelUserData)item.Tag).Nick);
            }
            Connection.addModeCommands(ChannelName,modes,this);

            //nameList_command("/MODE " + mode);
        }
        void nameList_openQuery() {
            nameList_command("/QUERY");
        }


        PrefixMap Prefixes
        {
            get { return Connection.InstanceData.ModeSpecs.Prefixes; }
        }
        string PrefixList {
            get { return Prefixes.PrefixList; }
        }

        public override void SetDefaultFocus() {
            editBox.Focus();
        }
        public string ChannelName {
            get { return this.Context; }
            set { this.Context = value; }
        }

        public void UserHostHint(string nick, string userHost) {
            ListViewItem item;
            if (findUser(nick, out item)) {
                ChannelUserData data = (ChannelUserData) item.Tag;
                if (userHost != data.UserHost) {
                    data.UserHost = userHost;
                    item.SubItems[2].Text = Nicks.ExtractHostmask(userHost);
                    columnUserHost.AutoResize(ColumnHeaderAutoResizeStyle.ColumnContent);
                }
            }
        }

        public void OnBanList(char type, IEnumerable<BanDescription> items) {
            BanListWindow wnd = null;
            if (!m_listWindows.TryGetValue(type,out wnd)) {
                wnd = new BanListWindow(this,type);
                m_listWindows.Add(type,wnd);
            }

            wnd.SetData(items);
            wnd.Show();
            wnd.SwitchTo();
        }

        public void ListWindowClosed(BanListWindow wnd) {
            m_listWindows.Remove(wnd.ListType);
        }
        public void RemoveListEntries(char type, IEnumerable<string> items) {
            LinkedList<string> modes = new LinkedList<string>();
            foreach(string entry in items) {
                modes.AddLast("-" + type.ToString() + " " + entry);
            }
            Connection.addModeCommands(ChannelName,modes,this);
        }

        private void whoisToolStripMenuItem_Click(object sender, EventArgs e) {
            nameList_command("/WHOIS");
        }

        void toggleNameListSelectionAvailable(bool state) {
            nameListContextMenu.Enabled = state;
        }
        
        string grabNameListMenuLabel() {
            if (nameList.SelectedItems.Count == 0) {
                return "<no selection>";
            } else if (nameList.SelectedItems.Count == 1) {
                ChannelUserData data = (ChannelUserData)nameList.SelectedItems[0].Tag;
                if (data.UserHost != "") return data.Nick + " (" + data.UserHost + ")";
                else return data.Nick;
            } else {
                return nameList.SelectedItems.Count.ToString() + " Users";
            }
        }

        void nameList_ItemSelectionChanged() {
            toggleNameListSelectionAvailable(nameList.SelectedItems.Count > 0);
            labelToolStripMenuItem.Text = grabNameListMenuLabel();
            openQueryWindowToolStripMenuItem.Text = nameList.SelectedItems.Count > 1 ? "Open Query Windows" : "Open Query Window";
        }

        private void nameList_ItemSelectionChanged(object sender, ListViewItemSelectionChangedEventArgs e) {
            nameList_ItemSelectionChanged();
        }

        private void editBox_OnEditBoxKeyDown(object sender, KeyEventArgs e) {
            if (!e.Handled) {
                if (OnEditBoxKey(e.KeyCode)) e.Handled = true;
            }
        }



        private bool editBox_OnInput(string[] content) {
            foreach(string line in content) {
                runCommand( IRCUtils.Misc.ProcessOutgoing(line) );
            }
            return true;
        }

        void runSelectionCTCP(string what, string data) {
            foreach(string nick in SelectedNicks) Connection.runCTCP(nick,what,data);
        }

        private void timeToolStripMenuItem_Click(object sender, EventArgs e) {
            runSelectionCTCP("TIME","");
        }

        class SelectedNickEnumerator : IEnumerator<string> {
            public SelectedNickEnumerator(ListView nameList) {
                enumerator = nameList.SelectedItems.GetEnumerator();
            }

            string IEnumerator<string>.Current { get { return _Current;} }
            object IEnumerator.Current { get { return _Current;} }

            string _Current { 
                get {
                    return ((ChannelUserData) ((ListViewItem)enumerator.Current).Tag).Nick;
                }
            }
            public bool MoveNext() {return enumerator.MoveNext();}
            public void Reset() {enumerator.Reset();}
            public void Dispose() {}

            IEnumerator enumerator;
        };

        class SelectedNickCollection : IEnumerable<string> {
            public SelectedNickCollection(ListView _nameList) {nameList = _nameList;}
            IEnumerator<string> IEnumerable<string>.GetEnumerator() {return new SelectedNickEnumerator(nameList);}
            IEnumerator IEnumerable.GetEnumerator() {return new SelectedNickEnumerator(nameList);}
            public int Count { get { return nameList.SelectedItems.Count;} }
            ListView nameList;
        }

        public IEnumerable<string> SelectedNicks {
            get {
                return new SelectedNickCollection(nameList);
            }
        }

        private void clientVersionToolStripMenuItem_Click(object sender, EventArgs e) {
            runSelectionCTCP("VERSION","");
        }

        private void pingToolStripMenuItem_Click(object sender, EventArgs e) {
            runSelectionCTCP("PING",Connection.GetPingParam());
        }
        private void browseLogsToolStripMenuItem_Click(object sender, EventArgs e) {
            if (m_logTarget != null) {
                Win32Utils.Shell.OpenContainingFolder(m_logTarget);
            }
        }

        public void SetLogTarget(object target) {
            m_logTarget = target as string;
            browseLogsToolStripMenuItem.Enabled = (target != null);
        }
        private bool editBox_IsInputAllowed() {
            return contentBox.EnsureBottomVisible();
        }

        int lastSortColumn;
        int[] lastSortDirection;
        Dictionary<string,ListViewItem> m_nickMap;
        Dictionary<char,BanListWindow> m_listWindows;
        LinkedList<int> sortHistory;
        string m_logTarget;

        private void selectAllToolStripMenuItem_Click(object sender, EventArgs e)
        {
            for(int walk = 0; walk < nameList.Items.Count; ++walk) {
                nameList.SelectedIndices.Add(walk);
            }
        }
        public override void CloseRequest()
        {
            DialogResult res = MessageBox.Show("Leave \"" + ChannelName + "\" ?", "Close", MessageBoxButtons.YesNo, MessageBoxIcon.None);
            if (res == DialogResult.Yes) Close();
        }

        private void showBanExemptionList_Click(object sender, EventArgs e)
        {
            Connection.beginShowList(ChannelName, 'e');
        }

        private void showQuietList_Click(object sender, EventArgs e)
        {
            Connection.beginShowList(ChannelName, 'q');
        }

        private void showInviteList_Click(object sender, EventArgs e)
        {
            Connection.beginShowList(ChannelName, 'I');
        }

        private void showBanList_Click(object sender, EventArgs e)
        {
            Connection.beginShowList(ChannelName, 'b');
        }
    }

    class ChannelUserData {
        public string Nick;
        public string UserHost;
    }
}