using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace monochrome
{
    public partial class ChannelView : UserControl
    {

        public ChannelView()
        {
            m_fontArray = new FontArray(Font);
            Disposed += new EventHandler(ChannelView_Disposed);
            m_clickPoint = m_selStart = m_selEnd = TextViewCoords.Invalid;
            m_lines = new List<LineEntry>();
            ApplyPreferences(PreferencesManager.Current);
            InitializeComponent();
            updateLineWidths();
        }

        void updateCustomColors(PreferencesData data) {
            m_userJoinColor = data.ColorUserJoin;
            m_userPartColor = data.ColorUserPart;
        }

        private void ApplyPreferences(PreferencesData newData) {
            m_urlCopy = newData.defaultURLCopy;
            updateCustomColors(newData);
            m_lineCountLimit = newData.lineCountLimit;
            ApplyLineCountLimit();
        }
        void OnPreferencesChanged(PreferencesData newData) {
            Invalidate();
            ApplyPreferences(newData);
//            ApplyFontChange();
            
        }

        int itemRangeHeight(int first, int count) {
            if (first < 0) {count += first; first = 0;}
            int maxCount = m_lines.Count - first;
            if (count > maxCount) count = maxCount;
            if (count <= 0) return 0;
            LineEntry last = m_lines[first+count-1];
            return last.m_offset + last.m_height - m_lines[first].m_offset;
        }

        Rectangle itemRectAbs(int index) {
            return new Rectangle(0,itemRangeHeight(0,index),ClientWidth,m_lines[index].m_height);
        }
        Rectangle itemRangeRectAbs(int first, int count) {
            return new Rectangle(0,itemRangeHeight(0,first),ClientWidth,itemRangeHeight(first,count));
        }
        Rectangle itemRect(int index) {
            Rectangle r = itemRectAbs(index);
            r.Offset( AutoScrollPosition );
            return r;
        }
        Rectangle itemRangeRect(int first, int count) {
            Rectangle r = itemRangeRectAbs(first,count);
            r.Offset( AutoScrollPosition );
            return r;
        }

        protected override void OnScroll(ScrollEventArgs se)
        {
            base.OnScroll(se);
            updateStickToBottom();
        }
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);
            updateStickToBottom();
        }

        public void addLine(TextLine line) {
            if (m_fontArray == null) return;
            
            LineEntry entry = new LineEntry();
            entry.m_text = line;
            entry.m_wordSpecs = line.GetWords(m_fontArray);
            CalculateHeight(entry);
            entry.m_offset = itemRangeHeight(0,m_lines.Count);
            
            m_lines.Add(entry);

            Invalidate(itemRect(m_lines.Count-1));

            ViewAreaSize += new Size(0,entry.m_height);

            /*if (!m_dragSelMode) AutoScrollPosition = new Point(0, Math.Max(0,ViewAreaSize.Height - ClientSize.Height));*/
            ApplyLineCountLimit();
        }

        public void Clear() {
            Capture = false;//cancel dragging operation in case it's active
            SelectNone();
            m_lines.Clear();
            ViewAreaSize = new Size(0,0);
            Invalidate();
        }

        int CalculateHeight(LineEntry entry) {
            return entry.m_height = TextLine.HeightFromWords(entry.m_wordSpecs,ClientWidth);
        }

        public string SelectedText {
            get {
                try {
                    if (HaveSelection) {
                        if (m_selStart.Line == m_selEnd.Line) {
                            return m_lines[m_selStart.Line].m_text.ToString().Substring(m_selStart.Character,m_selEnd.Character-m_selStart.Character);
                        } else {
                            string output = m_lines[m_selStart.Line].m_text.ToString().Substring(m_selStart.Character);
                            for(int walk = m_selStart.Line + 1; walk < m_selEnd.Line; ++walk) {
                                output += "\r\n" + m_lines[walk].m_text.ToString();
                            }
                            output += "\r\n" + m_lines[m_selEnd.Line].m_text.ToString().Substring(0,m_selEnd.Character);
                            return output;
                        }
                    } else {
                        return "";
                    }
                } catch(Exception e) {
                    Debug.WriteLine(e.Message);
                    return e.Message;
                }
            }
        }

        private void copyToolStripMenuItem_Click(object sender, EventArgs e) {
            try {
                string text = SelectedText;
                if (text.Length > 0) {
                    System.Windows.Forms.Clipboard.SetText(SelectedText);
                }
            } catch(Exception ex) {
                Debug.WriteLine("Clipboard error: " + ex.Message);
            }
        }


        ColorPair resolveColorSelPreprocess(TextStyle style,object tag) {
            if (tag is SelectionTag) {
                return new ColorPair(SystemColors.Highlight,SystemColors.Highlight);
            } else {
                return new ColorPair(Color.Transparent,Color.Transparent);
            }
        }

        ColorPair resolveColor(TextStyle style,object tag) {
            if (tag is SelectionTag) {
                return new ColorPair(SystemColors.HighlightText);
            } else {
                return new ColorPair(_resolveColor(style));
            }
        }
        static Byte ByteBlend(Byte byte1, Byte byte2, double position) {
            return (Byte) ( (double)byte1 + Math.Round( ((double)byte2 - (double)byte1) * position ) );
        }
        static Color ColorBlend(Color color1, Color color2,double position) {
            return Color.FromArgb( ByteBlend(color1.R,color2.R,position), ByteBlend(color1.G,color2.G,position), ByteBlend(color1.B,color2.B,position));
        }
        Color _resolveColor(TextStyle style) {
            switch(style.Type) {
                case TextStyleType.UserName:
                    {
                        PreferencesData prefs = PreferencesManager.Current;
                        if (prefs.useNickColors) {
                            return ColorUtils.RandomColor(style.TypeParam,style.Highlight ? 0.8 : 0.6, ColorUtils.LooksLikeBlackBackground(BackColor) ? 0.7 : 0.3);
                        } else {
                            return prefs.ColorHighlight;
                        }
                    }
                case TextStyleType.CustomColor:
                    return style.CustomColor.PlatformColor;
                default:
                    return ForeColor;
                case TextStyleType.Plain:
                    return style.Highlight ? PreferencesManager.Current.ColorHighlight : ForeColor;
                case TextStyleType.Gray:
                    return style.Highlight ? ColorBlend(SystemColors.GrayText,ForeColor,0.4) : SystemColors.GrayText;
                case TextStyleType.UserJoin:
                    return style.Highlight ? ColorBlend(m_userJoinColor,ForeColor,0.4) : m_userJoinColor;
                case TextStyleType.UserPart:
                    return style.Highlight ? ColorBlend(m_userPartColor,ForeColor,0.4) : m_userPartColor;
            }
        }

        private void ChannelView_Paint(object sender, PaintEventArgs e) {
            Graphics target = e.Graphics;
            Rectangle textRect = new Rectangle(0,0,ClientWidth,0);
            
            int first = LineFromPoint(e.ClipRectangle.Top);
            int last = LineFromPoint(e.ClipRectangle.Bottom);
            if (first >= 0) {
                if (last < 0) last = m_lines.Count - 1;
                for(int walk = first; walk <= last; ++walk) {
                    LineEntry entry = m_lines[walk];
                    textRect.Y = entry.m_offset + AutoScrollPosition.Y;
                    textRect.Height = entry.m_height;
                    if (target.IsVisible(textRect)) {
                        int selBase = 0, selCount = 0;
                        if (walk >= m_selStart.Line && walk <= m_selEnd.Line) {
                            if (walk == m_selStart.Line) {
                                selBase = m_selStart.Character;
                            } else {
                                selBase = 0;
                            }
                            if (walk == m_selEnd.Line) {
                                selCount = m_selEnd.Character - selBase;
                            } else {
                                selCount = entry.m_text.Length - selBase;
                            }
                        }
                        DrawSelectionHelper(entry.m_text,entry.m_wordSpecs,target,textRect.X,textRect.Y,textRect.Width,selBase,selCount);
                    }
                }
            }
        }

        class SelectionTag {}

        void DrawSelectionHelper(TextLine line, LinkedList<TextLine.WordSpecs> words, Graphics target, int X, int Y, int widthLimit, int selBase, int selCount) {
            if (selCount > 0) {
                TextLine lineTagged = line.Substring(0,selBase) + line.Substring(selBase,selCount).Retag(new SelectionTag()) + line.Substring(selBase + selCount);
                lineTagged.DrawEx(target,m_fontArray,X,Y,widthLimit,new TextStyle.ColorResolver(resolveColorSelPreprocess),words);
                lineTagged.DrawEx(target,m_fontArray,X,Y,widthLimit,new TextStyle.ColorResolver(resolveColor),words);
            } else {
                line.DrawEx(target,m_fontArray,X,Y,widthLimit,new TextStyle.ColorResolver(resolveColor),words);
            }
        }


        delegate void updateLineWidthsFunc();
        void updateLineWidths() {
            int newWidth = ClientSize.Width;
            
            if (newWidth != m_effectiveClientWidth) {
                m_effectiveClientWidth = newWidth;
                RecalculateLineHeights();
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            bool insideSizeChange = m_insideSizeChange;
            try {
                m_insideSizeChange = true;
                base.OnSizeChanged(e);

                if (!insideSizeChange) {
                
                    {
                        int height = ClientSize.Height;
                        if (height < m_lastSeenClentHeight) {
                            int oldOffset = ViewAreaOffset_FromClientHeight(m_lastSeenClentHeight);
                            if (oldOffset == 0) ViewAreaOffset = 0;
                        }
                        m_lastSeenClentHeight = height;
                    }

                    /*if (m_lines.Count > 0) {
                        BeginInvoke(new updateLineWidthsFunc(updateLineWidths));
                    } else */{
                        updateLineWidths();
                    }
                }
            } finally {
                m_insideSizeChange = insideSizeChange;
            }
        }

        void RecalculateLineHeights() {
            if (m_lines.Count > 0) {
                int total = 0;
                foreach(LineEntry entry in m_lines) {
                    int oldHeight = entry.m_height;
                    CalculateHeight(entry);
                    entry.m_offset = total;
                    total += entry.m_height;
                }

                ViewAreaSize = new Size(0,total);
                Invalidate();
            }
        }

        class LineEntry {
            public TextLine m_text;
            public int m_height, m_offset;
            public LinkedList<TextLine.WordSpecs> m_wordSpecs;
        }

        void onSelectionChanged() {
            copyToolStripMenuItem.Enabled = HaveSelection;
            openURLToolStripMenuItem.Enabled = URLUtils.IsURL(SelectedText);
        }

        
        class LineComparer : IComparer<LineEntry> {
            static bool IsAbove(LineEntry itemAbove, LineEntry itemBelow) {
                return itemAbove.m_offset + itemAbove.m_height <= itemBelow.m_offset;
            }
            int IComparer<LineEntry>.Compare(LineEntry item1, LineEntry item2) {
                if (IsAbove(item1,item2)) return -1;
                else if (IsAbove(item2,item1)) return 1;
                else return 0;
            }
        };

        TextViewCoords CoordsTop {
            get {
                return m_lines.Count > 0 ? new TextViewCoords(0,0) : TextViewCoords.Invalid;
            }
        }

        TextViewCoords CoordsBottom {
            get {
                if (m_lines.Count > 0) {
                    int index = m_lines.Count-1;
                    LineEntry entry = m_lines[index];
                    return new TextViewCoords(index,entry.m_text.Length);
                } else {
                    return TextViewCoords.Invalid;
                }
            }
        }
        
        TextViewCoords CoordsFromPointAbs(Point pt) {
            int line = LineFromPointAbs(pt.Y);
            if (line < 0) {
                return pt.Y < 0 ? CoordsTop : CoordsBottom;
            }
            int character = m_lines[line].m_text.PointToCharacter(new Point(pt.X,pt.Y - m_lines[line].m_offset),m_fontArray,ClientWidth, m_lines[line].m_wordSpecs);
            if (character < 0) return TextViewCoords.Invalid;
            return new TextViewCoords(line,character);
        }

        TextViewCoords CoordsFromPoint(Point pt) {
            Point position = AutoScrollPosition;
            return CoordsFromPointAbs(new Point(pt.X - position.X, pt.Y - position.Y) );
        }

        int LineFromPoint(Point pt) {
            return LineFromPoint(pt.Y);
        }
        int LineFromPoint(int Y) {
            return LineFromPointAbs(Y + ViewAreaPosition.Y);
        }
        int LineFromPointAbs(int Y) {
            LineEntry entry = new LineEntry();
            entry.m_offset = Y;
            entry.m_height = 1;
            entry.m_text = null;
            int bah = m_lines.BinarySearch(entry,new LineComparer());
            return bah < 0 ? -1 : bah;
        }
        void SelectSingleFromPoint(Point pt) {SelectSingleLine(LineFromPoint(pt));}
        void SelectNone() {SetSelection(TextViewCoords.Invalid,TextViewCoords.Invalid);}
        void SelectSingleLine(int index) {
            if (index < 0) SelectNone();
            else {
                SetSelection(new TextViewCoords(index,0),new TextViewCoords(index, m_lines[index].m_text.Length));
            }
        }

        void SelectSingle(TextViewCoords item) {
            if (item.Valid && item.Character < m_lines[item.Line].m_text.Length) {
                SetSelection(item, NextChar(item));
            } else {
                SelectNone();
            }
        }

        int SelectionLineBase {
            get { 
                if (HaveSelection) return m_selStart.Line;
                else return 0;
            }
        }
        int SelectionLineCount {
            get {
                if (HaveSelection) return m_selEnd.Line - m_selStart.Line + 1;
                else return 0;
            }
        }

        bool HaveSelection {
            get { return m_selStart.Valid && m_selEnd.Valid && m_selStart < m_selEnd; } 
        }

        void ValidateSelection() {
            if (!HaveSelection) throw new ArgumentException();
        }

        void SetSelection(TextViewCoords selStart, TextViewCoords selEnd){
            if (selStart != m_selStart || selEnd != m_selEnd) {
                Rectangle oldSel = itemRangeRect(SelectionLineBase,SelectionLineCount);
                m_selStart = selStart;
                m_selEnd = selEnd;
                Rectangle newSel = itemRangeRect(SelectionLineBase,SelectionLineCount);

                /*using(Region rgn = new Region(oldSel)) using(Region exclude = new Region(oldSel)) {
                    rgn.Union(newSel); exclude.Intersect(newSel);
                    rgn.Exclude(exclude);
                    Invalidate(rgn);
                }*/
                using(Region rgn = new Region(oldSel)) {
                    rgn.Union(newSel);
                    Invalidate(rgn);
                }
                onSelectionChanged();
            }
        } 

       
        void StartDragSel(TextViewCoords start) {
            m_clickPoint = start;
            m_dragSelMode = true;
            Capture = true;
        }

        bool IsItemSelected(TextViewCoords item) {
            if (HaveSelection && item.Valid) {
                return item >= m_selStart && item < m_selEnd;
            } else {
                return false;
            }
        }

        private void ChannelView_MouseDown(object sender, MouseEventArgs e) {
            if (Focus()) {
                if (e.Button == MouseButtons.Left) {
                    TextViewCoords item = CoordsFromPoint(e.Location);
                    SelectSingle(item);
                    if (item.Valid) StartDragSel(item);
                } else if (e.Button == MouseButtons.Right) {
                    TextViewCoords item = CoordsFromPoint(e.Location);
                    if (item.Valid) {
                        if (!IsItemSelected(item)) {
                            StartDragSel(item);
                        }
                    }
                }
            }
        }

        private void ChannelView_MouseUp(object sender, MouseEventArgs e) {
            if (e.Button == MouseButtons.Left || e.Button == MouseButtons.Right) {
                Capture = false;
            }
        }

        private TextViewCoords NextChar( TextViewCoords c ) {
            var line = m_lines[c.Line];
            TextViewCoords ret;
            ret.Line = c.Line;
            ret.Character = line.m_text.NextChar(c.Character);
            return ret;
        }

        private void ChannelView_MouseMove(object sender, MouseEventArgs e) {
            if (m_dragSelMode) {
                TextViewCoords from = m_clickPoint;
                TextViewCoords to = CoordsFromPoint(e.Location);
                if (from.Valid && to.Valid) {
                    TextViewCoords selStart, selEnd;
                    if (from < to) {
                        selStart = from; selEnd = NextChar(to);
                    } else {
                        selStart = to; selEnd = NextChar(from);
                    }
                    selEnd.Character = Math.Min(selEnd.Character,m_lines[selEnd.Line].m_text.Length);
                    SetSelection(selStart,selEnd);
                }
            }
        }

        private void ChannelView_MouseCaptureChanged(object sender, EventArgs e) {
            m_dragSelMode = false;
        }

        private void ChannelView_Leave(object sender, EventArgs e) {
            SelectNone();
        }
        
        int ClientWidth {
            get { return m_effectiveClientWidth; }
        }

        void ApplyLineCountLimit() {
            int limit = Math.Max(m_lineCountLimit, 10);
            if ( (Int64)m_lines.Count * 3 / 4 > limit) {
                m_lines.RemoveRange(0, m_lines.Count - limit);
                if (m_lines.Count > 0) {
                    int offsetDelta = m_lines[0].m_offset;
                    foreach(LineEntry entry in m_lines) {
                        entry.m_offset -= offsetDelta;
                    }
                }
                ViewAreaSize = new Size(0,itemRangeHeight(0,m_lines.Count));
                Invalidate();
            }
        }

        void ApplyFontChange() {
            foreach(LineEntry line in m_lines) {
                line.m_wordSpecs = line.m_text.GetWords(m_fontArray);
            }
            RecalculateLineHeights();
        }

        private void ChannelView_FontChanged(object sender, EventArgs e) {
            if (m_fontArray.BaseFont != Font) {
                m_fontArray.Reinitialize(Font);
                ApplyFontChange();
            }
        }

        private void openURLToolStripMenuItem_Click(object sender, EventArgs e) {
            tryOpenURL(false);
        }
        bool tryOpenURL(bool bCopy) {
            string selectedText = SelectedText;
            if (URLUtils.IsURL(selectedText)) {
                try {
                    if (!selectedText.Contains("://")) selectedText = "http://" + selectedText;
                    if (bCopy) {
                        Clipboard.SetText( selectedText );
                    } else {
                        System.Diagnostics.Process.Start(selectedText);
                    }
                } catch(Exception) {
                    return false;
                }
                return true;
            } else return false;
        }
        private void ChannelView_MouseDoubleClick(object sender, MouseEventArgs e) {
            if (e.Button == MouseButtons.Left) {
                Capture = false;
                TextViewCoords item = CoordsFromPoint(e.Location);
                if (item.Valid) {
                    int selBase, selLen;
                    if (StringUtils.DetectLink(m_lines[item.Line].m_text.ToString(),item.Character,out selBase,out selLen)) {
                        SetSelection(new TextViewCoords(item.Line,selBase),new TextViewCoords(item.Line,selBase+selLen));
                        tryOpenURL(m_urlCopy);
                    } else if (StringUtils.DetectWord(m_lines[item.Line].m_text.ToString(),item.Character,out selBase,out selLen)) {
                        SetSelection(new TextViewCoords(item.Line,selBase),new TextViewCoords(item.Line,selBase+selLen));
                    }
                }
            }
        }
        private void ChannelView_SystemColorsChanged(object sender, EventArgs e) {
            Invalidate();
        }

        public int TextLineHeight {
            get { return TextRenderer.MeasureText("ASDF",Font).Height; }
        }
        public int TextPageHeight {
            get { return ClientSize.Height * 2 / 3; }
        }

        public void ScrollDelta(int delta) {
            ViewAreaPosition = new Point(0, ViewAreaPosition.Y + delta);
            updateStickToBottom();
        }
        private void updateStickToBottom()
        {
            m_stickToBottom = this.ViewAreaOffset == 0;
        }

        protected override bool IsInputKey(Keys keyData) {
            switch(keyData) {
                case Keys.PageUp:
                case Keys.PageDown:
                case Keys.Up:
                case Keys.Down:
                case Keys.Home:
                case Keys.End:
                    return true;
                default:
                    return base.IsInputKey(keyData);
            }
        }

        protected override void OnKeyDown(KeyEventArgs e) {
            switch(e.KeyCode) {
                case Keys.Home:
                    ViewAreaPosition = new Point(0,0);
                    break;
                case Keys.End:
                    ViewAreaOffset = 0;
                    break;
                case Keys.Escape:
                    e.Handled = true;
                    SelectNone();
                    break;
                case Keys.PageUp:
                    e.Handled = true;
                    ScrollDelta(-TextPageHeight);
                    break;
                case Keys.PageDown:
                    e.Handled = true;
                    ScrollDelta(TextPageHeight);
                    break;
                case Keys.Up:
                    e.Handled = true;
                    ScrollDelta(-TextLineHeight);
                    break;
                case Keys.Down:
                    e.Handled = true;
                    ScrollDelta(TextLineHeight);
                    break;
                default:
                    base.OnKeyDown(e);
                    break;
            }
        }
        private void ChannelView_MouseClick(object sender, MouseEventArgs e) {
            Focus();
        }

        Point ViewAreaPosition {
            get {
                Point asdf = AutoScrollPosition;
                return new Point(-asdf.X,-asdf.Y);
            }
            set {
                AutoScrollPosition = value;
            }
        }

        int ViewAreaOffset_FromClientHeight(int clientHeight) {
            return Math.Max(0,ViewAreaSize.Height - (ViewAreaPosition.Y + clientHeight));
        }

        int ViewAreaOffset {
            get {
                int v = ViewAreaOffset_FromClientHeight(ClientSize.Height);
                // System.Diagnostics.Debug.WriteLine("ViewAreaOffset: " + v);
                return v;
            }
            set {
                int Y = Math.Max(0,ViewAreaSize.Height - (value + ClientSize.Height));
                // System.Diagnostics.Debug.WriteLine("Setting ViewAreaOffset to: " + value + ", Y: " + Y + " Client height: " + ClientSize.Height + ", ViewArea: " + ViewAreaSize.Height);
                ViewAreaPosition = new Point(0,Y);
                // System.Diagnostics.Debug.WriteLine("ViewAreaPosition set to: " + ViewAreaPosition.Y);
            }
        }

        bool IsDragSelBlockingScroll {
            get {
                if (m_dragSelMode && HaveSelection) {
                    if (m_selStart.Line != m_selEnd.Line) return true;
                    if (Math.Abs(m_selStart.Character - m_selEnd.Character) != 1) return true;
                }
                return false;
            }
        }

        Size ViewAreaSize {
            get {
                return AutoScrollMinSize;
            }
            set {
                AutoScrollMinSize = value;
                if (m_stickToBottom && !IsDragSelBlockingScroll) {
                    ViewAreaOffset = 0;
                } else {
                    m_stickToBottom = false;
                }
            }
        }

        void ChannelView_Disposed(object sender, EventArgs e) {
            if (m_fontArray != null) {
                m_fontArray.Dispose();
                m_fontArray = null;
            }
        }

        // does nothing and returns true when the bottom is already visible; returns false when the bottom wasn't visible and has just been scrolled to
        public bool EnsureBottomVisible() {
            if (ViewAreaOffset == 0) return true;
            m_stickToBottom = true;
            ViewAreaOffset = 0;
            return false;
        }
       
        TextViewCoords m_selStart, m_selEnd, m_clickPoint;
        List<LineEntry> m_lines;
        bool m_dragSelMode;
        int m_effectiveClientWidth;
        int m_lastSeenClentHeight;
        bool m_insideSizeChange;
        FontArray m_fontArray;
        Color m_userJoinColor;
        Color m_userPartColor;
        bool m_urlCopy;
        int m_lineCountLimit;
        bool m_stickToBottom = true;
    }
}
