using System;
using System.Collections.Generic;
using System.Data;
using System.Text;

namespace monochrome {
    public enum TextStyleType {
        Plain,
        Gray,
        UserJoin,
        UserPart,
        UserName,
        CustomColor,
        Hyperlink
    }

    public enum TextStyleFlags {
        Default = 0,
        Bold = 1 << 0,
        Italic = 1 << 1,
        Underline = 1 << 2,
        Inverse = 1 << 3,
        StrikeThrough = 1 << 4,
        DoNotMerge = 1 << 5,
    };

    public partial struct TextStyle {
        public TextStyle(ColorDesc _Color) {
            Type = TextStyleType.Plain;
            CustomColor = ColorDesc.Empty;
            TypeParam = 0;
            Flags = TextStyleFlags.Default;
            UserFlags = TextStyleFlags.Default;
            Highlight = false;
            SetCustomColor(_Color);
        }

        public TextStyle AddHighlight() {
            TextStyle style = this;
            style.Highlight = true;
            return style;
        }

        public static TextStyle UserName(int UserData,bool IsSelf) {
            TextStyle temp = new TextStyle();
            temp.Type = TextStyleType.UserName;
            temp.TypeParam = UserData;
            temp.Bold = IsSelf;
            return temp;
        }

        public TextStyle TurnToHyperlink() {
            TextStyle ts = new TextStyle();
            ts.Underline = true;
            ts.Type = TextStyleType.Hyperlink;
            ts.Highlight = this.Highlight;
            return ts;
        }

        public static TextStyle NickEx(TextStyle baseStyle) {
            return baseStyle.AddHighlight();
        }
        public static TextStyle Nick {
            get { return NickEx(new TextStyle()); }
        }
        public static TextStyle OwnNick {
            get { 
                TextStyle temp = Nick;
                temp.Bold = true;
                return temp;
            }
        }
        

        public TextStyleType Type;
        public ColorDesc CustomColor;
        public int TypeParam;
        public bool Highlight;
        public TextStyleFlags Flags;
        public TextStyleFlags UserFlags;

        bool GetFlag(TextStyleFlags flag) {return ((UserFlags | Flags) & flag) == flag;}
        void SetFlag(TextStyleFlags flag,bool state) {if (state) UserFlags |= flag; else UserFlags &= ~flag;}

        void RevertColor() {Type = TextStyleType.Plain; CustomColor = ColorDesc.Empty; TypeParam = 0;}
        public void RevertColor(TextStyle orig) {Type = orig.Type; CustomColor = orig.CustomColor; TypeParam = orig.TypeParam;}
        public void SetCustomColor(ColorDesc c) {RevertColor(); Type = TextStyleType.CustomColor; CustomColor = c;}

        public bool Bold { 
            get {return GetFlag(TextStyleFlags.Bold); }
            set {SetFlag(TextStyleFlags.Bold,value); }
        }

        public bool Italic {
            get {return GetFlag(TextStyleFlags.Italic); }
            set {SetFlag(TextStyleFlags.Italic,value); }
        }
        public bool Underline { 
            get {return GetFlag(TextStyleFlags.Underline); }
            set {SetFlag(TextStyleFlags.Underline,value); }
        }
        public bool Inverse { 
            get {return GetFlag(TextStyleFlags.Inverse); }
            set {SetFlag(TextStyleFlags.Inverse,value); }
        }
        public bool StrikeThrough { 
            get {return GetFlag(TextStyleFlags.StrikeThrough); }
            set {SetFlag(TextStyleFlags.StrikeThrough,value); }
        }
        public bool DoNotMerge {
            get { return GetFlag(TextStyleFlags.DoNotMerge); }
            set { SetFlag(TextStyleFlags.DoNotMerge, value); }
        }

        public override int GetHashCode() {
            return 0; // not used
        }
        public static bool Equals(TextStyle elem1, TextStyle elem2) {
            return elem1.Type == elem2.Type && elem1.CustomColor == elem2.CustomColor && elem1.TypeParam == elem2.TypeParam && elem1.Flags == elem2.Flags &&  elem1.UserFlags == elem2.UserFlags;
        }

        public static bool operator==(TextStyle elem1, TextStyle elem2) {
            return Equals(elem1,elem2);
        }
        public static bool operator!=(TextStyle elem1, TextStyle elem2) {
            return !Equals(elem1,elem2);
        }
        public override bool Equals(object obj) {
            if (obj is TextStyle) {
                return Equals(this,(TextStyle) obj);
            } else {
                return false;
            }
        }
        public static TextStyle FromType(TextStyleType type) {
            TextStyle temp = new TextStyle(); temp.Type = type; return temp;
        }
        public static TextStyle FromTypeHL(TextStyleType type) {
            TextStyle temp = new TextStyle(); temp.Type = type; temp.Highlight = true; return temp;
        }

        public static TextStyle Gray { get {return FromType(TextStyleType.Gray); } }
        public static TextStyle GrayHighlight { get {return FromTypeHL(TextStyleType.Gray); } }
        public static TextStyle Default { get {return FromType(TextStyleType.Plain); } }
        public static TextStyle DefaultHighlight { get {return FromTypeHL(TextStyleType.Plain); } }
        public static TextStyle UserJoin { get {return FromType(TextStyleType.UserJoin); } }
        public static TextStyle UserJoinHighlight { get {return FromTypeHL(TextStyleType.UserJoin); } }
        public static TextStyle UserPart { get {return FromType(TextStyleType.UserPart); } }
        public static TextStyle UserPartHighlight { get {return FromTypeHL(TextStyleType.UserPart); } }

    };

    public partial struct TextLineElem {
        public TextLineElem(string _text, TextStyle _style,object _tag) {Text = _text; Style = _style; Tag = _tag;}
        public TextLineElem(string _text, TextStyle _style) {Text = _text; Style = _style; Tag = null;}
        public TextLineElem(string _text) : this(_text,TextStyle.Default) {}

        public TextStyle Style;
        public string Text;
        public object Tag;

    };

    public partial class TextLine {

        LinkedList<TextLineElem> Elems;

        public IEnumerable<TextLineElem> Elements { get { return Elems; } }

        public int NextChar( int idxPrev ) {
            int walk = 0;
            foreach ( TextLineElem elem in Elems ) {
                int next = walk + elem.Text.Length;
                if (next > idxPrev) {
                    foreach (string character in TextUtils.CharsInString(elem.Text)) {
                        walk += character.Length;
                        if (walk > idxPrev) return walk;
                    }
                    // should not get here
                    return walk;
                }
                walk = next;
            }
            return idxPrev;
        }

        public static TextLine operator+(TextLine line1,TextLine line2) {
            TextLine ret = new TextLine();
            foreach(TextLineElem elem in line1.Elems) ret.Add(elem);
            foreach(TextLineElem elem in line2.Elems) ret.Add(elem);
            return ret;
        }
        public static TextLine operator+(string line1,TextLine line2) {
            TextLine ret = new TextLine();
            ret.Add( line1 );
            foreach(TextLineElem elem in line2.Elems) ret.Add(elem);
            return ret;
        }
        public static TextLine operator+(TextLine line1,string line2) {
            TextLine ret = new TextLine();
            foreach(TextLineElem elem in line1.Elems) ret.Add(elem);
            ret.Add( line2 );
            return ret;
        }

        public TextLine() {Elems = new LinkedList<TextLineElem>();}

        public static TextLine Simple(string text) {
            return Simple(text,TextStyle.Default);
        }

        public static TextLine ContextPrefix(string context) {
            return TextLine.Simple("[" + context + "] ",TextStyle.Gray);
        }

        public static TextLine Simple(string text,TextStyle style) {
            TextLine line = new TextLine();
            line.Add(text,style,null);
            return line;
        }
        public static TextLine Highlight(string text, bool IsHighlight) {
            return Simple(text,IsHighlight ? TextStyle.DefaultHighlight : TextStyle.Default);
        }

        static byte[] HashString(string str) {
            return PlatformMethods.StringMD5( str );
        }
        static int HashStringSimple(string str) {
            byte[] temp = HashString(str);
            int ret = 0;
            for(int walk = 0; walk < 4; ++walk) {
                if (temp.Length > walk) {
                    ret |= (int) temp[walk] << (walk * 8);
                }
            }
            return ret;
        }

        public static TextLine UserName(string text,bool self) {return Simple(text,TextStyle.UserName(HashStringSimple(text),self));}
        public static TextLine UserNameEx(string text,TextStyle baseStyle) {
            return Simple(text,TextStyle.NickEx(baseStyle));
        }

        public delegate string FilterFunc(string text);
        public TextLine Filter(FilterFunc fn) {
            TextLine ret = new TextLine();
            foreach(TextLineElem elem in Elems) {
                string processed = fn(elem.Text);
                if (processed != "") {
                    ret.Elems.AddLast(new TextLineElem(processed,elem.Style));
                }
            }
            return ret;
        }

        public int Length {
            get { 
                int total = 0;
                foreach(TextLineElem elem in Elems) total += elem.Text.Length;
                return total;
            }
        }

        public override string ToString() {
            string ret = "";
            foreach(TextLineElem elem in Elems) {
                ret += elem.Text;
            }
            return ret;
        }


        void Add(TextLineElem elem) {Add(elem.Text,elem.Style,elem.Tag);}
        void Add(string text) {Add(text,TextStyle.Default,null);}
        bool TryMergeWithLast(string text, TextStyle style,object tag) {
            if (Elems.Count > 0) {
                TextLineElem e = Elems.Last.Value;
                if (style == e.Style && tag == e.Tag && ! style.DoNotMerge ) {
                    e.Text += text;
                    Elems.Last.Value = e;
                    return true;
                }
            }
            return false;
        }
        void Add(string text, TextStyle style,object tag) {
            if (text.Length > 0) {
                if (!TryMergeWithLast(text,style,tag)) Elems.AddLast(new TextLineElem(text,style,tag));
            }
        }

        static string SubstringHelper(string source,int wantedOffset, int wantedLength, int sourceOffset) {
            wantedOffset -= sourceOffset;
            if (wantedOffset < 0) {
                wantedLength += wantedOffset;
                wantedOffset = 0;
            }

            wantedLength = Math.Min( source.Length - wantedOffset, wantedLength );
            if (wantedLength > 0) {
                return source.Substring( wantedOffset,wantedLength );
            }
            return "";
        }

        public TextLine Substring(int offset) {return Substring(offset,int.MaxValue);}
        public TextLine Substring(int offset, int lengthLimit) {
            TextLine output = new TextLine();
            int walk = 0;
            foreach(TextLineElem elem in Elems) {
                output.Add(SubstringHelper(elem.Text,offset,lengthLimit,walk),elem.Style,elem.Tag);
                walk += elem.Text.Length;
            }
            return output;
        }

        public TextLine Retag(object tag) {
            TextLine output = new TextLine();
            foreach(TextLineElem elem in Elems) output.Add(elem.Text,elem.Style,tag);
            return output;
        }

        struct LineSplitEntry {
            public LineSplitEntry(int _CharBase, int _CharCount, int _YOffset, int _Height) {CharBase = _CharBase; CharCount = _CharCount; YOffset = _YOffset; Height = _Height;}
            public int CharBase, CharCount;
            public int YOffset, Height;
        }

        class LineSplitHelper {
            public LineSplitHelper() {Data = new LinkedList<LineSplitEntry>();}
            LinkedList<LineSplitEntry> Data;

            public void OnHeight(int newHeight) {
                Height = Math.Max(Height,newHeight);
            }

            public void AddEntry(int NewBase) {
                if (NewBase > CharBase) {
                    Data.AddLast(new LineSplitEntry(CharBase,NewBase - CharBase,YOffset,Height));
                    CharBase = NewBase;
                    YOffset += Height;
                    Height = 0;
                }
            }
            public LinkedList<LineSplitEntry> Finalize() {
                Data.AddLast(new LineSplitEntry(CharBase,int.MaxValue,YOffset,Height));
                return Data;
            }
            
            int YOffset;
            int Height;
            int CharBase;
        };

        static LinkedList<LineSplitEntry> SplitLines(LinkedList<WordSpecs> words, int targetWidth) {
            LineSplitHelper helper = new LineSplitHelper();
            int XPos = 0;
            foreach(WordSpecs walk in words) {
                if (XPos > 0 && XPos + walk.WidthHT > targetWidth) {
                    helper.AddEntry(walk.BaseChar);
                    XPos = 0;
                }
                if (XPos + walk.WidthHT > targetWidth) {
                    for(int wcWalk = 0; wcWalk < walk.CharSpecs.Length; ++wcWalk) {
                        CharSpecs c = walk.CharSpecs[wcWalk];
                        if (XPos > 0 && XPos + c.WidthHT > targetWidth) {
                            helper.AddEntry(walk.BaseChar + wcWalk);
                            XPos = 0;
                        }
                        helper.OnHeight(c.Height);
                        XPos += c.Width;
                    }
                    XPos += walk.Spacing;
                } else {
                    helper.OnHeight(walk.Height);
                    XPos += walk.Width + walk.Spacing;
                }
            }
            return helper.Finalize();
        }

        /*public void Draw(Graphics target, Font font, int paddingWTF, int X, int Y, int widthLimit,TextStyle.ColorResolver colorResolver) {
            __Process(font,widthLimit,paddingWTF,X,Y,target,true,colorResolver);
        }*/

        static char readCharGuarded(string text, int index) {
            return index < text.Length ? text[index] : (char)0;
        }


        static ColorDesc translateMircColor(int index) {
            switch(index) {
                case 0:     return ColorDesc.FromString("FFFFFF"); // Color.White;
                case 1:     return ColorDesc.FromString("000000"); // Color.Black;
                case 2:     return ColorDesc.FromString("0000FF"); // Color.Blue;
                case 3:     return ColorDesc.FromString("008000"); // Color.Green;
                case 4:     return ColorDesc.FromString("FF0000"); // Color.Red;
                case 5:     return ColorDesc.FromString("A52A2A"); // Color.Brown
                case 6:     return ColorDesc.FromString("800080"); // Color.Purple; 
                case 7:     return ColorDesc.FromString("FFA500"); // Color.Orange;
                case 8:     return ColorDesc.FromString("FFFF00"); // Color.Yellow;
                case 9:     return ColorDesc.FromString("90EE90"); // Color.LightGreen;
                case 10:    return ColorDesc.FromString("00FFFF"); // Color.Cyan;
                case 11:    return ColorDesc.FromString("E0FFFF"); // Color.LightCyan;
                case 12:    return ColorDesc.FromString("ADD8E6"); // Color.LightBlue;
                case 13:    return ColorDesc.FromString("FFC0CB"); // Color.Pink
                case 14:    return ColorDesc.FromString("808080"); // Color.Gray;
                case 15:    return ColorDesc.FromString("D3D3D3"); // Color.LightGray;
                default:
                    return ColorDesc.FromString("000000"); // Color.Black;
            }
        }


        static int readMircNonsense(string text, int offset, out int width) {
            char c1 = readCharGuarded(text,offset);
            char c2 = readCharGuarded(text,offset+1);
            if (TextUtils.IsDigitSimple(c1)) {
                if (TextUtils.IsDigitSimple(c2)) {
                    width = 2;
                    return Convert.ToInt32(c1.ToString() + c2.ToString());
                } else {
                    width = 1;
                    return Convert.ToInt32(c1.ToString());
                }
            } else {
                width = 0;
                return 0;
            }
        }

        public static TextLine FromIRC(string text, TextStyle defaultStyle) {
            //return Simple(IRCUtils.Misc.ProcessIncoming(input),defaultStyle);
            
            TextStyle style = defaultStyle;
            TextLine output = new TextLine();
            int added = 0;
            for(int walk = 0; walk < text.Length; ) {
                
                int code = (int) text[walk];
                if (code < 32) {
                    TextStyle stylePrev = style;
                    int skip = 1;
                    string append = null;
                    switch(code) {
                        case 4:
                            try {
                                if (walk + 6 < text.Length) {
                                    int red = Convert.ToInt32(text.Substring(walk+1,2),16);
                                    int green = Convert.ToInt32(text.Substring(walk+3,2),16);
                                    int blue = Convert.ToInt32(text.Substring(walk+5,2),16);
                                    style.CustomColor = ColorDesc.FromARGB(red,green,blue);
                                    skip = 7;
                                }
                            } catch(FormatException) {
                            }
                            break;
                        case 3:
                            {
                                skip = 1;
                                int cWidth;
                                int cCode = readMircNonsense(text,walk+skip,out cWidth);
                                skip += cWidth;
                                if (cCode > 0) {
                                    style.SetCustomColor(translateMircColor(cCode));
                                    if (readCharGuarded(text,walk+skip) == ',') {
                                        cCode = readMircNonsense(text,walk+skip+1,out cWidth);
                                        if (cWidth > 0) {
                                            skip += cWidth + 1;
                                            //todo set background color
                                        }
                                    }
                                } else {
                                    style.RevertColor(defaultStyle);
                                }
                            }
                            break;
                        case 15:
                            style.RevertColor(defaultStyle);
                            break;
                        case 2:
                            style.Bold = !style.Bold;
                            break;
                        case 18:
                            style.Inverse = !style.Inverse;
                            break;
                        case 29:
                            style.Italic = !style.Italic;
                            break;
                        case 31:
                            style.Underline = !style.Underline;
                            break;
                        case 27:
                            if (readCharGuarded(text,walk+1) == '[') {
                                int walkEx = walk + 2;
                                for(;;) {
                                    int numLen = 0;
                                    while(numLen < 2 && TextUtils.IsDigitSimple( readCharGuarded(text,walkEx + numLen) ) ) numLen++;
                                    char following = readCharGuarded(text,walkEx + numLen);
                                    if (numLen < 1 || (following != ';' && following != 'm')) break;
                                    int what = Convert.ToInt32(text.Substring(walkEx,numLen));
                                    switch(what) {
                                        case 0:
                                            style.RevertColor(defaultStyle);
                                            style.UserFlags = TextStyleFlags.Default;
                                            break;
                                        case 1:
                                            style.Bold = true;
                                            break;
                                        case 3:
                                            style.Italic = true;
                                            break;
                                        case 4:
                                            style.Underline = true;
                                            break;
                                        case 7:
                                            style.Inverse = true;
                                            break;
                                        case 9:
                                            style.StrikeThrough = true;
                                            break;
                                        case 22:
                                            style.Bold = false;
                                            break;
                                        case 23:
                                            style.Italic = false;
                                            break;
                                        case 24:
                                            style.Underline = false;
                                            break;
                                        case 27:
                                            style.Inverse = false;
                                            break;
                                        case 29:
                                            style.StrikeThrough = false;
                                            break;
                                        case 30:
                                            style.SetCustomColor(/*Color.Black*/ ColorDesc.FromString("000000"));
                                            break;
                                        case 31:
                                            style.SetCustomColor(/*Color.Red*/ ColorDesc.FromString("FF0000"));
                                            break;
                                        case 32:
                                            style.SetCustomColor(/*Color.Green*/ ColorDesc.FromString("008000"));
                                            break;
                                        case 33:
                                            style.SetCustomColor(/*Color.Yellow*/ ColorDesc.FromString("FFFF00"));
                                            break;
                                        case 34:
                                            style.SetCustomColor(/*Color.Blue*/ ColorDesc.FromString("0000FF"));
                                            break;
                                        case 35:
                                            style.SetCustomColor(/*Color.Magenta*/ ColorDesc.FromString("FF00FF"));
                                            break;
                                        case 36:
                                            style.SetCustomColor(/*Color.Cyan*/ ColorDesc.FromString("00FFFF"));
                                            break;
                                        case 37:
                                            style.SetCustomColor(/*Color.White*/ ColorDesc.FromString("FFFFFF"));
                                            break;
                                        case 39:
                                            style.RevertColor(defaultStyle);
                                            break;
                                    }
                                    walkEx += numLen + 1;
                                    if (following == 'm') break;
                                }
                                skip = walkEx - walk;
                            }
                            break;
                        default:
                            //append = "[" + code + "]";
                            break;
                    }
                    output.Add(text.Substring(added,walk-added),stylePrev,null);
                    if (append != null) output.Add(append,style,null);
                    walk = added = walk + skip;
                } else {
                    ++walk;
                }
            }
            
            if (added < text.Length) output.Add(text.Substring(added),style,null);
            return output;
        }
        public static TextLine FromIRC(string input) {
            return FromIRC(input,TextStyle.Default);
        }

        /*public static TextLine FromIRCHighlightable(string input, bool highlight) {
            return FromIRC(input, highlight ? TextStyle.Highlight : TextStyle.Default);
        }*/

        static int ExtractWord(string str, int extractBase) {
            int walk = extractBase;
            while(walk < str.Length && !char.IsSeparator(str[walk])) ++walk;
            return walk - extractBase;
        }
        static int ExtractSpacing(string str, int extractBase) {
            int walk = extractBase;
            while(walk < str.Length && char.IsSeparator(str[walk])) ++walk;
            return walk - extractBase;
        }
        
        public struct CharSpecs {
            public CharSpecs(int width, int widthHT, int height) { Width = width; WidthHT = widthHT; Height = height; }
            public int Width, WidthHT, Height;
        };

        public struct WordSpecs {
            public int BaseChar, Height, Width, Spacing, WidthHT;
            public CharSpecs[] CharSpecs;
        };

        
        public static int HeightFromWords(LinkedList<WordSpecs> words,int targetWidth) {
            LinkedList<LineSplitEntry> entries = SplitLines(words,targetWidth);
            if (entries.Count == 0) return 0;
            LineSplitEntry last = entries.Last.Value;
            return last.YOffset + last.Height;
        }

       
        public delegate TextStyle FilterStyleFunc(TextStyle style);

        public TextLine FilterStyle(FilterStyleFunc func) {
            TextLine output = new TextLine();
            foreach(TextLineElem elem in Elems) {
                output.Add(elem.Text,func(elem.Style),elem.Tag);
            }
            return output;
        }

        static TextStyle FilterStyle_CustomFormatting(TextStyle style) {
            style.UserFlags = TextStyleFlags.Default;
            return style;
        }
        static TextStyle Filterstyle_CustomColors(TextStyle style) {
            if (style.Type == TextStyleType.CustomColor) {
                style.Type = TextStyleType.Plain;
                style.CustomColor = ColorDesc.Empty;
            }
            return style;
        }

        static TextStyle FilterStyle_AddHighlight(TextStyle style) {
            return style.AddHighlight();
        }

        public TextLine AddHighlight() {
            return FilterStyle(new FilterStyleFunc(FilterStyle_AddHighlight));
        }

        public TextLine StripCustomFormatting() {
            return FilterStyle(new FilterStyleFunc(FilterStyle_CustomFormatting));
        }
        public TextLine StripCustomColors() {
            return FilterStyle(new FilterStyleFunc(Filterstyle_CustomColors));
        }

        
        public TextLine DetectHyperlinks() {
            TextLine output = new TextLine();
            foreach (TextLineElem elem in Elems) {
                var words = WordInString.SplitEx( elem.Text, new StringUtils.CharFilter( char.IsSeparator ) );
                bool gotLinks = false;
                foreach( var word in words ) {
                    string strWord = elem.Text.Substring(word.Base, word.Length);
                    if ( URLUtils.IsURL( strWord ) ) {
                        gotLinks = true; break;
                    }
                }

                if ( ! gotLinks ) {
                    // the easy case, nothing changes here
                    output.Add( elem );
                    continue;
                }

                // the hard case, extract URL and flag properly
                int iDone = 0, iWalk = 0;
                foreach( var word in words ) {
                    if (URLUtils.IsURL(elem.Text.Substring(word.Base, word.Length))) {
                        iWalk = word.Base;
                        if (iDone < iWalk) {
                            output.Add(elem.Text.Substring(iDone, iWalk - iDone), elem.Style, elem.Tag);
                        }

                        output.Add( elem.Text.Substring( word.Base, word.Length ), elem.Style.TurnToHyperlink(), elem.Tag );

                        iDone = word.Base + word.Length;
                    }
                    iWalk = word.Base + word.Length;
                }
                iWalk = elem.Text.Length;
                if ( iDone < iWalk ) {
                    output.Add( elem.Text.Substring(iDone, iWalk-iDone), elem.Style, elem.Tag );
                }
            }
            return output;
        }
    };

    public struct TextViewCoords {
        public TextViewCoords(int line, int character) {Line = line; Character = character;}
        public int Line, Character;

        static public TextViewCoords Invalid {
            get { return new TextViewCoords(-1,-1);}
        }

        public bool Valid {
            get { return Line >= 0 && Character >= 0; }
        }

        public override int GetHashCode() {
            return Line * 10000 + Character;
        }

        public override bool Equals(object obj) {
            if (obj is TextViewCoords) {
                TextViewCoords other = (TextViewCoords) obj;
                return Line == other.Line && Character == other.Character;
            } else {
                return false;
            }
        }

        public static int Compare(TextViewCoords item1,TextViewCoords item2) {
            if (item1.Line > item2.Line) return 1;
            else if (item1.Line < item2.Line) return -1;
            else if (item1.Character > item2.Character) return 1;
            else if (item1.Character < item2.Character) return -1;
            else return 0;
        }

        public static bool operator>(TextViewCoords item1,TextViewCoords item2) {return Compare(item1,item2) > 0;}
        public static bool operator<(TextViewCoords item1,TextViewCoords item2) {return Compare(item1,item2) < 0;}
        public static bool operator==(TextViewCoords item1,TextViewCoords item2) {return Compare(item1,item2) == 0;}
        public static bool operator!=(TextViewCoords item1,TextViewCoords item2) {return Compare(item1,item2) != 0;}
        public static bool operator>=(TextViewCoords item1,TextViewCoords item2) {return Compare(item1,item2) >= 0;}
        public static bool operator<=(TextViewCoords item1,TextViewCoords item2) {return Compare(item1,item2) <= 0;}

        public override string ToString() {
            return "(" + Line + "," + Character + ")";            
        }

    }


    class URLUtils {
        static string[] prefixes = new string[] { "www.", "http://", "https://", "ftp://" };

        public static bool IsURL(string text) {
            // Detected apparent URL: (uid114373@gateway/web/irccloud.com/x-urfvdvjhrdnedwha)
            foreach( var p in prefixes ) {
                if (text.StartsWith(p, StringComparison.OrdinalIgnoreCase ) ) return true;
            }

            if ( text.Contains("@") ) return false;


            // domain.tld/path  see if there's a dot before the first slash
            int iSlash = text.IndexOf('/');
            if ( iSlash >= 0 ) {
                int iDot = text.IndexOf('.');
                if ( iDot >= 0 && iDot < iSlash ) {
                    return true;
                }
            }
            return false;
        }
    }

    class StringUtils {
        public static bool IsWordSeparator(char c) {
            return char.IsSeparator(c) || char.IsPunctuation(c);
        }
        public delegate bool CharFilter(char c);

        public static bool DetectWord(string text, int baseChar, out int wordBase, out int wordLen) {
            return DetectWordEx(text, baseChar,out wordBase, out wordLen,new CharFilter(IsWordSeparator));
        }
        public static bool DetectLink(string text, int baseChar, out int wordBase, out int wordLen) {
            if (!DetectWordEx(text, baseChar,out wordBase, out wordLen,new CharFilter(char.IsSeparator))) return false;
            return URLUtils.IsURL(text.Substring(wordBase,wordLen));
        }

        public static bool DetectWordEx(string text, int baseChar, out int wordBase, out int wordLen,CharFilter separatorCheck) {
            int walk = baseChar;
            while(walk > 0 && !separatorCheck(text[walk-1])) --walk;
            wordBase = walk;
            walk = baseChar;
            while(walk < text.Length && !separatorCheck(text[walk])) ++walk;
            wordLen = walk - wordBase;
            return wordLen > 0;
        }
    };
    struct WordInString {
        public int Base, Length;
        public static LinkedList<WordInString> Split(string s) {
            return SplitEx( s, new StringUtils.CharFilter( StringUtils.IsWordSeparator ) );
        }
        public static LinkedList<WordInString> SplitEx( string s, StringUtils.CharFilter filter ) {
            var ret = new LinkedList<WordInString>();
            int wordBase = 0;
            bool inWord = false;
            for( int i = 0; i < s.Length; ++ i) {
                char c = s[i];
                if ( filter(c) ) {
                    if ( inWord ) {
                        inWord = false;
                        ret.AddLast(new WordInString() { Base = wordBase, Length = i - wordBase });
                    }
                } else {
                    if (!inWord) {
                        inWord = true;
                        wordBase = i;
                    }
                }
            }
            if ( inWord ) {
                int i = s.Length;
                ret.AddLast( new WordInString() { Base = wordBase, Length = i - wordBase } );
            }

            return ret;
        }
    }

    public class TextUtils {

        public static string Rainbow( string str ) {
            string ret = "";
            // index 2...13
            const int min = 2;
            const int max = 13;
            const int range = max - min + 1;
            for( int i = 0; i < str.Length; ++i ) {
                ret += (char) 3;
                ret += (i%range+min).ToString();
                ret += str[i];
            }
            return ret;
        }

        static void swapItems<T>(ref T item1, ref T item2) {
            T temp = item1; item1 = item2; item2 = temp;
        }

        static char[] CharSwapTable = {'(',')','[',']','{','}','<','>'};
        static char ReverseChar(char c) {
            for(int walk = 0; walk < CharSwapTable.Length; ++walk) {
                if (CharSwapTable[walk] == c) {
                    return CharSwapTable[walk^1];
                }
            }
            return c;
        }

        public static string Reverse(string s) {
            char[] chars = s.ToCharArray();
            for(int walk = 0; walk < chars.Length / 2; ++walk) {
                int walk2 = chars.Length - walk - 1;
                swapItems(ref chars[walk],ref chars[walk2]);
            }
            //fix any linebreaks that we just broke
            for(int walk = 0; walk + 1 < chars.Length; ) {
                if (chars[walk] == '\n' && chars[walk+1] == '\r') {
                    swapItems(ref chars[walk],ref chars[walk+1]);
                    walk += 2;
                } else {
                    ++walk;
                }
            }
            for(int walk = 0; walk < chars.Length; ++walk) chars[walk] = ReverseChar(chars[walk]);
            return new string(chars);
        }
        public static bool FindTruncationPoint(Encoding encoding,string text,int truncationBytes,out int truncatLenChars) {
            return FindTruncationPoint(encoding,text,truncationBytes,0,out truncatLenChars);
        }
        public static bool FindTruncationPoint(Encoding encoding,string text,int truncationBytes,int baseOffset,out int truncatLenChars) {
            int bytes = 0;
            for(int walk = baseOffset; walk < text.Length; ++walk) {
                bytes += encoding.GetByteCount(new char[]{text[walk]});
                if (bytes > truncationBytes) {
                    truncatLenChars = walk - baseOffset; 
                    return true;
                }
            }
            truncatLenChars = 0;
            return false;
        }

        public static bool IsStringNumeric( string s ) {
            bool v = false;
            foreach( char c in s ) {
                if (!IsDigitSimple( c ) ) return false;
                v = true;
            }
            return v;
        }
        public static bool IsStringIPv4(string hostname) {
            var components = hostname.Split('.');
            if (components.Length != 4) return false;
            foreach (string s in components) {
                if ( !IsStringNumeric(s) ) return false;
            }
            foreach (string s in components) {
                try {
                    var val = UInt64.Parse(s);
                    if ( val > 255 ) return false;
                } catch {
                    return false;
                }
            }
            return true;

        }
        public static bool IsDigitSimple(char c) {
            switch(c) {
                case '0':
                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7':
                case '8':
                case '9':
                    return true;
                default:
                    return false;
            }
        }
        public static LinkedList<string> CharsInString(string s) {
            var ret = new LinkedList<string>();
            char prev = (char)0;
            foreach (char c in s) {
                if (c > 0xDC00) {
                    if (prev != (char)0) {
                        ret.AddLast(prev.ToString() + c.ToString());
                    }
                    prev = (char)0;
                } else if (c > 0xD800) {
                    prev = c;
                } else {
                    ret.AddLast(c.ToString());
                    prev = (char)0;
                }
            }
            return ret;
        }

    };


}
