﻿using System;
using System.Collections.Generic;
using IRCUtils;

namespace monochrome
{
    public struct ServerParams {
        public string hostName;
        public int hostPort;
        public string charEncodingName;
        public string nicks, nickPassword, nickAccount;
        public bool useSASL;
        public string autoJoinChannels;
        public string userModes;
        public bool autoConnect;
        public string serverUser, serverPassword;
        public bool useSSL, verifySSL;
        public bool allowRandomNick;
        public string inviteHandlers;
        public string extraStartupCommands;
        
        public string[] GetExtraStartupCommands() {
            return extraStartupCommands.Split(new char[] { '\n', '\r' }, StringSplitOptions.RemoveEmptyEntries);
        }
        public Dictionary<string, string> GetInviteHandlers() {
            Dictionary<string, string> ret = new Dictionary<string,string>(Nicks.Comparer);
            foreach(string line in inviteHandlers.Split(new char[]{'\n','\r'},StringSplitOptions.RemoveEmptyEntries)) {
                string[] strings = line.Split(new char[]{':'},2);
                try {
                    if (strings.Length == 2) {
                        ret.Add(strings[0],strings[1]);
                    }
                } catch(ArgumentException) {
                }
            }
            return ret;
        }

        public static ServerParams Defaults {
            get {
                ServerParams val = new ServerParams();
                val.hostName = "";
                val.hostPort = 6697; val.useSSL = true; val.verifySSL = true;
                val.nicks = "Guest";
                val.nickPassword = "";
                val.nickAccount = "";
                val.useSASL = true;
                val.autoJoinChannels = "";
                val.charEncodingName = defaultCharEncodingName;
                val.userModes = "";
                val.autoConnect = false;
                val.serverUser = "";
                val.serverPassword = "";
                val.allowRandomNick = false;
                val.inviteHandlers = "";
                val.extraStartupCommands = "";
                return val;
            }
        }

        public static string defaultCharEncodingName {
            get { return "UTF-8"; }
        }
        public static string fallbackCharEncodingName {
            get { return "ISO-8859-15"; }
        }
        
        public System.Text.Encoding textEncoding {
            get {
                return System.Text.Encoding.GetEncoding(charEncodingName);
            }
        }

        public ICollection<ChannelJoinEntry> autoJoinChannelsList {
            get {
                return Commands.SplitChannelJoinList(autoJoinChannels);
            }
        }

        public bool isValid() {
            //todo validate inviteHandlers?
            if (hostName == null || nicks == null || nickPassword == null || autoJoinChannels == null) return false;
            try {
                System.Text.Encoding encoding = textEncoding;
                if (encoding == null) return false;
                if (encoding.GetByteCount("blah") != 4) return false;
            } catch(ArgumentException) {
                return false;
            }
            
            {
                bool gotValid = false;
                foreach(string nick in nickList) {
                    if (!Nicks.IsValid(nick)) return false;
                    gotValid = true;
                }
                if (!gotValid) return false;
            }

            return hostPort > 0 && hostName != "";
        }

        public string[] nickList {
            get { return nicks.Split(new char[]{' ',',',';'},StringSplitOptions.RemoveEmptyEntries); }
        }

        public static bool Equals(ServerParams item1, ServerParams item2) {
            var v1 = new PreferencesIO_BuildMap();
            var v2 = new PreferencesIO_BuildMap();
            item1.Process(v1); item2.Process(v2);
            return PreferencesIO_BuildMap.Equals(v1, v2);
        }

        public override int GetHashCode() {
            return hostName.GetHashCode();
        }
        public override bool Equals(object obj) {
            if (obj is PreferencesData) {
                return Equals(this, (PreferencesData)obj);
            } else {
                return false;
            }
        }
        public static bool operator ==(ServerParams item1, ServerParams item2) {
            return Equals(item1, item2);
        }
        public static bool operator !=(ServerParams item1, ServerParams item2) {
            return !Equals(item1, item2);
        }

        public void Process(IPreferencesIO io) {
            if (io is PreferencesIO_Read) this = Defaults;
            io.Process(ref autoJoinChannels,"autoJoinChannels");
            io.Process(ref hostName,"hostName");
            io.Process(ref hostPort,"hostPort");
            io.Process(ref nicks,"nick");
            io.Process(ref nickPassword,"nickPassword");
            io.Process(ref nickAccount, "nickAccount");
            io.Process(ref charEncodingName,"charEncoding");
            io.Process(ref userModes,"userModes");
            io.Process(ref autoConnect,"autoConnect");
            io.Process(ref serverUser, "serverUser");
            io.Process(ref serverPassword,"serverPassword");
            io.Process(ref useSSL,"useSSL");
            io.Process(ref verifySSL, "verifySSL");
            io.Process(ref allowRandomNick,"allowRandomNick");
            io.Process(ref inviteHandlers,"inviteHandlers");
            io.Process(ref extraStartupCommands, "extraStartupCommands");
            io.Process(ref useSASL, "useSASL");
        }
    };
}
