﻿using System;
using System.Data;
using System.Collections.Generic;
using IRCUtils;
using monochrome.IRCScripts;
using System.Net;
using System.Threading;
using System.Net.Sockets;
using System.Threading.Tasks;

namespace monochrome
{
    public interface IMessageWindow
    {
        void SwitchTo();
        void PrintOutputV2( TextLine line );
        void Clear();
        void Close();
        void SetLogTarget( object logTarget );
        void Flash( int flashLevel );
        void HandleDisconnection();
        void SetUserAway(string nick, bool isAway,string reasonOptional);
        void HandleNickChange( string oldNick, string newNick, bool clash );
        void UserHostHint(string nick, string userHost);
    }

    public interface IChannelWindow : IMessageWindow
    {
        void UserJoined( string user, bool isSelf );
        void UserParted( string user, bool isSelf );
        void UserQuit( string nick );
        bool HasNick( string nick );
        void OnBanList(char type, IEnumerable<BanDescription> items);
        void HandleMode(IRCEvent ev,ServerModeSpecs modeSpecs);
    }

    public interface IServerConnectionDelegate
    {
        void OnActivity();
        IMessageWindow NewQueryWindow( string forUser );
        IChannelWindow NewChannelWindow( string forChannel, ChannelStatus status );
        bool SwitchToWindowIndex( int index );
        void SwitchToWindowDelta( int delta );
        void MoveWindowDelta ( IMessageWindow window, int delta );
        bool MoveWindowTo( IMessageWindow window, int index );
        bool CanExternalUpdatePreferences();
        void ExternalUpdatePreferences( PreferencesData prefs );
        void LogActivity(string context, TextLine line,int level);
        void OpenFileForDCCSend( Action<object, string> callMe, IMessageWindow context );
        void BeginDCCSend( object pathObj, DCCDescription desc );
        void BeginDCCReceive( DCCDescription desc, UInt64 _fileSize,PreferencesData _prefs,bool _autoAcceptTrigger);
        void OnManualAway(bool isAway);
    };


    public class ServerConnection : IDisposable
    {
        private static LinkedList<ServerConnection> g_connections = new LinkedList<ServerConnection>();
        private static string g_awayMessage = "";
        private static bool g_isAway = false;

        private static void SetAwayWithMessage(string message) {
            if (message == "") message = null;
            if (g_awayMessage != message) {
                g_awayMessage = message;
                IsAway = (message != null);
            }
        }

        public static bool IsAway {
            get {
                return g_isAway;
            }
            set {
                if (value != g_isAway) {
                    g_isAway = value;
                    foreach (ServerConnection c in g_connections) {
                        c.applyAway();
                    }
                }
            }
        }


        static bool isValidAwayMsg(string msg) { return msg != null && msg != ""; }

        private string AwayMessage {
            get {
                string msg = g_awayMessage;
                if (!isValidAwayMsg(msg)) {
                    msg = Prefs.defaultAwayMessage();
                    if (!isValidAwayMsg(msg)) {
                        msg = ".";
                    }
                }
                return msg;
            }
        }

        class StartupScriptTag {
            public override string ToString() {return "Startup Script";}
        };
        class EventStreamImpl : EventStream {
            public EventStreamImpl( ServerConnection _owner ) : base(_owner.AbortToken ) {
                m_owner = _owner;
                m_owner.eventStreams.AddLast( this );
            }

            protected override void Disposing() {
                m_owner.eventStreams.Remove(this);
                base.Disposing();
            }
            private ServerConnection m_owner;
        };
        class ScriptContextImpl : IScriptContext {
            public ScriptContextImpl(ServerConnection _owner, object _tag) {
                tag = _tag;
                owner = _owner;
                owner.ScriptNew(this);
            }
            public CancellationToken GetAborter() {
                return owner.AbortToken;
            }
            public void Command(IRCCommand cmd) {
                if (!aborting) {
                    owner.addCommandImmediate(cmd);
                }
            }
            public void CommandFloodable(IRCCommand cmd) {
                if (!aborting) {
                    owner.addCommand(cmd);
                }
            }
            public EventStream MakeEventStream() {
                return new EventStreamImpl( owner );
            }
            public void Done() {
                owner.ScriptDone(this, false);
            }
            public void DoneQuiet() {
                owner.ScriptDone(this, true);
            }
            public void Failure(Exception e) {
                owner.ScriptFailure(this,e);
            }
            public string GetOwnNick() {
                return owner.ownNick;
            }

            public void OnAborted() {
                aborting = true;
            }

            public object Tag {
                get {return tag;}
            }
            public void PrintOutput(string message) {
                if (tag != null) owner.printOutput_threadSafe(tag.ToString() + ": " + message);
            }
            public string ServerHostName() {return owner.ServerHostName; }
            object tag;
            volatile bool aborting;
            ServerConnection owner;
        };

        string queryAwayMessage(bool state) {
            if (state) {
                string msg = Prefs.defaultAwayMessage();
                if (msg == null || msg == "") {
                    return ".";
                }
                return msg;
            } else {
                return null;
            }
        }

        public ServerConnection(IServerConnectionDelegate _delegate, IMessageWindow _serverWindow,ServerParams _params, string netName)
        {
            g_connections.AddLast( this );

            m_delegate = _delegate;
            m_serverWindow = _serverWindow;
            serverParams = _params;
            m_netName = netName;

            m_activeDCCSends = new LinkedList<DCCSend>();
            m_activeDCCReceives = new LinkedList<DCCReceive>();
            
            m_instanceData = new ServerConnectionInstanceData();
            m_droppedChannels = new IRCObjectList();

            m_awayFilter = new AwayMessageFilter();
            m_abortHandler = new CancellationTokenSource();
            
            m_ownNick = PreprocessNick(serverParams.nickList[0]); //pre-sanitize too long nicks to prevent the net from barfing on us on connect while we don't know what nick lengths they do allow
            m_fullName = Prefs.fullName;
            if (m_fullName == null || m_fullName == "") m_fullName = ".";
            IRCWindows = new Dictionary<string,IMessageWindow>(Nicks.Comparer);
            commandQueue = new LinkedList<object>();
            commandQueueReady = new SemaphoreSlim(0);
            scriptsRunning = new LinkedList<ScriptContextImpl>();
            eventStreams = new LinkedList<EventStream>();

            OnPreferencesChanged(Prefs);

            printContextOutput(null,"Client initialized.");

            var host = serverParams.hostName;
            if (!host.EndsWith(".debug")) {
                _ = WorkerProc();
            } else {
                m_debug = true;
                DebugProc(host);
            }
        }

        private ChannelStatus ChannelStatus(string channel) {
            ChannelStatus status = null;
            if (!m_channelStatus.TryGetValue(channel, out status)) {
                status = new ChannelStatus(this, channel);
                m_channelStatus.Add(channel, status);
            }
            return status;
        }

        private IChannelWindow OpenChannel(string channel) {
            IMessageWindow window = null;
            if (!IRCWindows.TryGetValue(channel, out window)) {
                if ( ! (window is IChannelWindow) ) {
                    IRCWindows.Remove( channel );
                } else {
                    return (IChannelWindow) window;
                }
            }
            window = m_delegate.NewChannelWindow(channel, ChannelStatus( channel ));
            IRCWindows.Add( channel, window );
            return (IChannelWindow) window;
        }

        public void PreopenChannels() {
            foreach(ChannelJoinEntry walk in serverParams.autoJoinChannelsList) {
                OpenChannel( walk.Name );
            }
        }
        public LinkedList<string> GetACStrings() {
            return GetACStringsFor("");
        }
        public LinkedList<string> GetACStringsFor(string ChannelOrUserName) {
            LinkedList<string> acData = new LinkedList<string>();
            if (ChannelOrUserName != null && ChannelOrUserName != "") {
                ChannelStatus chan;
                if (m_channelStatus.TryGetValue(ChannelOrUserName, out chan)) {
                    var users = new List<KeyValuePair<string, DateTime>>();

                    foreach (var nick in chan.UserNicks) {
                        if (!Nicks.Equals(nick, ownNick)) {
                            users.Add(new KeyValuePair<string, DateTime>(nick, chan.LastActivityOfNick(nick)));
                        }
                    }

                    users.Sort((x, y) => -x.Value.CompareTo(y.Value));
                    foreach (var user in users) {
                        acData.AddLast(user.Key);
                    }
                }
            }
            acData.AddLast(ownNick);
            LinkedList<string> globalsTemp = new LinkedList<string>();
            this.grabAutoCompleteGlobals(globalsTemp);
            if (ChannelOrUserName != null && ChannelOrUserName != "") {
                acData.AddLast(ChannelOrUserName);
            }
            foreach (string val in globalsTemp) if (val != ChannelOrUserName) acData.AddLast(val);
            return acData;
        }
        public string GetACChanTypes() {return ChanTypes;}
        
        public void OnPreferencesChanged(PreferencesData newData) {
            m_aliases = new AliasList(newData.aliases);
            m_hideMessages = newData.hideMessageTypeList;
        }

        LinkedList<string> GrabACData() {
            LinkedList<string> list = new LinkedList<string>();
            grabAutoCompleteGlobals(list);
            return list;
        }

        public void grabAutoCompleteGlobals(LinkedList<string> output) {
            foreach(string label in IRCWindows.Keys) output.AddLast(label);
        }

        public bool isUserVisible(string nick) {
            foreach( ChannelStatus chan in m_channelStatus.Values ) {
                if ( chan.HaveNick( nick ) ) return true;
            }
            return false;
        }

        void applyAwayEx(bool isStartup) {
            if (InstanceData.LoggedIn) {
                if (!g_isAway) {
                    if (!isStartup) addCommand(new IRCCommand("AWAY"));
                } else {
                    addCommand(new IRCCommand("AWAY", AwayMessage ));
                }
            }
        }
       
        public void applyAway() {applyAwayEx(false);}


        public void WindowClosed(string context,IMessageWindow window) {
            if (window is IChannelWindow) {
                ChannelStatus status;
                if ( m_channelStatus.TryGetValue( context, out status ) ) {
                    if ( status.IsOn ) {
                        runContextCommand(context, "/PART", window);
                    }
                }
            }
            IRCWindows.Remove(context);
        }

        void bounceCommand(IRCCommand command) {
            IRCEvent bounce = IRCEvent.BounceCommand(ownNick,command);
            if (bounce != null) processReceivedEvent(bounce);
        }

        async void floodTimerLoop() {
            var instance = m_instanceData;
            var abort = m_abortHandler.Token;
            while( !abort.IsCancellationRequested ) {
                try {await Task.Delay(100, abort); } catch { return; }
                if ( m_abortHandler.IsCancellationRequested ) return;
                if ( instance != m_instanceData ) return;
                while (instance.FloodQueue.Count > 0) {
                    if (tryAddFloodControlledCommand(instance.FloodQueue.First.Value)) instance.FloodQueue.RemoveFirst();
                    else break;
                }
                if (instance.FloodQueue.Count == 0 ) return;
            }

        }

        void addCommandImmediate(object command) {
            if (command is IRCCommand) {
                bounceCommand((IRCCommand)command);
            }
            commandQueue.AddLast( command );
            commandQueueReady.Release();
        }

        bool tryAddFloodControlledCommand(IRCCommand command) {
            if (InstanceData.FloodControl.RequestSend()) {
                addCommandImmediate(command);
                return true;
            }
            return false;
        }

        void addFloodControlledCommand_mainThreadOnly(IRCCommand command) {
            bool wasEmpty = InstanceData.FloodQueue.Count == 0;
            if (wasEmpty) {
                if (tryAddFloodControlledCommand(command)) return;
            }
            InstanceData.FloodQueue.AddLast(command);
            if ( wasEmpty ) {
                floodTimerLoop();
            }
        }


        public void addReconnectRequest(TimeSpan delay) {
            addCommand(new ReconnectRequest(delay));
        }

        public void addCommand(object command) {
            if (command is IRCCommand) {
                IRCCommand _command = (IRCCommand)command;
                if (_command.Floodable && !m_debug) {
                    InMainThread( () => {
                        addFloodControlledCommand_mainThreadOnly( _command );
                    } );
                    return;
                }
            }
            addCommandImmediate(command);
        }

        public string MergeModes(IEnumerable<string> modes) {
            string actions = "";
            string args = "";
            bool currentSign = false, currentSignSet = false;
            foreach(string mode in modes) {
                string[] split = mode.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries);
                if (split.Length != 1 && split.Length != 2) throw new ArgumentException();
                string action = split[0];
                if (action.Length != 2) throw new ArgumentException();
                bool sign = false;
                switch(action[0]) {
                    case '+':
                        sign = true;
                        break;
                    case '-':
                        sign = false;
                        break;
                    default:
                        throw new ArgumentException();
                }
                if (!currentSignSet || sign != currentSign) {
                    currentSign = sign; currentSignSet = true;
                    actions += (sign ? '+' : '-');
                }
                actions += action[1];
                if (split.Length >= 2) args += " " + split[1];
            }
            return actions + args;
        }

        public void addModeCommands(string context, IEnumerable<string> modes) {
            IMessageWindow wnd;
            if ( printContextOutput_ResolveWindow( context, out wnd ) ) {
                addModeCommands( context, modes, wnd );
            }
        }
        public void addModeCommands(string context, IEnumerable<string> modes, IMessageWindow contextWindow ) {
            try {
                LinkedList<string> temp = new LinkedList<string>();
                foreach(string mode in modes) {
                    temp.AddLast(mode);
                    if (temp.Count >= InstanceData.Modes) {
                        runCommand("/MODE " + context + " " + MergeModes(temp),contextWindow);
                        temp.Clear();
                    }
                }
                if (temp.Count > 0) runCommand("/MODE " + context + " " + MergeModes(temp),contextWindow);
            } catch(Exception e) {
                printOutputHere( contextWindow, e.Message );
            }
        }

        private void printOutputHere( IMessageWindow context, string message ) {
            printOutputHere( context, TextLine.Simple( message ) ) ;
        }
        private void printOutputHere( IMessageWindow context, TextLine message, bool isHighlight = false ) {
            string timestamp;
            try {
                timestamp = DateTime.Now.ToString(PreferencesManager.Current.timestampFormat);
            } catch {
                timestamp = "[timestamp error]";
            }

            var line = TextLine.Simple(timestamp, TextStyle.Gray) + " " + message;
            PreferencesData prefs = PreferencesManager.Current;
            if (!prefs.displayColors) line = line.StripCustomColors();
            if (!prefs.displayFormatting) line = line.StripCustomFormatting();
            if (isHighlight) line = line.AddHighlight();
            context.PrintOutputV2( line );
        }


        public string GetPingParam() {
            return "O:" + DateTime.UtcNow.Ticks.ToString();
        }

        public void runPING( string target )
        {
            runCTCP( target, "PING" ,GetPingParam() );
        }

        public void runCTCP(string target,string command,string data) {
            string commandEx = command;
            if (data.Length > 0) commandEx += " " + data;
            addCommand(new IRCCommand("PRIVMSG",new string[]{target,IRCUtils.CTCP.Enclose(commandEx)}));
        }

        int outgoingStringBytes(string s) {
            return serverParams.textEncoding.GetByteCount(s);
        }
        void sendContextInputLine(string context, string input) {
            addCommand(IRCUtils.Commands.Message(context,input));
        }
        int findSplitMarker(string line, int limit) {
            int outLen;
            if (!TextUtils.FindTruncationPoint(serverParams.textEncoding,line,limit,out outLen)) return -1;
            return Math.Max(outLen,1);
        }
        static int adjustWordSplitMark(string line,int split) {
            for(int walk = split - 1;;) {
                if (walk <= 0) return split;
                if (char.IsWhiteSpace(line[walk])) return walk + 1;
                --walk;
            }
        }
        IEnumerable<string> handleLineSplit(string line,int limit) {
            LinkedList<string> list = new LinkedList<string>();
            if (limit > 0 && outgoingStringBytes(line) > limit) {
                while(line.Length > 0) {
                    int split = findSplitMarker(line,limit);
                    if (split <= 0) {list.AddLast(line);break;}
                    split = adjustWordSplitMark(line,split);
                    list.AddLast(line.Substring(0,split));
                    line = line.Substring(split);
                }
            } else {
                list.AddLast(line);
            }
            return list;
        }
        void sendContextInput(string context,string input) {
            m_delegate.OnActivity();
            
            string[] lines = input.Split(new char[]{'\r','\n'},StringSplitOptions.RemoveEmptyEntries);
            int lineLimit = Prefs.splitLines;
            foreach(string line in lines) {
                foreach(string lineEx in handleLineSplit(line,lineLimit)) {
                    sendContextInputLine(context,lineEx);
                }
            }
        }
        bool TryAlias(string context, string command, IMessageWindow contextWindow) {
            if (m_aliases != null) {
                object obj = m_aliases.EvalAlias(command, context);
                if (obj != null) {
                    if (obj is string) {
                        Debug.WriteLine("Alias evaluated to: " + (string) obj);
                        runCommand((string)obj, contextWindow);
                        return true;
                    } else if (obj is IRCCommand) {
                        addCommand((IRCCommand)obj);
                        return true;
                    }
                }
            }
            return false;
        }

        public void runContextCommandAliasable(string context,string command,IMessageWindow contextWindow) {
            if (!TryAlias(context, command, contextWindow)) runContextCommand(context, command, contextWindow);
        }
        public void runContextCommand(string context,string command,IMessageWindow contextWindow) {
            try {
                
                if (command.Length > 0) {
                    if (command[0] == '/') {
                        int walk = 0;
                        walk += Misc.ScanWord(command,walk);
                        string firstword = command.Substring(0,walk).ToUpperInvariant();
                        walk += Misc.ScanSpace(command,walk);
                        string remaining = command.Substring(walk);
                        if (firstword == "/ME") {
                            runCTCP(context, "ACTION", remaining);
                        } else if (firstword == "/REMOVE" || firstword == "/TOPIC" || firstword == "/KICK" || firstword == "/PART" || firstword == "/MODE" || firstword == "/OP" || firstword == "/DEOP" || firstword == "/VOICE" || firstword == "/DEVOICE") {
                            if (remaining.Length > 0 && IsKnownChanType(remaining[0])) {
                                runCommand(command, contextWindow);
                            } else {
                                runCommand(firstword + " " + context + " " + remaining, contextWindow);
                            }
                        } else if (firstword == "/RKICK") {
                            foreach (string victim in remaining.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries)) {
                                runCommand("/KICK " + context + " " + victim + " ROUNDHOUSE KICK", contextWindow);
                            }
                        } else if (firstword == "/BAN") {
                            runContextCommand(context, "/MODE +b " + remaining, contextWindow);
                        } else if (firstword == "/UNBAN") {
                            runContextCommand(context, "/MODE -b " + remaining, contextWindow);
                        } else if (firstword == "/") {
                            if (remaining.Length > 0) addCommand(IRCUtils.Commands.Message(context, remaining));
                        } else if (firstword == "/LGBT") {
                            sendContextInput(context, TextUtils.Rainbow(remaining));
                        } else {
                            runCommand(command,contextWindow);
                        }
                    } else {
                        sendContextInput(context,command);
                    }
                }
            } catch(Exception e) {
                printOutputHere( contextWindow, e.Message );
            }
        }

        public bool contextWindowPresent(string context) {
            return IRCWindows.ContainsKey(context);
        }

        public void printContextOutput(string context,string message) {
            printContextOutput(context,TextLine.Simple(message));
        }

        public void openQuery(string user,bool activate) {
            if (IsClosing) return;
            IMessageWindow window;
            if (!IRCWindows.TryGetValue(user,out window)) {
                window = m_delegate.NewQueryWindow( user );
                IRCWindows.Add(user,window);
            }
            if (activate) {
                window.SwitchTo();
            }
        }

        
        void runStandardCommand(string command,string parameters,int maxArgs) {
            if (command.Length > 0) {
                if (command[0] == '/') command = command.Substring(1);
                string[] data = parameters.Split(new char[]{' '},maxArgs,StringSplitOptions.RemoveEmptyEntries);
                addCommand(new IRCCommand(command,data));
            }
        }
        void runStandardCommand(string command,string parameters) {
            runStandardCommand(command,parameters,int.MaxValue);
        }
        void runStandardCommandImmediate(string command,string parameters,int maxArgs) {
            if (command.Length > 0) {
                if (command[0] == '/') command = command.Substring(1);
                string[] data = parameters.Split(new char[]{' '},maxArgs,StringSplitOptions.RemoveEmptyEntries);
                addCommandImmediate(new IRCCommand(command,data));
            }
        }
        void runStandardCommandImmediate(string command,string parameters) {
            runStandardCommand(command,parameters,int.MaxValue);
        }

        static string preprocessDCCFilename(string fn) {
            return fn.Replace('\"','_');
        }

        public string DCCIPString {
            get {
                return m_instanceData.OwnIPForDCC;
            }
        }


        public void runSendFile(string targetNicks,IMessageWindow contextWindow) {

            string[] targets = targetNicks.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries);
            if (targets.Length == 0) return;
            var dccIP = DCCIPString;
            if (dccIP == null) {
                printOutputHere( contextWindow, "Own host name unknown, can't send file at this time." );
            } else {
                Action<object, string> handler = (object pathObj, string fileName) => {
                    try {
                        DCCMode mode = DCCMode.Send;
                        /*
                        if (fileSize > 1024 * 1024 * 64) { }
                             mode = DCCMode.TSend;
                        }
                        */

                        //remove forbidden chars etc
                        string fnPreprocessed = preprocessDCCFilename(fileName);
                            
                        foreach(string target in targets) {
                            DCCDescription desc = new DCCDescription(target,fnPreprocessed,0,mode);
                            m_delegate.BeginDCCSend( pathObj, desc );
                        }
                    } catch(Exception e) {
                        printOutputHere(contextWindow, "Could not initiate DCC SEND: " + e.Message);
                    }                        
                };
                m_delegate.OpenFileForDCCSend( handler, contextWindow );
            }
        }

        static bool IsNumber(string str) {
            bool gotDigits = false;
            foreach(char c in str) {
                if (!char.IsDigit(c)) return false;
                gotDigits = true;
            }
            return gotDigits;
        }

        class UnknownWINCommand : InvalidIRCCommand {
            public UnknownWINCommand() : base("Unknown /WIN command") {}
        };

        void runWinCommand(string command, IMessageWindow contextWindow) {
            string[] commands = command.ToUpper().Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries);
            if (commands.Length == 1) {
                if (commands.Length == 1 && IsNumber(commands[0])) {
                    if (! m_delegate.SwitchToWindowIndex(Convert.ToInt32(commands[0]) - 1)) {
                        printOutputHere(contextWindow, "No such window.");
                    }
                } else if (commands[0] == "NEXT") {
                    m_delegate.SwitchToWindowDelta( 1 );
                } else if (commands[0] == "PREV") {
                    m_delegate.SwitchToWindowDelta( -1 );
                } else if (commands[0] == "CLOSE") {
                    contextWindow.Close();
                } else {
                    throw new UnknownWINCommand();
                }
            } else if (commands.Length == 2) {
                if (commands[0] == "MOVE") {
                    string where = commands[1].ToUpperInvariant();
                    if (where == "UP" || where == "LEFT") {
                        m_delegate.MoveWindowDelta( contextWindow, -1 );
                    } else if (where == "DOWN" || where == "RIGHT") {
                        m_delegate.MoveWindowDelta( contextWindow, 1 );
                    } else if (!m_delegate.MoveWindowTo(contextWindow,Convert.ToInt32(where) - 1)) {
                        printOutputHere(contextWindow, "Specified index out of range.");
                    }
                } else {
                    throw new UnknownWINCommand();
                }
            } else {
                throw new UnknownWINCommand();
            }
        }

        void PrintOutput( string msg )
        {
            printOutputHere( m_serverWindow, msg );
        }

        /*
        private void runSelfTest() {
            try {
                this.PrintOutput("Running self-test...");
                foreach(KeyValuePair<string,SubscribedChannel> chan in InstanceData.SubscribedChannels) {
                    IChannelWindow wnd;
                    if (FindChannel(chan.Key,out wnd)) {
                        IRCObjectList wndNames = new IRCObjectList(wnd.NameList);
                        IRCObjectList subNames = chan.Value.Users;
                        foreach(string user in wndNames) {
                            if (!subNames.Contains(user)) throw new Exception("User \"" + user + "\" missing from " + chan.Key);
                        }
                        foreach(string user in subNames) {
                            if (!wndNames.Contains(user)) throw new Exception("User \"" + user + "\" mis-registered on " + chan.Key);
                        }
                        if (wndNames.Count != subNames.Count) {
                            throw new Exception("User count mismatch on " + chan.Key);
                        }
                    } else {
                        PrintOutput("Self-test: no channel window present for: " + chan.Key);
                    }
                }
                this.PrintOutput("Self-test OK");
            } catch(Exception e) {
                this.PrintOutput("Self-test failure: " + e.Message);
            }
        }
        */

        static string grabBanMask(string nick) {
            return Nicks.Validate(nick) + "!*@*";
        }
        public void beginShowList( string channel, char mode ) {
            addCommand(new IRCCommand("MODE", new string[] { channel, "+" + mode.ToString() }));
        }
        public void runChannelCommand( string command, string channelName, IChannelWindow context) {
            try {
                string firstword, remaining;
                int walk = 0;
                walk += Misc.ScanWord(command, walk);
                firstword = command.Substring(0, walk).ToUpperInvariant();
                walk += Misc.ScanSpace(command, walk);
                remaining = command.Substring(walk);
                if (firstword == "/KICKBAN") {
                    string[] data = remaining.Split(new char[] { ' ' }, 2, StringSplitOptions.RemoveEmptyEntries);
                    if (data.Length > 0) {
                        string victim = data[0];
                        string kickmsg = "/KICK " + victim;
                        if (data.Length > 1) kickmsg += " " + data[1];
                        runContextCommand(channelName, kickmsg, context);
                        runContextCommand(channelName, "/MODE +b " + grabBanMask(victim), context);
                    }
                } else if (firstword == "/BANLIST") {
                    beginShowList( channelName, 'b' );
                } else if (firstword == "/EXEMPTIONLIST") {
                    beginShowList( channelName, 'e' );
                } else {
                    runContextCommandAliasable(channelName, command, context);
                }
            } catch (Exception e) {
                printContextOutput(channelName, e.Message);
            }

        }
        public void runCommandAliasable(string command,IMessageWindow contextWindow) {
            if (!TryAlias(null, command, contextWindow)) runCommand(command, contextWindow);
        }

        public void EchoAllServers( string what ) {
            var list = new LinkedList<ServerConnection>(g_connections);
            foreach (ServerConnection c in list) {
                c.PrintServerOutput(what);
            }
        }
        public void runCommand(string command,IMessageWindow contextWindow) {
            try {
                if (command.Length > 0) {
                    int walk = 0;
                    walk += Misc.ScanWord(command,walk);
                    string firstword = command.Substring(0,walk).ToUpperInvariant();
                    walk += Misc.ScanSpace(command,walk);
                    string remaining = command.Substring(walk);
                    /*if (firstword == "/SELFTEST") {
                        runSelfTest();
                    } else */
                    if (firstword == "/QUIT") {
                        RequestClose(remaining);
                    } else if (firstword == "/ECHOALL" ) {
                        EchoAllServers(remaining);
                    } else if (firstword == "/QUITALL") {
                        var list = new LinkedList<ServerConnection>( g_connections );
                        foreach (ServerConnection c in list) {
                            try {
                                c.RequestClose(remaining);
                            } catch(Exception e) {
                                Debug.WriteLine("/QUITALL RequestClose failure: " + e.Message);
                                Debug.WriteLine("On: " + c.NetName);
                            }
                        }
                    } else if (firstword == "/JOIN" || firstword == "/J") {
                        if (TestStartupInterrupt(contextWindow)) {
                            Dictionary<string,string> inviteHandlers = serverParams.GetInviteHandlers();
                            foreach(ChannelJoinEntry entry in Commands.SplitChannelJoinList(remaining)) {
                                string inviteHandler;
                                inviteHandlers.TryGetValue(entry.Name, out inviteHandler);
                                new IRCScripts.ChannelJoinScript(new ScriptContextImpl(this, null), entry.Name,entry.Key, inviteHandler);
                            }
                        }
                    } else if (firstword == "/PART") {
                        if (TestStartupInterrupt(contextWindow)) {
                            //PART channel reason
                            string[] data = remaining.Split(new char[]{' '},2,StringSplitOptions.RemoveEmptyEntries);
                            if (data.Length < 1) throw new InvalidIRCCommand();
                            if (data.Length == 1) {
                                string defmsg = Prefs.defaultPartMessage();
                                if (defmsg != null) {
                                    data = new string[]{data[0],defmsg};
                                }
                            }
                            addCommand(new IRCCommand("PART",data));
                        }
                    } else if (firstword == "/NICK") {
                        if (TestStartupInterrupt(contextWindow)) runStandardCommand(firstword,remaining,1);
                    } else if (firstword == "/KICK") {
                        //KICK channel user reason
                        string[] data = remaining.Split(new char[]{' '},3,StringSplitOptions.RemoveEmptyEntries);
                        if (data.Length < 2) throw new InvalidIRCCommand();
                        if (data.Length == 2) {
                            string defmsg = Prefs.defaultKickMessage();
                            if (defmsg != null) {
                                data = new string[]{data[0],data[1],defmsg};
                            }
                        }
                        addCommand(new IRCCommand("KICK",data));
                    } else if (firstword == "/REMOVE") {
                        runStandardCommand(firstword,remaining,3);
                    } else if (firstword == "/QUERY") {
                        string[] users = remaining.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries);
                        foreach(string user in users) {
                            openQuery(user,true);
                        }
                    } else if (firstword == "/NOTICE" || firstword == "/^NOTICE") {
                        object ircCommand = new IRCCommand("NOTICE",remaining.Split(new char[]{' '},2,StringSplitOptions.RemoveEmptyEntries));
                        if (firstword[1] == '^') {
                            addCommandImmediate(ircCommand);
                        } else {
                            addCommand(ircCommand);
                        }
                    } else if (firstword == "/MSG" || firstword == "/^MSG" || firstword == "/PRIVMSG" || firstword == "/^PRIVMSG") {
                        string[] pah = remaining.Split(new char[]{' '},2,StringSplitOptions.RemoveEmptyEntries);
                        if (pah.Length != 2) printOutputHere( contextWindow, "Usage: " + firstword + " <nickname> <message>");
                        else {
                            object ircCommand = new IRCCommand("PRIVMSG",pah);
                            if (firstword[1] == '^') {
                                addCommandImmediate(ircCommand);
                            } else {
                                addCommand(ircCommand);
                            }
                        }
                    } else if (firstword == "/NICKSERV" || firstword == "/CHANSERV" || firstword == "/MEMOSERV" || firstword == "/SEENSERV") {
                        if (TestStartupInterrupt(contextWindow)) runStandardCommand(firstword,remaining);
                    } else if (firstword == "/GHOST") {
                        if (TestStartupInterrupt(contextWindow)) {
                            string[] args = remaining.Split(new char[]{' '},2,StringSplitOptions.RemoveEmptyEntries);
                            if (args.Length > 0) {
                                string victim = args[0];
                                string password;
                                if (args.Length > 1) password = args[1];
                                else {
                                    password = serverParams.nickPassword;
                                }
                                if (password == "") printOutputHere( contextWindow, "No password specified.");
                                else {
                                    new IRCScripts.GhostScript(new ScriptContextImpl(this,"Ghost Script"),victim, password);
                                }
                            } else {
                                printOutputHere(contextWindow, "Usage: /GHOST <nick> [<password>]");
                            }
                        }
                    } else if (firstword == "/IDENTIFY") {
                        if (TestStartupInterrupt(contextWindow)) {
                            string[] args = remaining.Split(new char[]{' '},1,StringSplitOptions.RemoveEmptyEntries);
                            string password;
                            string account = "";
                            if (args.Length > 0) {
                                password = args[0];
                            } else {
                                password = serverParams.nickPassword;
                                account = serverParams.nickAccount;
                            }
                            if (password == "") {
                                printOutputHere(contextWindow, "No password specified.");
                            } else {
                                string[] arg;
                                if (account.Length > 0) {
                                    arg = new string[] {"IDENTIFY", password};
                                } else {
                                    arg = new string[] {"IDENTIFY", account, password};
                                }
                                addCommand(new IRCCommand("NICKSERV",arg));
                            }
                        }
                    } else if (firstword == "/TOPIC") {
                        runStandardCommand(firstword,remaining,2);
                    } else if (firstword == "/MODE") {
                        runStandardCommand(firstword,remaining);
                    } else if (firstword == "/UMODE") {
                        runCommand("/MODE " + ownNick + " " + remaining,contextWindow);
                    } else if (firstword == "/INVITE") {
                        runStandardCommand(firstword,remaining,2);
                    } else if (firstword == "/^MODE") {
                        runStandardCommandImmediate(firstword,remaining);
                    } else if (firstword == "/RECONNECT") {
                        addCommand(new ReconnectRequest());
                    } else if (firstword == "/RAW") {
                        if (TestStartupInterrupt(contextWindow)) addCommand(remaining);
                    } else if (firstword == "/HELLO") {
                        string target = null;
                        try {
                            target = Commands.ValidateTarget(remaining);
                        } catch {
                            printOutputHere(contextWindow, "/hello: Sends an ascii shock image via notice to user or channel");
                            printOutputHere(contextWindow, "Example: /hello rasengan");
                        }
                        if ( target != null ) IRCScripts.HelloScript.Run(new ScriptContextImpl(this,null),target);
                    } else if (firstword == "/PING") {
                        addCommand(Commands.CTCP(remaining,"PING",GetPingParam()));
                    } else if (firstword == "/AWAY") {
                        SetAwayWithMessage( remaining );
                        m_delegate.OnManualAway( IsAway );
                    } else if (firstword == "/LIST") {
                        addCommand(new IRCCommand("LIST"));
                    } else if (firstword == "/WHO") {
                        string what = Commands.ValidateTarget(remaining);
                        m_instanceData.ExpectedWhoResponses.Add(what);
                        runStandardCommand(firstword,remaining);
                    } else if (firstword == "/WHOIS" || firstword == "/WHOWAS") {
                        string[] args = remaining.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries);
                        string what = firstword.Substring(1);
                        switch(args.Length) {
                            case 1:
                                addCommand(new IRCCommand(what,new string[]{args[0],args[0]}));
                                break;
                            case 2:
                                addCommand(new IRCCommand(what,new string[]{args[0],args[1]}));
                                break;
                            default:
                                printOutputHere(contextWindow, "Usage: " + firstword + " [server] nick");
                                break;
                        }
                    } else if (firstword == "/CTCP" || firstword == "/CTCPREPLY") {
                        string[] data = remaining.Split(new char[]{' '},3,StringSplitOptions.RemoveEmptyEntries);
                        if (data.Length < 2) {
                            printOutputHere(contextWindow, "Usage: /CTCP nick command [parameters]");
                            throw new InvalidIRCCommand();
                        }
                        string ctcpCommand = data[1].ToUpperInvariant();
                        bool isReply = firstword == "/CTCPREPLY";
                        if (data.Length < 3) {
                            if (isReply) {
                                addCommand(Commands.CTCPReply(data[0],ctcpCommand));
                            } else {
                                addCommand(Commands.CTCP(data[0],ctcpCommand));
                            }
                        } else {
                            if (isReply) {
                                addCommand(Commands.CTCPReply(data[0],ctcpCommand,data[2]));
                            } else {
                                addCommand(Commands.CTCP(data[0],ctcpCommand,data[2]));
                            }
                        }
                    } else if (firstword == "/SENDFILE") {
                        runSendFile(remaining,contextWindow);
                    } else if (firstword == "/WIN") {
                        runWinCommand(remaining,contextWindow);
                    } else if (firstword == "/ECHO") {
                        printOutputHere(contextWindow, remaining );
                    } else if (firstword == "/CLEAR") {
                        contextWindow.Clear();
                    } else if (firstword == "/OPER") {
                        runStandardCommand(firstword,remaining,2);
                    } else if (firstword == "/STOP") {
                        if (InstanceData.FloodQueue.Count > 0) {
                            InstanceData.FloodQueue.Clear();
                            printOutputHere(contextWindow, "Flood control: queue purged.");
                        } else {
                            printOutputHere(contextWindow, "Flood control: no queued messages.");
                        }
                    } else if (firstword == "/OP" || firstword == "/DEOP" || firstword == "/VOICE" || firstword == "/DEVOICE") {
                        string[] data = remaining.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries);
                        if (data.Length < 2) {
                            printOutputHere(contextWindow, "Usage: " + firstword + " [#channel] user [user ..]");
                        } else {
                            char action; bool sign;
                            if (firstword == "/OP") {action = 'o'; sign = true;}
                            else if (firstword == "/DEOP") {action = 'o'; sign = false;}
                            else if (firstword == "/VOICE") {action = 'v'; sign = true;}
                            else if (firstword == "/DEVOICE") {action = 'v'; sign = false;}
                            else throw new ArgumentException();
                            
                            string chan = data[0];
                            string actionString = (sign ? "+" : "-") + action.ToString();
                            LinkedList<string> modes = new LinkedList<string>();
                            for(int userWalk = 1; userWalk < data.Length; ++userWalk) {
                                modes.AddLast( actionString + " " + data[userWalk] );
                            }
                            addModeCommands(chan,modes,contextWindow);
                        }
                    } else if (firstword == "/DNS") {
                        foreach(string entry in remaining.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries)) {
                            _ = HandleDNS( entry, contextWindow );
                        }
                    } else if (firstword == "/IGNORE") {
                        UpdateIgnore(remaining, false, true, contextWindow);
                    } else if (firstword == "/HALFIGNORE") {
                        UpdateIgnore(remaining, true, true, contextWindow);
                    } else if (firstword == "/UNIGNORE") {
                        UpdateIgnore(remaining, false, false, contextWindow);
                    } else if (firstword == "/UNHALFIGNORE") {
                        UpdateIgnore(remaining, true, false, contextWindow);
                    }
                    /*
                    else if (firstword == "/BSFILTER") {
                        var raw = PreferencesManager.Current.bsFilter;
                        printOutputHere(contextWindow, "bsFilter raw: " + raw);
                        var lines = PreferencesManager.Current.bsFilterTokens;
                        printOutputHere( contextWindow, "bsFilter: " + lines.Length + " tokens");
                        foreach( string line in lines ) {
                            printOutputHere( contextWindow, line );
                        }
                    }
                    */
                    /*else if (firstword == "/FLOODSELF") {
                        for(int asl = 0; asl < 1000; ++asl) {
                            contextWindow.PrintOutput(Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx());
                        }
                    } */ /*else if (firstword == "/FAKEEVENT") {
                        processReceivedEvent(IRCEvent.FromInput(remaining));
                    } */else {
                        throw new Exception("Unknown Command: " + command);
                    }
                }
            } catch(Exception e) {
                printOutputHere( contextWindow, e.Message );
            }
        }

        static bool _updateIgnoreHaveItem(string[] input, string param) {
            foreach(string v in input) {
                if (v == param) return true;
            }
            return false;
        }

        void UpdateIgnore(string param, bool half, bool add, IMessageWindow contextWindow) {
            if (! m_delegate.CanExternalUpdatePreferences() ) {
                printOutputHere( contextWindow, "Cannot update ignore lists while the Preferences window is open.");
            } else {
                //this code sucks and needs refactoring to sanely deal with large lists
                PreferencesData prefs = Prefs;
               
                string[] input = param.Split(new char[]{' '}, StringSplitOptions.RemoveEmptyEntries);
                string[] config = half ? prefs.getHalfIgnoreList() : prefs.getIgnoreList();

                string joint = "";
                bool changed = false;
                string sep = "\r\n";
                string word = half ? "halfignored" : "ignored";
                if (add) {
                    foreach(string entry in config) {
                        joint += entry + sep;
                    }
                    foreach(string entry in input) {
                        if (!_updateIgnoreHaveItem(config, entry)) {
                            joint += entry + sep;
                            changed = true;
                            printOutputHere( contextWindow, entry + " " + word + ".");
                        } else {
                            printOutputHere( contextWindow, entry + " already " + word + "!");
                        }
                    }
                } else {
                    foreach(string entry in config) {
                        if (!_updateIgnoreHaveItem(input, entry)) {
                            joint += entry + sep;
                        } else {
                            changed = true;
                            printOutputHere( contextWindow, entry + " un" + word + ".");
                        }
                    }
                }

                if (changed) {
                    if (half) prefs.halfIgnore = joint;
                    else prefs.ignore = joint;

                    m_delegate.ExternalUpdatePreferences ( prefs );
                }
            }
        }

        class DNSResolveState {
            public DNSResolveState(IMessageWindow context, string host) {Context = context; Host = host;}
            public IMessageWindow Context;
            public string Host;
        };

        async Task HandleDNS( string inHostName, IMessageWindow contextWindow )
        {
            printOutputHere( contextWindow, "Resolving: " + inHostName );

            try {
                string got = await NetStream.ResolveHostName( inHostName );
                if ( IsClosing ) return;
                printOutputHere( contextWindow, "Resolved " + inHostName + " to: " + got);
            } catch(Exception e) {
                printOutputHere( contextWindow, "Could not resolve " + inHostName + " : " + e.Message);
            }
        }

        async Task<object> pollCommand(TimeSpan timeout) {
            try {
                if (!await commandQueueReady.WaitAsync(timeout, AbortToken)) return null;
            } catch(OperationCanceledException) { return null; }
            object ret = null;
            if (commandQueue.Count > 0) {
                ret = commandQueue.First.Value;
                commandQueue.RemoveFirst();
            }
            return ret;
        }

        void printOutput_threadSafe(string message) {
            PrintOutput( message );
        }

        class ReconnectException : Exception {
            public ReconnectException(TimeSpan requestedDelay) : base("Reconnect Request") { m_requestedDelay = requestedDelay;}

            public TimeSpan RequestedDelay { get { return m_requestedDelay;} }

            TimeSpan m_requestedDelay;
        };
        class ReconnectRequest {
            public ReconnectRequest(TimeSpan delay) {
                m_delay = delay;
            }
            public ReconnectRequest() : this(TimeSpan.FromSeconds(5)) {}

            public TimeSpan Delay { get { return m_delay; } }

            TimeSpan m_delay;

            public static TimeSpan ErrorDelay { get {return TimeSpan.FromHours(1);} }
        };


        static IRCCommand parseCommand(string cmd) {
            string[] data = cmd.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries);
            if (data.Length == 0) throw new InvalidIRCCommand();
            IRCCommand ret = new IRCCommand(data[0]);
            ret.parameters = new string[data.Length-1];
            for(int walk = 0; walk < ret.parameters.Length; ++walk) ret.parameters[walk] = data[walk+1];
            return ret;
        }

        void userHostHintDispatch(string nick, string userHost) {
            foreach(IMessageWindow wnd in IRCWindows.Values) {
                wnd.UserHostHint(nick,userHost);
            }
        }

        void userStringHint(string userString) {
            if (userString.IndexOf('!') >= 0 && userString.IndexOf('@') >= 0) {
                userHostHintDispatch(Nicks.Extract(userString), Nicks.ExtractUserHost(userString));
            }
        }

        void handleEventNotice(string source,string target,string content) {
            string sourceNick = Nicks.Extract(source);
            string targetNick = /*Nicks.Extract*/(target);
            string context = "";
            bool outgoing = false;
            if (sourceNick == "") {
                //server notice
                context = "";
            } else if (Nicks.Equals(targetNick,ownNick)) {
                //receiving user-to-user
                context = sourceNick;
            } else if (Nicks.Equals(sourceNick,ownNick)) {
                //outgoing
                context = targetNick;
                outgoing = true;
            } else {
                //channel notice
                context = targetNick;
                onChannelActivity(context,sourceNick);
            }
            
            TextLine output;
            if (outgoing) {
                output = ">" + TextLine.UserName(targetNick,false) + "< " + TextLine.FromIRC(content);
            } else {
                output = TextLine.FromIRC(content);
                if (sourceNick.Length > 0) output = "-" + TextLine.UserName(sourceNick,false) + "- " + output;
            }
            
            if (!contextWindowPresent(context)) context = null;

            printContextOutput(context,output,/*outgoing ? 1 : 2*/1);
        }

        static bool shouldProcessBouncedCTCP(string what) {
            return what == "ACTION";
        }

        bool testDCCAAChan(string mask, string nick) {
            string chan = null, net = null;
            {
                string[] split = mask.Split('@');
                switch(split.Length) {
                    case 1:
                        chan = split[0];
                        break;
                    case 2:
                        chan = split[0];
                        net = split[1];
                        break;
                }
            }
            if (chan == null) return false;
            
            if (net != null) {
                if (!Misc.WildCardTest(net,m_netName)) return false;
            }

            foreach (ChannelStatus chanWalk in m_channelStatus.Values) {
                if (Misc.WildCardTest(chan, chanWalk.ChannelName) && chanWalk.HaveNick(nick)) {
                    return true;
                }
            }
            return false;
        }
        
        bool testDCCAA(string source) {
            /*if (Prefs.xferAAAvailable)*/ {
                foreach(string mask in Prefs.getXferAAUsers()) {
                    if (Nicks.MaskTest(mask,source)) return true;
                }
                string nick = Nicks.Extract(source);
                foreach(string chanmask in Prefs.getXferAAChannels()) {
                    if (testDCCAAChan(chanmask,nick)) return true;
                }
            }
            return false;
        }

        static string preprocessIncomingDCCName(string name) {
            string ret = "";
            char[] invalid = System.IO.Path.GetInvalidFileNameChars();
            foreach(char c in name) {
                bool isInvalid = false;
                foreach(char i in invalid) {
                    if (i == c) {
                        isInvalid = true;
                        break;
                    }
                }
                ret += isInvalid ? '_' : c;
            }
            return ret;
        }

        bool DCCTryInitiateResume(DCCDescription desc) {
            foreach(DCCReceive recv in m_activeDCCReceives) {
                if (DCCDescription.Match(desc,recv.Description)) {
                    if (recv.TryInitiateResume(desc)) {
                        return true;
                    }
                }
            }
            return false;
        }

        void handleDCC(string source, string command) {
            string[] data = Misc.SplitWordWithQuotes(command);
            //type argument address port size
            //DCC SEND hello.txt 1125078033 46657 1254
            if (data.Length < 1) throw new InvalidIRCEvent();
            string requestType = data[0];
            string sourceNick = Nicks.Extract(source);
            if (requestType == "SEND" || requestType == "TSEND") {
                if (data.Length < 4) throw new InvalidIRCEvent();
                string fileName = preprocessIncomingDCCName(data[1]);
                string host = CTCP.ParseDCCHost(data[2]);
                UInt16 port = Convert.ToUInt16(data[3]);
                UInt64 size = data.Length >= 5 ? Convert.ToUInt64(data[4]) : UInt64.MaxValue;
                DCCMode mode = requestType == "TSEND" ? DCCMode.TSend : DCCMode.Send;
                DCCDescription desc = new DCCDescription(sourceNick,fileName,port,mode,host);
                if (!DCCTryInitiateResume(desc)) {
                    m_delegate.BeginDCCReceive(desc,size,Prefs,testDCCAA(source));
                }
            } else if (requestType == "RESUME") {
                //resume request
                //DCC RESUME <argument> <port> <position>
                //find a matching send, tell them that the target is trying to resume, fire an ACCEPT back on success
                string fileName = data[1];
                UInt16 port = Convert.ToUInt16(data[2]);
                UInt64 resumeOffset = Convert.ToUInt64(data[3]);
                string nick = Nicks.Extract(source);
                DCCResumeRequested(nick,fileName,port,resumeOffset);
            } else if (requestType == "ACCEPT") {
                //accepting a resume
                //DCC ACCEPT <argument> <port> <position>
                //find a matching receive, tell them that resume request got accepted
                string fileName = data[1];
                UInt16 port = Convert.ToUInt16(data[2]);
                UInt64 resumeOffset = Convert.ToUInt64(data[3]);
                string nick = Nicks.Extract(source);
                DCCResumeAccepted(nick,fileName,port,resumeOffset);
            }
        }
        void handleDCCReply(string source, string command) {
        }

        void handleEventCTCP(bool isReply,string source,string target,string content) {
            string sourceNick = Nicks.Extract(source);
            int walk = 0;
            walk += IRCUtils.Misc.ScanWord(content,walk);
            string what = content.Substring(0,walk).ToUpperInvariant();
            if (CurEvent.Bounced && !shouldProcessBouncedCTCP(what)) return;
            walk += IRCUtils.Misc.ScanSpace(content,walk);
            string remaining = content.Substring(walk);
            
            string context = Nicks.Extract(target);
            if (Nicks.Equals(target,ownNick)) {
                context = Nicks.Extract(source);
            }

            if (isReply) {
                bool shouldPrint = true;
                bool shouldPrintFull = true;
                if (what == "PING") {
                    try {
                        string[] strings = remaining.Split(':');
                        if (strings.Length != 2) throw new InvalidIRCEvent("Invalid CTCP PING reply");
                        TimeSpan elapsed = TimeSpan.FromTicks( DateTime.UtcNow.Ticks - Convert.ToInt64( strings[1] ) );
                        if (strings[0] == "O") {
                            printContextOutput(sourceNick,"PING Reply from " + sourceNick + ": " + elapsed.ToString());
                        } /*else if (strings[0] == "S") {
                            TimeSpan pingTime = TimeSpan.FromTicks(elapsed.Ticks/2);
                            if (pingTime >= TimeSpan.FromSeconds(10)) {
                                PrintServerOutput("Lag warning: " + pingTime.ToString());
                            }
                        }*/
                    } catch(Exception e) {
                        PrintServerOutput("Could not process PING reply: " + e.Message);
                    }
                    shouldPrint = false;
                } else if (what == "DCC") {
                    handleDCCReply(source,remaining);
                    shouldPrint = false;
                }                
                if (shouldPrint) {
                    string msg = "CTCP " + what + " Reply";
                    if (shouldPrintFull && remaining != "") msg += ": " + remaining;
                    handleEventNotice(source,target,msg);
                }
            } else {
                bool shouldPrint = true;
                bool shouldPrintFull = true;
                if (what == "PING") {
                    shouldPrint = false;
                    addCommand(Commands.CTCPReply(Nicks.Extract(source),"PING",remaining));
                } else if (what == "VERSION") {
                    addCommand(Commands.CTCPReply(Nicks.Extract(source),"VERSION","Monochrome " + Globals.VersionForCTCP));
                } else if (what == "ACTION") {
                    shouldPrint = false;
                    handleAction(context,source,remaining);
                } else if (what == "DCC") {
                    shouldPrint = false;
                    handleDCC(source,remaining);
                } else if (what == "TIME") {
                    //Fri Mar 30 11:33:23 2007
                    //ddd, MMM d "'"yy
                    addCommand(Commands.CTCPReply(Nicks.Extract(source),"TIME",DateTime.Now.ToString("ddd MMM d HH:mm:ss yyyy")));
                }
                if (shouldPrint) {
                    string msg = "CTCP " + what;
                    if (shouldPrintFull && remaining != "") msg += ": " + remaining;
                    handleEventNotice(source,target,msg);
                }
            }
        }

        void handleEventMessageEx(bool isNotice,string source, string target, string content) {
            string messageFiltered = "";
            for(int walk = 0;;) {
                int quoteStart = content.IndexOf((char)1,walk);
                if (quoteStart < 0) {
                    messageFiltered += content.Substring(walk);
                    break;
                } else {
                    messageFiltered += content.Substring(walk,quoteStart);
                }
                quoteStart++;
                int quoteEnd = content.IndexOf((char)1,quoteStart);
                if (quoteEnd < 0) break;
                try {
                    handleEventCTCP(isNotice,source,target,content.Substring(quoteStart,quoteEnd-quoteStart));
                } catch(Exception e) {
                    PrintServerOutput((isNotice ? "CTCP reply" : "CTCP") + " handling failure: " + e.Message);
                }
                walk = quoteEnd + 1;
            }
            if (messageFiltered.Length > 0) {
                setUserAwayDispatch(Nicks.Extract(source),false,null);
                if (isNotice) {
                    try {
                        handleEventNotice(source,target,messageFiltered);
                    } catch(Exception e) {
                        PrintServerOutput("Notice handling failure: " + e.Message);
                        throw;
                    }
                } else {
                    try {
                        handleEventMessage(source,target,messageFiltered);
                    } catch(Exception e) {
                        PrintServerOutput("Message handling failure: " + e.Message);
                        throw;
                    }
                }
            }
        }

        void handleAction(string context, string source, string action) {
            openQuery(context,false);
            bool trigger = IsTrigger(action) && !testHalfIgnore(context);
            printContextOutput(
                context,
                TextLine.Simple("* ") + TextLine.UserName(Nicks.Extract(source),CurEvent.Bounced) + TextLine.Simple(" ") + TextLine.FromIRC(action),
                trigger);
            //printContextOutput(context,"<" + TextLine.UserName( contextUserPrefix(context,sourceNick) + sourceNick,CurEvent.Bounced) + "> " + TextLine.FromIRC(content),isTrigger);

            //printContextOutput(context, formatAction(source,action), IsTrigger(action));
        }

        public bool testHalfIgnore(string context) {
            string source = CurEventSource;
            if (source == null) return false;
            //if (CurEvent != null && CurEvent.Bounced) return true;
            string[] masks = Prefs.getHalfIgnoreList();
            bool contextIsChannel = context != null ? InstanceData.IsChannelName(context) : false;
            foreach(string entry in masks) {
                if (InstanceData.IsChannelName(entry)) {
                    if (contextIsChannel) {
                        string[] split = entry.Split('@');
                        switch(split.Length) {
                            case 1:
                                if (Misc.WildCardTest(split[0],context)) return true;
                                break;
                            case 2:
                                if (Misc.WildCardTest(split[0],context) && Misc.WildCardTest(split[1],m_netName)) return true;
                                break;
                        }
                    }
                } else {
                    if (Nicks.MaskTest(entry,source)) return true;
                }
            }
            return false;
        }

        bool printContextOutput_ResolveWindow(string context, out IMessageWindow window) {
            if (context == null) {
                window = m_serverWindow;
                return true;
            } else return IRCWindows.TryGetValue(context,out window);
        }
        private void SetLogTarget( string context, object target ) {
            IMessageWindow window;
            if (printContextOutput_ResolveWindow( context, out window ) ) {
                window.SetLogTarget( target );
            }
        }

        public void printContextOutput(string context, TextLine line) {printContextOutput(context,line,1);}
        public void printContextOutput(string context, TextLine line, bool isHighlight) {printContextOutput(context,line,isHighlight ? 3 : 2);}
        public void printContextOutput(string context, TextLine line, int highlightLevel) {
            if (!BSFilter.Test( new string[] {  line.ToString() } ) ) return;

            try {
                Logging.WriteLine(m_netName,context,line.ToString(), (object obj) => { SetLogTarget(context, obj); } );
            } catch(Exception e) {
                PrintOutput("Logging failure: " + e.Message);
            }
            int flashLevel = testHalfIgnore(context) ? 1 : highlightLevel;
            m_delegate.LogActivity( context, line, flashLevel );
            IMessageWindow window;
            if (printContextOutput_ResolveWindow(context,out window)) {
                printOutputHere( window, line, highlightLevel >= 3);
                window.Flash(flashLevel);
            } else {
                if (context.Length > 0) line = TextLine.ContextPrefix(context) + line;
                printOutputHere(m_serverWindow, line, highlightLevel >= 3);
            }
        }

        bool IsTrigger(string content) {
            bool caseSensitive = Prefs.triggerWordsCaseSensitive;
            if (Nicks.IsTrigger(ownNick,content,caseSensitive)) return true;
            foreach(string entry in Prefs.triggerWordList) {
                if (Nicks.IsTrigger(entry,content,caseSensitive)) return true;
            }
            return false;
        }

        bool GetContextWindow(string context, out IMessageWindow wnd) {
            if (context == null) {wnd = m_serverWindow; return true;}
            else return IRCWindows.TryGetValue(context,out wnd);
        }

        string contextUserPrefix(string context, string nick) {
            ChannelStatus chan;
            if ( m_channelStatus.TryGetValue( context, out chan ) ) {
                var prefix = chan.PrefixOfNick( nick );
                prefix = InstanceData.ModeSpecs.Prefixes.SimplifyPrefix( prefix );
                return prefix;
            }
            return "";
        }

        void onChannelActivity(string channel, string nick) {
            ChannelStatus s;
            if (m_channelStatus.TryGetValue(channel, out s)) {
                s.HandleActivity( nick );
            }
        }

        void handleEventMessage(string source,string target,string content) {
            string context;
            string targetNick = /*Nicks.Extract*/ (target);
            string sourceNick = Nicks.Extract(source);
            if (Nicks.Equals(target,ownNick)) {
                context = sourceNick;
            } else {
                context = targetNick;
            }
            
            openQuery(context,false);
            bool isTrigger = false;
            if (InstanceData.IsChannelName(context)) {
                onChannelActivity(context,sourceNick);
                if (!CurEvent.Bounced) isTrigger = IsTrigger(content) && !testHalfIgnore(source);
            }

            printContextOutput(context,"<" + contextUserPrefix(context,sourceNick) + TextLine.UserName( sourceNick,CurEvent.Bounced) + "> " + TextLine.FromIRC(content),isTrigger);
        }
        

        async Task SetOwnHostName(string hostName) {
            if (m_instanceData.OwnHostName == null) {
                m_instanceData.OwnHostName = hostName;
                try {
                    m_instanceData.OwnIPForDCC = await NetStream.ResolveHostNameForDCC( hostName );
                } catch(Exception e)
                {
                    PrintServerOutput("Could not resolve own host IP: " + e.Message);
                }
            }
        }

        bool isBeingWhoisd(string nick) {
            if (m_instanceData.CurrentWhoisResponseNick == null) return false;
            else return Nicks.Equals(nick,m_instanceData.CurrentWhoisResponseNick);
        }

        void showAwaySpam(string nick, string reason) {
            if (isBeingWhoisd(nick) || m_awayFilter.Filter(nick,reason)) printContextOutput(nick,TextLine.Simple(nick + " is away: ") + TextLine.FromIRC(reason));
        }

        void processReceivedEvent(IRCEvent eventData) {
            // No longer needs thread care
            __processReceiventEvent_synced( eventData );
        }

        void handleWhois(UInt32 code, IRCEvent ev) {
            string nick = ev.Data[1];
            m_instanceData.CurrentWhoisResponseNick = (code != 318 && code != 369) ? nick : null;
            switch(code) {
                case 314://RPL_WHOWASUSER
                    printContextOutput(nick, nick + " was " + ev.Data[2] + "@" + ev.Data[3] + " : " + ev.Data[5]);
                    break;
                case 311://RPL_WHOISUSER
                    printContextOutput(nick, nick + " is " + ev.Data[2] + "@" + ev.Data[3] + " : " + ev.Data[5]);
                    break;
                case 312://RPL_WHOISSERVER
                    printContextOutput(nick, nick + " uses " + ev.Data[2] + " : " + ev.Data[3]);
                    break;
                case 313:
                    printContextOutput(nick, nick + " is an IRC operator");
                    break;
                case 317:
                    printContextOutput(nick, nick + " has been idle for " + TimeSpan.FromSeconds(Convert.ToUInt32(ev.Data[2])).ToString() );
                    break;
                case 319:
                    printContextOutput(nick, nick + " is on : " + ev.Data[2]);
                    break;
                case 301:
                    setUserAwayDispatch(nick,true,ev.Data[2]);
                    break;
                case 318:
                case 369:
                    break;
                default:
                    {
                        string message = nick;
                        for(int walk = 2; walk < ev.Data.Length; ++walk) message += " " + ev.Data[walk];
                        printContextOutput(nick,message);
                    }
                    break;
            }
        }

        static bool EventIgnorable(IRCEvent ev) {
            string what = ev.What.ToUpperInvariant();
            return what == "PRIVMSG" || what == "NOTICE" || what == "INVITE";
        }

        bool IgnoreEvent(IRCEvent ev) {
            if (EventIgnorable(ev)) {
                foreach(string entry in Prefs.getIgnoreList()) {
                    if (Nicks.MaskTest(entry,ev.Source)) return true;
                }
            }
            return false;
        }

        void setServerParam(string name) {}
        void setServerParam(string name, string value) {
            if (name == "CHANMODES") {
                string[] modes = value.Split(',');
                if (modes.Length != 4) throw new InvalidIRCEvent("Invalid CHANMODES parameter");
                m_instanceData.ModeSpecs.ChanModes = modes;
            } else if (name == "CHANTYPES") {
                m_instanceData.ChanTypes = value;
            } else if (name == "PREFIX") {
                PrefixMap map;
                if (PrefixMap.TryParse(value,out map)) m_instanceData.ModeSpecs.Prefixes = map;
                else PrintServerOutput("Invalid PREFIX parameter");
            } else if (name == "MODES") {
                int modes = Convert.ToInt32(value);
                if (modes < 1) throw new InvalidIRCEvent("Invalid MODES parameter");
                InstanceData.Modes = modes;
            } else if (name == "NICKLEN") {
                int nickLen = Convert.ToInt32(value);
                if (nickLen < 6) throw new InvalidIRCEvent("Invalid NICKLEN parameter");
                InstanceData.NickLen = nickLen;
            }
        }

        public string PretruncateNick(string nick) {
            int len = InstanceData.NickLen;
            if (len > 0) {
                int outLen;
                if (TextUtils.FindTruncationPoint(serverParams.textEncoding,nick,len,out outLen)) {
                    if (outLen == 0) throw new Exception("Invalid nick");
                    nick = nick.Substring(0,outLen);
                }
            }
            return nick;
        }

        public string PreprocessNick(string nick) {
            //todo try stripping invalid chars?
            return PretruncateNick(nick);
        }

        public string ChanTypes { get { return m_instanceData.ChanTypes;} }

        bool IsKnownChanType(char c) {
            return m_instanceData.ChanTypes.IndexOf(c) >= 0;
        }

        public void setUserNotAwayDispatch(string nick) {
            setUserAwayDispatch(nick,false,null);
        }

        public void setUserAwayDispatch(string nick, bool isAway,string reasonOptional) {
            foreach(IMessageWindow wnd in IRCWindows.Values) {
                wnd.SetUserAway(nick,isAway,reasonOptional);
            }
        }

        bool isCurrentWhoOutput(string request) {
            return m_instanceData.CurrentWhoReplyOutput != null && Nicks.Equals(request,m_instanceData.CurrentWhoReplyOutput);
        }

        void __processReceiventEvent_synced(IRCEvent eventData) {
            IRCEvent oldEvent = CurEvent;//should always be null at this point but let's store the old value to be futureproof
            try {
                CurEvent = eventData;
                __processReceiventEvent_synced_ex();
            } finally {
                CurEvent = oldEvent;
            }
        }

        void __processReceiventEvent_synced_ex() {
            IRCEvent eventData = CurEvent;
            try {
                if (IgnoreEvent(eventData)) return;
                if (!eventData.Bounced) userStringHint(eventData.Source);
                foreach( EventStream str in eventStreams ) str.Add( eventData );
                if (eventData.IsThreeDigitCode) {
                    UInt32 statusCode = Convert.ToUInt32(eventData.What);
                    string statusTarget = eventData.Data[0];
                    if (Nicks.IsValid(statusTarget)) {
                        if (!Nicks.Equals(ownNick,statusTarget)) {
                            PrintServerOutput("WARNING: own nick changed silently.");
                            onNickChange(ownNick,statusTarget);
                        }

                        //OnLoggedInCheck();
                    }
                    bool shouldPrint = true;
                    if (statusCode == 311 || statusCode == 314 || InstanceData.CurrentWhoisResponseNick != null) {
                        handleWhois(statusCode,eventData);
                        shouldPrint = false;
                    }
                    if (InstanceData.ReceivingNetworkSpecs && statusCode != 5) {
                        InstanceData.ReceivingNetworkSpecs = false;
                        OnLoggedInCheck();
                    }
                    string printContext = null;
                    switch(statusCode) {
                        case 401:
                            printContext = eventData.Data[1];
                            break;
                        case 372://MOTD
                            OnLoggedInCheck();
                            break;
                        case 437:
                            if (IRCUtils.Events.IsNickTemporarilyUnavailable(eventData) && !InstanceData.LoggedIn) {
                                shouldPrint = false;
                                string newNick;
                                for (; ;)
                                {
                                    newNick = TryNextNick();
                                    if (newNick != ownNick) break;
                                }
                                PrintServerOutput(eventData.StatusMessageAvailable ? eventData.StatusMessage : "Nick temporarily unavailable.");
                                onNickChange(ownNick, newNick);
                                addCommand(new IRCCommand("NICK", newNick));
                            }
                            break;
                        case 433:
                            if (!InstanceData.LoggedIn) {
                                shouldPrint = false;
                                string newNick;
                                for(;;) {
                                    newNick = TryNextNick();
                                    if (newNick != ownNick) break;
                                }
                                PrintServerOutput(eventData.StatusMessageAvailable ? eventData.StatusMessage : "Nick already in use.");
                                onNickChange(ownNick,newNick);
                                addCommand(new IRCCommand("NICK",newNick));
                            }
                            break;
                        case 305:
                            setUserAwayDispatch(ownNick,false,null);
                            break;
                        case 306:
                            setUserAwayDispatch(ownNick,true,null);
                            break;
                        case 5:
                            if (InstanceData.LoggedIn) throw new InvalidIRCEvent("Network Specifications not expected at this point");
                            InstanceData.ReceivingNetworkSpecs = true;
                            //network specs
                            for(int walk = 1; walk + 1 < eventData.Data.Length; ++walk) {
                                string[] split = eventData.Data[walk].Split('=');
                                switch(split.Length) {
                                    case 1:
                                        setServerParam(split[0]);
                                        break;
                                    case 2:
                                        setServerParam(split[0],split[1]);
                                        break;
                                }
                            }
                            break;
                        case 352:// /WHO reply
                            {
                                shouldPrint = false;
                                string request = eventData.Data[1];
                                string ident = eventData.Data[2];
                                string host = eventData.Data[3];
                                string server = eventData.Data[4];
                                string nick = eventData.Data[5];
                                string mode = eventData.Data[6];
                                string hopCount, realName;
                                {
                                    string[] hopCount_realName = eventData.Data[7].Split(new char[]{' '},2,StringSplitOptions.RemoveEmptyEntries);
                                    hopCount = hopCount_realName[0];
                                    if (hopCount_realName.Length == 2) {
                                        realName = hopCount_realName[1];
                                    } else {
                                        realName = "";
                                    }
                                }

                                userHostHintDispatch(nick,ident + "@" + host);
                                
                                bool isAway;
                                if (mode[0] == 'G') isAway = true;
                                else if (mode[0] == 'H') isAway = false;
                                else throw new InvalidIRCEvent();

                                setUserAwayDispatch(nick,isAway,null);
                                
                                if (m_instanceData.ExpectedWhoResponses.Contains(request)) {
                                    if (!isCurrentWhoOutput(request)) {
                                        m_instanceData.CurrentWhoReplyOutput = request;
                                        PrintServerOutput("Users on " + request + ":");
                                    }
                                    PrintServerOutput(nick.PadRight(10) + " : " + ident + "@" + host + " : " + realName);
                                }
                            }
                            break;
                        case 315://end of /WHO list
                            {
                                shouldPrint = false;
                                string request = eventData.Data[1];
                                if (isCurrentWhoOutput(request)) {
                                    PrintServerOutput("End of user list.");
                                }
                                m_instanceData.CurrentWhoReplyOutput = null;
                                m_instanceData.ExpectedWhoResponses.Remove(request);
                            }
                            break;
                        case 302:
                            shouldPrint = false;
                            if (eventData.Data.Length == 2) {
                                foreach(string entry in eventData.Data[1].Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries)) {
                                    string[] split1 = entry.Split(new char[]{'='},2);
                                    if (split1.Length == 2) {
                                        string[] split2 = split1[1].Split('@');
                                        if (split2.Length == 2) {
                                            string nick = split1[0];
                                            string host = split2[1];
                                            if (Nicks.Equals(m_ownNick,nick)) {
                                                PrintServerOutput("Own host: " + host);
                                                _= SetOwnHostName(host);
                                            } else {
                                                PrintServerOutput(nick + "'s host: " + host);
                                            }
                                        }
                                    }
                                }
                            }
                            break;
                        case 329:
                            //unsure WTF this is, but we don't want to spam user with it, happens as a response to /MODE
                            shouldPrint = false;
                            break;
                        case 324:
                            {
                                string channel = eventData.Data[1];
                                string total = eventData.Data[2];
                                for(int walk = 3; walk < eventData.Data.Length; ++walk) total += " " + eventData.Data[walk];
                                ChannelStatus( channel ).ChannelMode = total;
                                shouldPrint = false;
                            }
                            break;
                        case 341:
                            printContextOutput(eventData.Data[2],"Inviting " + eventData.Data[1]);
                            shouldPrint = false;
                            break;
                        case 353:
                            {
                                string channel = eventData.Data[2];
                                if (channel != InstanceData.NameListChannel) {
                                    nameList = new List<string>();
                                    InstanceData.NameListChannel = channel;
                                }
                                nameList.AddRange(eventData.Data[3].Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries));
                                shouldPrint = false;
                            }
                            break;
                        case 366:
                            {
                                if (nameList == null || InstanceData.NameListChannel == null) {
                                    nameList = null; InstanceData.NameListChannel = null;
                                    throw new InvalidIRCEvent("Unexpected Name List event");
                                }
                                SetChannelNameList(InstanceData.NameListChannel, nameList);
                                InstanceData.NameListChannel = null;
                                nameList = null;
                                shouldPrint = false;
                            }
                            break;
                        case 332:
                            {
                                string context = eventData.Data[1], topic = eventData.Data[2];
                                ChannelStatus( context ).Topic = topic;
                                printContextOutput(context,TextLine.FromIRC("Topic is: " + topic,TextStyle.Gray));
                                shouldPrint = false;
                            }
                            break;
                        case 333:
                            
                            //"topic set by...."
                            if ( ShowStatusMessages("TOPIC") )
                            {
                                string context = eventData.Data[1];
                                string setBy = eventData.Data[2];
                                string timestamp = eventData.Data[3];
                                
                                TextStyle style = TextStyle.Gray;

                                printContextOutput( 
                                    context,
                                    TextLine.Simple("Topic set by ",style) + TextLine.Simple(setBy,TextStyle.NickEx(style)) + TextLine.Simple(" on " +
                                    Time.IRCtoUTC(timestamp).ToLocalTime().ToString() 
                                    , style)
                                    );
                            }
                            
                            shouldPrint = false;
                            break;
                        case 482:
                            {
                                IChannelWindow channelWindow;
                                if (FindChannel(eventData.Data[1],out channelWindow)) {
                                    printOutputHere(channelWindow, eventData.Data[2]);
                                    shouldPrint = false;
                                }
                            }
                            break;
                        case 301:
                            if (!Nicks.Equals(ownNick,eventData.Data[1])) {
                                setUserAwayDispatch(eventData.Data[1],true,eventData.Data[2]);
                                showAwaySpam(eventData.Data[1],eventData.Data[2]);
                            }
                            shouldPrint = false;
                            break;
                        case 348: // exempt
                        case 367: // ban
                        case 728: // quiet
                        case 346: // invite
                            {
                                char type = 'b';
                                switch(statusCode)
                                {
                                    case 348: type = 'e'; break;
                                    case 367: type = 'b'; break;
                                    case 728: type = 'q'; break;
                                    case 346: type = 'I'; break;
                                }
                                string channel = eventData.Data[1];
                                BanDescription desc = new BanDescription();
                                desc.Mask = eventData.Data[2];
                                if (eventData.Data.Length > 3) desc.ByWhom = eventData.Data[3];
                                else desc.ByWhom = "Unknown";
                                if (eventData.Data.Length > 4) desc.Date = Time.IRCtoUTC(eventData.Data[4]);
                                else desc.Date = DateTime.MinValue;
                                
                                InstanceData.SetCurrentBanListChannel(type,channel);
                                InstanceData.CurrentBanList.AddLast(desc);
                                shouldPrint = false;
                            }                            
                            break;
                        case 349: // exept
                        case 368: // ban
                        case 729: // quiet
                        case 347: // invite
                            {
                                char type = 'b';
                                switch (statusCode)
                                {
                                    case 349: type = 'e'; break;
                                    case 368: type = 'b'; break;
                                    case 729: type = 'q'; break;
                                    case 347: type = 'I'; break;
                                }
                                string channel = eventData.Data[1];
                                InstanceData.SetCurrentBanListChannel(type,channel);

                                if ( ChannelStatus( channel ).IsOn ) {
                                    IChannelWindow wnd;
                                    if (FindChannel(channel,out wnd)) {
                                        wnd.OnBanList(type,InstanceData.CurrentBanList);
                                    }
                                }
                                
                                InstanceData.ResetBanList();
                                shouldPrint = false;
                            }
                            break;
                        default:

                            Debug.WriteLine("Unknown event: " + eventData.What);
                            break;
                    }

                    if (shouldPrint && eventData.Data.Length >= 2) {
                        TextLine message = TextLine.FromIRC(eventData.Data[1]);
                        for(int walk = 2; walk < eventData.Data.Length; ++walk) message += " " + TextLine.FromIRC(eventData.Data[walk]);
                        printContextOutput(printContext,message);
                    }
                } else if (eventData.What == "NOTICE") {
                    handleEventMessageEx(true,eventData.Source,eventData.Data[0],eventData.Data[1]);
                } else if (eventData.What == "PRIVMSG") {
                    handleEventMessageEx(false,eventData.Source,eventData.Data[0],eventData.Data[1]);
                } else if (eventData.What == "JOIN") {
                    UserJoined(eventData.Data[0],eventData.Source);
                } else if (eventData.What == "PART") {
                    UserParted(eventData.Data[0],eventData.Source,eventData.OptionalParam(1));
                } else if (eventData.What == "QUIT") {
                    string nick = Nicks.Extract(eventData.Source);
                    if ( ShowStatusMessages("QUIT") ) {
                        printUserActivity(nick, Formatting.formatMiscAction(eventData.Source, "has quit", eventData.OptionalParam(0), TextStyle.UserPart));
                    }
                    
                    foreach(ChannelStatus ch in m_channelStatus.Values) {
                        ch.RemoveNick( nick );
                    }
                    foreach(object wnd_ in IRCWindows.Values) {
                        var wnd = wnd_ as IChannelWindow;
                        if ( wnd != null ) {
                            wnd.UserQuit(eventData.Source);
                        }
                    }
                } else if (eventData.What == "NICK") {
                    string newNick = eventData.Data[0];
                    string oldNick = Nicks.Extract(eventData.Source);
                    onNickChange(oldNick,newNick);
                } else if (eventData.What == "KICK") {
                    string channel = eventData.Data[0];
                    string nick = eventData.Data[1];
                    string reason = eventData.OptionalParam(2);

                    UserKicked(channel, eventData.Source, nick, reason);
                } else if (eventData.What == "TOPIC") {
                    string channel = eventData.Data[0];
                    string newTopic = eventData.Data[1];
                    if ( ShowStatusMessages("TOPIC") ) {
                        printContextOutput(channel, Formatting.formatMiscAction(eventData.Source, "changes topic to: " + newTopic, null));
                    }                    
                    ChannelStatus( channel ).Topic = newTopic;
                } else if (eventData.What == "MODE") {
                    string target = eventData.Data[0];
                    if ( ShowStatusMessages("MODE") ) {
                        printContextOutput(Nicks.Equals(ownNick, target) ? null : target, Formatting.formatPrintMode(eventData, this));
                    }
                    
                    
                    ChannelStatus( target ).HandleMode(eventData, m_instanceData.ModeSpecs);

                    IChannelWindow window;
                    if ( FindChannel( target, out window ) ) {
                        window.HandleMode(eventData,m_instanceData.ModeSpecs);
                    }
                } else if (eventData.What == "INVITE") {
                    string target = eventData.Data[0];
                    if (Nicks.Equals(target,ownNick)) {
                        printOutput_threadSafe(Nicks.Extract(eventData.Source) + " invites you to " + eventData.Data[1]);
                    }
                /*} else if (eventData.What == "WALLOPS") {
                    PrintServerOutput("WALLOPS: " + eventData.Data[0]);*/
                } else if (eventData.What == "KILL") {
                    addReconnectRequest(ReconnectRequest.ErrorDelay);
                } else if (eventData.What == "PONG") {
                    string msg = "PONG from " + eventData.Source + " :";
                    foreach(string entry in eventData.Data) msg += " " + entry;
                    PrintServerOutput(msg);
                } else if (eventData.What == "ERROR") {
                    string message = eventData.Data[0];
                    PrintServerOutput("Error: " + message);
                } else if ( eventData.What == "CAP" ) {
                    if ( eventData.Data.Length == 3 ) {
                        if ( eventData.Data[1] == "ACK" && eventData.Data[2] == "sasl") {
                            _ = handleSASL();
                        }
                    }
                } else if ( eventData.What == "AUTHENTICATE" ) {
                    // SASL
                } else {
                    throw new Exception("Unknown Event");
                }
            } catch(Exception e) {
                HandleEventError();
                PrintServerOutput("Could not process \"" + eventData.What + "\" event: " + e.Message);
                if (eventData.Bounced) PrintServerOutput("Event was bounced");
                if (eventData.Raw != null) PrintServerOutput("Event content: " + eventData.Raw);
            }
        }
        async Task handleSASL()
        {
            try
            {
                InstanceData.SASLSuccessful = false;
                await IRCScripts.Auth.SASL.Run(new ScriptContextImpl(this, null), serverParams.nickAccount, serverParams.nickPassword);
                InstanceData.SASLSuccessful = true;
                // SASL OK
            } catch
            {
                // SASL failure
            }
            
        }

        void printUserActivity(string context, TextLine activity) {
            var nick = Nicks.Extract( context );
            if (IRCWindows.ContainsKey(nick)) printContextOutput(nick,activity);
            foreach( string key in IRCWindows.Keys ) {
                IChannelWindow cWnd = IRCWindows[key] as IChannelWindow;
                if (cWnd != null) {
                    var chanName = key;
                    if ( ChannelStatus( chanName ).HaveNick( nick ) ) {
                        printContextOutput( chanName, activity );
                    }
                }
            }
        }

        void onNickChange(string oldNick,string newNick) {
            if ( ShowStatusMessages("NICK") ) {
                printUserActivity(oldNick, Formatting.formatNickChange(oldNick, newNick));
            }
            

            foreach (ChannelStatus chan in m_channelStatus.Values) {
                chan.NickChanged(oldNick, newNick);
            }

            bool clash = IRCWindows.ContainsKey(newNick);

            if (!clash) {
                IMessageWindow window;
                if (IRCWindows.TryGetValue(oldNick,out window)) {
                    IRCWindows.Remove(oldNick);
                    IRCWindows.Add(newNick,window);
                }
            }
            foreach(IMessageWindow wnd in IRCWindows.Values) {
                wnd.HandleNickChange(oldNick,newNick,clash);
            }

            if (Nicks.Equals(oldNick,ownNick)) {
                m_ownNick = newNick;
                PrintServerOutput("Nick changed to " + newNick);
            }
        }

        void UserPartedCommon(string channel, string nick) {

            var channelStatus = ChannelStatus( channel );
            channelStatus.RemoveNick( nick );

            IChannelWindow channelWindow;
            bool isSelf = Nicks.Equals(nick,ownNick);
            if (FindChannel(channel,out channelWindow)) {
                channelWindow.UserParted(nick,isSelf);
            }
            if (isSelf) {
                IEnumerable<string> chanUsers = channelStatus.UserNicks;

                channelStatus.Unsubscribe();

                foreach(string name in chanUsers) {
                    if (IRCWindows.ContainsKey(name) && !isUserVisible(name)) {
                        printContextOutput(name,Formatting.formatMiscAction(name,"is offline: last seen on " + channel,null,TextStyle.UserPart));
                    }
                }
            } else {
                if (IRCWindows.ContainsKey(nick) && !isUserVisible(nick)) {
                    printContextOutput(nick,Formatting.formatMiscAction(nick,"is offline: left " + channel,null,TextStyle.UserPart));
                }
            }
        }
        bool ShowStatusMessages(string type) {
            return !m_hideMessages.Contains( type.ToUpper() );
        }
        void UserKicked(string channel, string source, string nick, string reason) {
            TextStyle style = TextStyle.UserPart;
            if (ShowStatusMessages("KICK"))  {
                printContextOutput(channel, Formatting.formatMiscAction(nick, TextLine.Simple("was kicked by ", style) + TextLine.UserNameEx(Nicks.Extract(source), style), reason, TextStyle.UserPart));
            }
            
            UserPartedCommon(channel,nick);
        }
        void UserParted(string channel, string user, string reason) {
            string nick = Nicks.Extract(user);
            if ( ShowStatusMessages("PART") ) {
                printContextOutput(channel, Formatting.formatMiscAction(user, "has left", reason, TextStyle.UserPart));
            }
            UserPartedCommon(channel, nick);
        }

        void UserJoined(string channel, string user) {
            string nick = Nicks.Extract(user);
            bool wasUserVisible = isUserVisible(nick);

            ChannelStatus( channel ).AddNick( nick );

            bool isSelf = Nicks.Equals(nick,ownNick);
            if (!IsClosing) {
                IChannelWindow channelWindow;
                if (!FindChannel(channel,out channelWindow)) {
                    channelWindow = OpenChannel( channel );
                }

                channelWindow.UserJoined(user,isSelf);
            }

            if (isSelf && !IsClosing) {
                //we don't know channel modes. query them.
                addCommand(new IRCCommand("MODE",channel));
                //we don't know user's hostnames & away status
                addCommand(new IRCCommand("WHO", channel));
            }

            if (IRCWindows.ContainsKey(nick) && !wasUserVisible) {
                printContextOutput(nick,Formatting.formatMiscAction(user,"is online: joined " + channel,null,TextStyle.UserJoin));
            }

            //important - printContextOutput after opening channel window
            if ( ShowStatusMessages("JOIN") ) {
                printContextOutput(channel, Formatting.formatMiscAction(user, "has joined", null, TextStyle.UserJoin));
            }
            
        }

        void SetChannelNameList(string channel, IEnumerable<string> userSpecList) {

            foreach(string userspec in userSpecList) {
                string nick = Nicks.Undecorate(userspec,InstanceData.ModeSpecs.Prefixes.PrefixList);
                if (IRCWindows.ContainsKey(nick) && !isUserVisible(nick)) {
                    printContextOutput(nick,Formatting.formatMiscAction(nick,"is online: seen on " + InstanceData.NameListChannel,null,TextStyle.UserJoin));
                }
            }

            ChannelStatus(InstanceData.NameListChannel).SetNameList( userSpecList );
        }

        static string reconnectMessage(TimeSpan delay) {
            return delay <= TimeSpan.FromSeconds(5) ? "Reconnecting ..." : "Reconnecting in " + delay + "; type \"/reconnect\" to reconnect immediately.";
        }

        void onDisconnected(Exception e) {

            ClearConnectedState();

            foreach (ChannelStatus chan in SubscribedChannels) {
                m_droppedChannels.Add( chan.ChannelName );
                chan.Unsubscribe();
            }


            bool wasConnected = InstanceData.LoggedIn;
            if (!Nicks.Equals(ownNick,serverParams.nickList[0])) onNickChange(ownNick,serverParams.nickList[0]);

            m_instanceData = new ServerConnectionInstanceData();
            
            foreach(ScriptContextImpl script in scriptsRunning) script.OnAborted();
            scriptsRunning.Clear();
            foreach(EventStream str in eventStreams) str.Cancel();
            eventStreams.Clear();

            commandQueue.Clear();
            commandQueueReady = new SemaphoreSlim(0);


            {
                string msg;
                e = walkNetException(e);

                if (e is SocketException) {
                    msg = "Disconnected: " + e.Message;
                } else if (e is ReconnectException) {
                    msg = reconnectMessage(((ReconnectException)e).RequestedDelay);
                } else {
                    msg = "Unexpected Protocol Error: " + e.Message;
                }
                if (wasConnected) PrintOutput_Dispatch(msg);
                else PrintServerOutput(msg);

                m_serverWindow.HandleDisconnection();
                foreach(IMessageWindow wnd in IRCWindows.Values) wnd.HandleDisconnection();
            }
            
        }

        void PrintOutput_Dispatch(string msg) {
            PrintServerOutput(msg);
            foreach(string name in IRCWindows.Keys) printContextOutput(name,msg);
        }


        string _TryNextNick() {
            string[] nickList = serverParams.nickList;
            if (InstanceData.Connect_OwnNickProgress < nickList.Length) {
                return nickList[InstanceData.Connect_OwnNickProgress++];
            }
            if (nickList.Length == 0) return Nicks.RandomEx();
            return Nicks.RandomDerive(nickList[0],ref InstanceData.Connect_RandomNickProgress);
        }
        string TryNextNick() {return PreprocessNick(_TryNextNick());}

        void processReceivedEvent_inThread(string eventString) {
            Debug.WriteLine(" > " + eventString);
            
            try {
                IRCEvent eventData = IRCEvent.FromInput(eventString);
                if (eventData.What == "PING") {
                    addCommand(new IRCCommand("PONG",eventData.Data[0]));
                } else {
                    processReceivedEvent(eventData);
                }
            } catch(Exception e) {
                try {
                    HandleEventError();
                    printOutput_threadSafe("Low-level event processing failure: " + e.Message);
                    printOutput_threadSafe(eventString);
                } catch {}
            }
        }

        void HandleEventError() {
            bool fail = false;
            lock(InstanceData.ServerErrorHelper) {
                InstanceData.ServerErrorHelper.OnError();
                fail = InstanceData.ServerErrorHelper.ShouldAbort;
            }
            if (fail) {
                printOutput_threadSafe("Too many errors - disconnecting from server");
                addReconnectRequest(ReconnectRequest.ErrorDelay);
            }
        }

        static Exception walkNetException(Exception e) {
            for(Exception walk = e;;) {
                if (walk is SocketException) return walk;
                if (walk.InnerException == null) return e;
                walk = walk.InnerException;
            }
        }

        void OnConnecting() {
            _ = logInTimerProc();
        }
        async Task logInTimerProc() {
            var instance = m_instanceData;
            await Task.Delay( 180000 );
            if ( IsClosing || instance != m_instanceData ) return;
            if (!instance.LoggedIn) {
                PrintOutput("Log in failure: timeout");
                addCommand(new ReconnectRequest(TimeSpan.FromMinutes(2)));
            }
        }

        static String debugRandomStr() {
            return Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx() + " " + Nicks.RandomEx();
        }

        string DebugUserString {
            get {
                return ownNick + "!bugger@debug";
            }
        }

        async void DebugInputParser() {
            try {
                while( !AbortToken.IsCancellationRequested ) {  
                    var cmd = (await pollCommand(TimeSpan.FromSeconds(10))) as IRCCommand;
                    if ( cmd != null ) {
                        Debug.WriteLine(cmd.MakeString());
                        if (cmd.command == "JOIN" && cmd.parameters.Length == 1) {
                            string chan = cmd.parameters[0];
                            var evt = new IRCEvent(DebugUserString, "JOIN", new string[] { chan });
                            processReceivedEvent( evt );
                            for (int i = 0; i < 100; ++i) {
                                evt = new IRCEvent(this.serverParams.hostName, "353", new string[] { this.ownNick, "=", chan, Nicks.Random() } );
                                processReceivedEvent(evt);
                            }
                            evt = new IRCEvent(this.serverParams.hostName, "366", new string[] { this.ownNick, chan, "End of /NAMES list" } );
                            processReceivedEvent( evt );
                            
                        } else if (cmd.command == "WHO") {
                            // FIX ME feed more random data
                        } else if (cmd.command == "MODE" && cmd.parameters.Length == 1) { 
                            var what = cmd.parameters[0];
                            var evt = new IRCEvent(this.serverParams.hostName, "324", new string[] { this.ownNick, what, "+" } );
                            processReceivedEvent( evt );
                        }
                    }
                }
            } catch { }
        }
        async void DebugProc(string task) {
            m_instanceData.StartupScriptFinished = true;
            DebugInputParser();
            try {
                foreach( var chan in this.serverParams.autoJoinChannelsList ) {
                    addCommand( new IRCCommand("JOIN", chan.Name ));
                }
                
                if ( task == "infinite-text.debug") {
                    for (;;) {
                        await Task.Delay(500, AbortToken);

                        // PrintOutput(debugRandomStr());
                        foreach (IMessageWindow wnd in IRCWindows.Values) {
                            printOutputHere(wnd, debugRandomStr());
                        }
                    }
                }
                if ( task == "100-lines.debug") {
                    await Task.Delay(500, AbortToken);
                    foreach (IMessageWindow wnd in IRCWindows.Values) {
                        for (int i = 0; i < 100; ++i) {
                            printOutputHere(wnd, "Line " + (i + 1));
                        }
                    }
                }
            } catch { }
        }

        CancellationToken AbortToken
        {
            get { return m_abortHandler.Token; }
        }

        async Task WorkerProc() {

            bool useSASL = serverParams.useSASL && serverParams.nickPassword.Length > 0 && serverParams.nickAccount.Length > 0;

            while(!IsClosing) {
                try {
                    printOutput_threadSafe("Connecting...");
                    InMainThread( () => { OnConnecting(); } );
                    
                    string identName = InstanceData.IdentName;
                    if (identName != null) IdentManager.Bump(identName);

                    ClearConnectedState();

                    using (NetStreamText stream = new NetStreamText(serverParams.textEncoding))
                    using ( CancellationTokenSource readLinesCancel = new CancellationTokenSource() )
                    {
                        DateTime timeOutBase = DateTime.UtcNow;
                        bool timeOutValid = false;

                        await stream.Open(
                            new NetStreamArg { remoteHost = serverParams.hostName.Trim(), remotePort = serverParams.hostPort, useSSL = serverParams.useSSL, verifySSL = serverParams.verifySSL },
                            AbortToken );

                        var readLinesTask = stream.ReadLinesLoop((string line) =>
                        {
                            timeOutValid = false;
                            processReceivedEvent_inThread(line);
                        }, readLinesCancel.Token );

                        if (useSASL)
                        {
                            await stream.SendLine("CAP REQ :sasl");
                        }

                        {
                            string strUser = serverParams.serverUser;
                            if ( strUser == null ) strUser = "";
                            if (strUser == "" && identName != null) strUser = identName;
                            if (strUser == "") strUser = Nicks.RandomEx();
                            await stream.SendLine("USER " + strUser + " asdf asdf :" + m_fullName, AbortToken);
                        }
                        if (serverParams.serverPassword.Length > 0)
                        {
                            await stream.SendLine("PASS " + serverParams.serverPassword, AbortToken);
                        }
                        await stream.SendLine("NICK " + ownNick, AbortToken);

                        Connected = true;

                        for (;;)
                        {

                            if (IsClosing)
                            {
                                Debug.WriteLine("Quitting: " + m_requestedQuitMessage);
                                try { await stream.SendLine(IRCUtils.Commands.Quit(m_requestedQuitMessage).MakeString() ); } catch { }
                                readLinesCancel.Cancel();
                                try { await readLinesTask; } catch { }
                                try { await stream.CloseGracefully(); } catch { }
                                return;
                            }

                            object cmd = await pollCommand(TimeSpan.FromSeconds(1));

                            if ( cmd != null )
                            {

                                try
                                {
                                    if (cmd is IRCCommand && !timeOutValid)
                                    {
                                        if (((IRCCommand)cmd).ExpectsServerReply)
                                        {
                                            timeOutBase = DateTime.UtcNow;
                                            timeOutValid = true;
                                        }
                                    }
                                    if (cmd is string || cmd is IRCCommand)
                                    {
                                        string tosend;
                                        if (cmd is string) tosend = (string)cmd;
                                        else tosend = ((IRCCommand)cmd).MakeString();
                                        Debug.WriteLine( "> " + tosend );
                                        await stream.SendLine(tosend, AbortToken);
                                    }
                                    else if (cmd is ReconnectRequest)
                                    {
                                        throw new ReconnectException(((ReconnectRequest)cmd).Delay);
                                    }
                                    else
                                    {
                                        Debug.WriteLine("Unknown Command");
                                    }
                                }
                                catch (Exception e)
                                {
                                    if (walkNetException(e) is SocketException || e is ReconnectException) throw;
                                    printOutput_threadSafe("Error while executing command: " + e.Message);
                                }
                            }

                            if (timeOutValid)
                            {
                                if (DateTime.UtcNow > timeOutBase + TimeSpan.FromSeconds(180))
                                {
                                    throw new TimeoutException();
                                }
                            }
                        }
                    }
                } catch(Exception e) {
                    try {
                        if (IsClosing) return;

                        {
                            // Fire onDisconnected() synchronously
                            // No longer needs special care, we *are* main thread now
                            onDisconnected(e);
                            /*
                            IAsyncResult result = InMainThread( () => {onDisconnected(e);} );
                            if (result != null) {
                                for(;;) {
                                    if (result.AsyncWaitHandle.WaitOne(TimeSpan.FromMilliseconds(10),false)) break;
                                    if (IsClosing) break;
                                }
                            }
                            */
                        }

                        TimeSpan delay;
                        if (e is ReconnectException) {
                            delay = ((ReconnectException)e).RequestedDelay;
                        } else {
                            delay = TimeSpan.FromSeconds(5);
                        }

                        DateTime waitBase = DateTime.UtcNow;
                        do {
                            if (IsClosing) return;
                            object obj = await pollCommand(TimeSpan.FromSeconds(1));
                            if (obj != null) {
                                if (obj is ReconnectRequest) {
                                    delay = ((ReconnectRequest)obj).Delay;
                                    waitBase = DateTime.UtcNow;
                                    printOutput_threadSafe(reconnectMessage(delay));
                                }
                            }
                        } while(DateTime.UtcNow - waitBase < delay);
                    } catch {
                        return;
                    }
                }

            }
        }

        bool Connected {
            get {
                return m_connectionAbortHandler != null;
            }
            set {
                if (! value ) ClearConnectedState();
                else SetConnectedState();
            }
        }

        CancellationToken ConnectionAbortToken {
            get {
                if ( ! Connected ) throw new InvalidOperationException();
                return m_connectionAbortHandler.Token;
            }
        }

        void SetConnectedState() {
            if ( m_connectionAbortHandler != null ) return;
            m_connectionAbortHandler = new CancellationTokenSource();
        }

        void ClearConnectedState() {
            if (m_connectionAbortHandler != null) {
                m_connectionAbortHandler.Cancel(); m_connectionAbortHandler.Dispose(); m_connectionAbortHandler = null;
            }
        }



        LinkedList<ChannelStatus> SubscribedChannels {
            get {
                LinkedList<ChannelStatus> ret = new LinkedList<ChannelStatus>();
                foreach( ChannelStatus chan in m_channelStatus.Values ) {
                    if ( chan.HaveNick( ownNick ) ) ret.AddLast( chan );
                }
                return ret;
            }
        }

        public void RequestClose(string quitMessage) {
            Debug.WriteLine("Request close on: " + this.NetName + " reason: " + quitMessage);
            if (IsClosing) return;

            if (quitMessage == null || quitMessage == "") {
                quitMessage = Prefs.defaultQuitMessage();
            }
            m_requestedQuitMessage = quitMessage;
            m_abortHandler.Cancel();

            string shutdownMsg = (quitMessage != null && quitMessage != "") ? "Shutting down: " + quitMessage : "Shutting down.";
            foreach (ChannelStatus channel in SubscribedChannels) {
                printContextOutput(channel.ChannelName, shutdownMsg);
            }
            printContextOutput(null, shutdownMsg);

            try {
                m_serverWindow.Close();
            } catch(Exception e) {
                Debug.WriteLine("ServerWindow.Close() failure: " + e.Message);
            }
            
            {
                var temp = new List<IMessageWindow>(IRCWindows.Values);
                foreach (IMessageWindow entry in temp) {
                    try {
                        entry.Close();
                    } catch(Exception e) {
                        Debug.WriteLine("IMessageWindow.Close() failiure: " + e.Message);
                    }
                    
                }
            }
        }

        public void WindowClosed() {
            RequestClose(null);
        }

        public bool IsClosing { get { return m_abortHandler == null || m_abortHandler.IsCancellationRequested; } }

        bool FindChannel(string name,out IChannelWindow window) {
            IMessageWindow temp;
            if (!IRCWindows.TryGetValue(name,out temp)) {window = null;return false;}
            if (!(temp is IChannelWindow)) {window = null; return false;}
            window = (IChannelWindow) temp;
            return true;
        }


        void OnLoggedInCheck() {
            if (!InstanceData.LoggedIn) {
                InstanceData.LoggedIn = true;
                OnLoggedIn();
            }
        }

        public bool IsLoggedIn { get { return InstanceData.StartupScriptFinished;} }

        void OnLoggedIn() {
            //PrintServerOutput("Logged in.");
            applyAwayEx(true);
            {
                IRCScripts.ServerStartupScript.Param param;
                param.NickList = (string[])serverParams.nickList.Clone();
                for(int walk = 0; walk < param.NickList.Length; ++walk) {
                    param.NickList[walk] = PreprocessNick(param.NickList[walk]);
                }
                param.ChanList = ArrayHelper.Arrayify(serverParams.autoJoinChannelsList);
                param.NickPassword = serverParams.nickPassword;
                param.NickAccount = serverParams.nickAccount;
                param.UserModes = serverParams.userModes;
                param.AllowRandomNick = serverParams.allowRandomNick;
                param.InviteHandlers = serverParams.GetInviteHandlers();
                param.ExtraStartupCommands = serverParams.GetExtraStartupCommands();
                param.ShouldAuthorize = ! InstanceData.SASLSuccessful;
                new IRCScripts.ServerStartupScript(new ScriptContextImpl(this,new StartupScriptTag()),param,m_droppedChannels);
            }

            _ = AwayCheckLoop();
        }

        async Task AwayCheckLoop() {
            var cancel = this.ConnectionAbortToken;
            var random = new System.Random();
            try {
                for( ;; ) {
                    await Task.Delay( TimeSpan.FromMinutes(1), cancel );
                    
                    // /WHO a random channel
                    var chans = this.SubscribedChannels;
                    if ( chans.Count > 0 ) {
                        int idx = random.Next() % chans.Count;
                        ChannelStatus chan = null;
                        foreach (var walk in chans) {
                            if (idx == 0) { chan = walk; break; }
                            --idx;
                        }
                        if (chan != null) {
                            this.addCommand(new IRCCommand("WHO", chan.ChannelName));
                        }
                    }
                }
            } catch {

            }
        }

        void StartupScriptDone() {
            m_instanceData.StartupScriptFinished = true;
            //if some of them could not be rejoined, it's not our problem anymore
            m_droppedChannels.Clear();
        }

        void ScriptNew(ScriptContextImpl script) {
            scriptsRunning.AddLast(script);
        }
        void ScriptDone(ScriptContextImpl script, bool quiet) {
            scriptsRunning.Remove(script);
            if (script.Tag != null && !quiet) {
                printOutput_threadSafe(script.Tag.ToString() + " finished OK");
            }

            if (script.Tag is StartupScriptTag) {
                StartupScriptDone();
            }
        }
        void InMainThread(Action a) {
            if ( IsClosing ) return;
            a();
        }
        
        
        void ScriptFailure(ScriptContextImpl script,Exception e) {
            lock(scriptsRunning) scriptsRunning.Remove(script);
            if (script.Tag != null) {
                printOutput_threadSafe(script.Tag.ToString() + " failed: " + e.Message);
                if (script.Tag is StartupScriptTag && !(e is ScriptAborted)) {
                    addReconnectRequest(TimeSpan.FromMinutes(5));
                }
            }
        }

        public string ownNick { get { return m_ownNick;} }

        static PreferencesData Prefs {
            get {return PreferencesManager.Current;}
        }

        void PrintServerOutput(TextLine message) {
            printContextOutput(null,message);
        }
        void PrintServerOutput(string message) {
            printContextOutput(null,message);
        }

        public string NetName { get { return m_netName; } }

        public ServerConnectionInstanceData InstanceData { get { return m_instanceData;} }

        public void DCCSendRemove( DCCSend send )
        {
            m_activeDCCSends.Remove( send );
        }
        public void DCCSendAdd( DCCSend send )
        {
            m_activeDCCSends.AddLast( send );
        }
        public void DCCReceiveRemove(DCCReceive recv )
        {
            m_activeDCCReceives.Remove( recv );
        }
        public void DCCReceiveAdd( DCCReceive recv )
        {
            m_activeDCCReceives.AddLast( recv );
        }


        static bool DCCMatch(DCCDescription desc,string nick, string fileName, UInt16 port) {
            return Nicks.Equals(desc.Nick,nick) && desc.FileName == fileName && desc.Port == port;
        }

        bool DCCResumeAccepted(string nick, string fileName,UInt16 port,UInt64 resumeOffset) {
            foreach(DCCReceive recv in m_activeDCCReceives) {
                if (DCCMatch(recv.Description,nick,fileName,port)) {
                    if (recv.ResumeAccepted( resumeOffset )) return true;
                }
            }
            return false;
        }

        bool DCCResumeRequested(string nick, string fileName,UInt16 port,UInt64 resumeOffset) {
            foreach(DCCSend send in m_activeDCCSends) {
                if (DCCMatch(send.Description,nick,fileName,port)) {
                    if (send.TryResume(resumeOffset)) {
                        addCommand(Commands.CTCP(nick,"DCC","ACCEPT " + IRCUtils.Commands.EncloseFilename(fileName) + " " + port + " " + resumeOffset));
                        return true;
                    }
                }
            }
            return false;
        }

        bool TestStartupInterrupt(IMessageWindow context) {
            if (!InstanceData.StartupScriptFinished) {
                printOutputHere(context, "Command ignored - Startup Script still active, please wait.");
                return false;
            } else {
                return true;
            }
        }

        public string ServerHostName { get { return serverParams.hostName; } }

        IMessageWindow m_serverWindow;
        IServerConnectionDelegate m_delegate;

        LinkedList<DCCSend> m_activeDCCSends;
        LinkedList<DCCReceive> m_activeDCCReceives;
        LinkedList<object> commandQueue;
        SemaphoreSlim commandQueueReady;
        Dictionary<string,IMessageWindow> IRCWindows;
        Dictionary<string, ChannelStatus> m_channelStatus = new Dictionary<string, ChannelStatus>( Nicks.Comparer );
        LinkedList<EventStream> eventStreams;
        LinkedList<ScriptContextImpl> scriptsRunning;
        IRCObjectList m_droppedChannels;
        string m_ownNick, m_fullName;
        ServerParams serverParams;
        string m_requestedQuitMessage;
        List<string> nameList;
        CancellationTokenSource m_abortHandler;
        CancellationTokenSource m_connectionAbortHandler = null;
        AwayMessageFilter m_awayFilter;
        

        string m_netName;
        ServerConnectionInstanceData m_instanceData;
        IRCEvent CurEvent;

        IRCUtils.AliasList m_aliases;
        SortedSet<string> m_hideMessages;

        bool m_debug;

        public bool CurrentEventBounced {
            get {
                IRCEvent ev = CurEvent;
                if (ev == null) return false;
                return ev.Bounced;
            }
        }
        string CurEventSource {
            get {
                if (CurEvent == null) return null;
                return CurEvent.Source;
            }
        }

        public void Dispose()
        {
            RequestClose(null);

            ClearConnectedState();


            commandQueueReady.Dispose();

            g_connections.Remove( this );

            if (m_abortHandler != null) {
                m_abortHandler.Cancel();
                m_abortHandler.Dispose();
                m_abortHandler = null;
            }
        }
    }


    public class ServerConnectionInstanceData {
        public ServerConnectionInstanceData() {
            PreferencesData prefs = PreferencesManager.Current;
            ExpectedWhoResponses = new IRCObjectList();
            ChanTypes = "#";
            ModeSpecs.Prefixes = PrefixMap.Default;
            FloodControl = new FloodControl();
            FloodQueue = new LinkedList<IRCCommand>();
            if (prefs.useIdent) IdentName = prefs.identName;
            ServerErrorHelper = new ServerErrorHelper();
            Modes = 3;
            ModeSpecs = ServerModeSpecs.Defaults;
            NickLen = 10;
        }
        public IRCObjectList ExpectedWhoResponses;
        public string CurrentWhoReplyOutput;
        public string OwnHostName;
        public string OwnIPForDCC;
        public string CurrentWhoisResponseNick;
        public bool StartupScriptFinished;
        public string ChanTypes;
        public ServerModeSpecs ModeSpecs;

        public LinkedList<IRCCommand> FloodQueue;
        public FloodControl FloodControl;

        public string IdentName;

        public ServerErrorHelper ServerErrorHelper;
        public string NameListChannel;
        public int Modes;

        public void SetCurrentBanListChannel(char type,string channel) {
            if (CurrentBanListType != type || CurrentBanList == null || CurrentBanListChannel == null || !Nicks.Equals(channel,CurrentBanListChannel)) {
                CurrentBanList = new LinkedList<BanDescription>();
                CurrentBanListChannel = channel;
                CurrentBanListType = type;
            }
        }
        public void ResetBanList() { CurrentBanListChannel = null; CurrentBanList = null; CurrentBanListType = '\0';}

        public string CurrentBanListChannel;
        public LinkedList<BanDescription> CurrentBanList;
        public char CurrentBanListType;
        public int Connect_RandomNickProgress;
        public int Connect_OwnNickProgress;
        public int NickLen;
        public bool LoggedIn = false;
        public bool SASLSuccessful = false;
        public bool ReceivingNetworkSpecs;
        
        public bool IsChannelName(string name) {
            return Nicks.IsChannel(name,ChanTypes);
        }

    };
}
