﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;

namespace monochrome
{
    class ServerConfigManager {

        public static LinkedList<string> ServerNames() {
            var ret = new LinkedList<string>();
            try {
                XmlNode servers = ServerConfigManager.OpenServerListNode();
                foreach (XmlNode node in servers.SelectNodes("server")) {
                    XmlAttribute name = node.Attributes.GetNamedItem("name") as XmlAttribute;
                    if (name != null) ret.AddLast(XmlConvert.DecodeName(name.Value));
                }
            } catch (Exception e) {
                Debug.WriteLine(e.Message);
            }
            return ret;
        }
        public static string DefaultServerStub() {
            return "[default]";
        }
        public static string DefaultServer() {
            string defaultServer = "";
            try {
                XmlNode servers = ServerConfigManager.OpenServerListNode();
                XmlNode node = servers.SelectSingleNode("defaultServer");
                if (node != null) {
                    XmlAttribute name = node.Attributes.GetNamedItem("value") as XmlAttribute;
                    if (name != null) defaultServer = name.Value;
                }
            } catch {
            }
            if (defaultServer == "") defaultServer = DefaultServerStub();
            return defaultServer;
        }
        public static void RemovePresets(LinkedList<string> removedPresets) {
            if (removedPresets.Count > 0) {
                try {
                    XmlNode serverList = ServerConfigManager.OpenServerListNode();
                    foreach (string entry in removedPresets) {
                        XmlNode node = ServerConfigManager.WalkServerNode(serverList, entry);
                        if (node != null) serverList.RemoveChild(node);
                    }
                    PreferencesManager.SetPreferencesXML(serverList.OwnerDocument);
                } catch (Exception e) {
                    Debug.WriteLine(e.Message);
                }
            }
        }

        public static void SetDefaultServer(string defaultServer) {
            XmlNode serverList = ServerConfigManager.OpenServerListNode();
            XmlNode node = serverList.SelectSingleNode("defaultServer");
            if (node == null) {
                node = serverList.OwnerDocument.CreateElement("defaultServer");
                serverList.AppendChild(node);
            }
            XmlAttribute attrib = node.Attributes.GetNamedItem("value") as XmlAttribute;
            if (attrib == null) {
                attrib = node.OwnerDocument.CreateAttribute("value");
                node.Attributes.Append(attrib);
            }
            if (attrib.Value != defaultServer) {
                attrib.Value = defaultServer;
                PreferencesManager.SetPreferencesXML(serverList.OwnerDocument);
            }
        }


        static ServerParams GrabServerParams(IPreferencesIO io) {
            ServerParams value = new ServerParams();
            value.Process(io);
            if (!value.isValid()) throw new ArgumentException();
            return value;
        }


        public static XmlNode OpenServerListNode() {
            XmlNode root = PreferencesManager.GetPreferencesXMLRoot();
            XmlNode list = root.SelectSingleNode("Servers");
            if (list == null) {
                list = root.OwnerDocument.CreateElement("Servers");
                root.AppendChild(list);
            }
            return list;
        }

        public static XmlNode WalkServerNode(XmlNode serverList, string serverName) {
            foreach (XmlNode n in serverList.SelectNodes("server")) {
                XmlAttribute name = n.Attributes.GetNamedItem("name") as XmlAttribute;
                if (name != null) {
                    if (XmlConvert.DecodeName( name.Value ) == serverName ) return n;
                }
            }
            return null;
        }

        public static XmlNode OpenServerNode(string name) {
            XmlNode list = OpenServerListNode();
            if (list == null) return null;
            return WalkServerNode(list, name);
        }
        public static XmlNode CreateServerNode(string name) {
            XmlNode list = OpenServerListNode();
            if (list == null) return null;
            XmlNode node = WalkServerNode(list, name);
            if (node == null) {
                XmlDocument doc = list.OwnerDocument;
                node = doc.CreateElement("server");
                XmlAttribute attrib = doc.CreateAttribute("name");
                attrib.Value = XmlConvert.EncodeName(name);
                node.Attributes.Append(attrib);
                list.AppendChild(node);
            }
            return node;
        }


        class ConfigNotFound : Exception {
            public ConfigNotFound() : base("Configuration Entry Not Found") { }
            public ConfigNotFound(string msg) : base(msg) { }
        };

        public static ServerParams GrabServerParamsByName(string name) {
            if (name == "") throw new ConfigNotFound();
            XmlNode node = OpenServerNode(name);
            if (node == null) throw new ConfigNotFound();
            return GrabServerParams(new PreferencesIO_Read_XML(node));
        }


        public static void StoreServerParamsByName(string name, ServerParams value) {
            if (name == "") throw new ConfigNotFound();
            XmlNode node = CreateServerNode(name);
            value.Process(new PreferencesIO_Write_XML(node));
            PreferencesManager.SetPreferencesXML(node.OwnerDocument);
        }

        public static LinkedList<KeyValuePair<string, ServerParams>> GrabAutoConnectEntries() {
            LinkedList<KeyValuePair<string, ServerParams>> output = new LinkedList<KeyValuePair<string, ServerParams>>();
            foreach (XmlNode node in OpenServerListNode().SelectNodes("server")) {
                try {
                    XmlAttribute name = node.Attributes.GetNamedItem("name") as XmlAttribute;
                    if (name != null) {
                        ServerParams svParams = GrabServerParams(new PreferencesIO_Read_XML(node));
                        if (svParams.autoConnect) {
                            output.AddLast(new KeyValuePair<string, ServerParams>(XmlConvert.DecodeName(name.Value), svParams));
                        }
                    }
                } catch { }
            }
            return output;
        }
    };
}
