﻿using System;
using System.Xml;
using System.Collections.Generic;

namespace monochrome
{
    

    partial class PreferencesManager {

        public const string RootSectionName = "MonochromeConfig";
        public const string ConfigFileName = "MonochromeConfig.xml";


        public static XmlNode GetPreferencesXMLRoot() {
            XmlDocument doc = GetPreferencesXML();
            XmlNode root = doc.SelectSingleNode(PreferencesManager.RootSectionName);
            if (root == null) {
                root = doc.CreateElement(PreferencesManager.RootSectionName);
                doc.AppendChild(root);
            }
            return root;
        }

        public static PreferencesData Current {
            get {
                if (!dataInitialized) {
                    try {
                        data = PreferencesData.FromXML(GetPreferencesXMLRoot());
                    } catch {
                        data = PreferencesData.Defaults;
                    }
                    //data = PreferencesData.ReadFromRegistry();
                    dataInitialized = true;
                }
                return data;
            }
        }
        public static void Set(PreferencesData newData) {
            XmlNode node = GetPreferencesXMLRoot();
            newData.ToXML(node);
            SetPreferencesXML(node.OwnerDocument);
            OnPreferencesChanged(newData);
        }

        public static void OnPreferencesChanged(PreferencesData newData) {
            data = newData;
            dataInitialized = true;
            if (Notify != null) Notify(newData);
        }

        public delegate void NotifyFunc(PreferencesData newData);
        public static event NotifyFunc Notify;

        static PreferencesData data;
        static bool dataInitialized;
    }

    public interface IPreferencesIO {
        void Process(ref string value, string name);
        void Process(ref bool value, string name);
        void Process(ref ColorDesc value, string name);
        void Process(ref int value, string name);
    }
    class PreferencesIO_BuildMap : IPreferencesIO {
        public void Process(ref string value, string name) {
            m_content.Add(name, value);
        }
        public void Process(ref bool value, string name) {
            m_content.Add(name, value);
        }
        public void Process(ref ColorDesc value, string name) {
            m_content.Add(name, value);
        }
        public void Process(ref int value, string name) {
            m_content.Add(name, value);
        }
        public static bool Equals(PreferencesIO_BuildMap v1, PreferencesIO_BuildMap v2) {
            if (v1.m_content.Count != v2.m_content.Count) return false;
            foreach (string key in v1.m_content.Keys) {
                object val1 = null, val2 = null;
                if (!v1.m_content.TryGetValue(key, out val1) || !v2.m_content.TryGetValue(key, out val2)) {
                    return false;
                }
                if (!val1.Equals(val2)) return false;
            }
            return true;
        }
        Dictionary<string, object> m_content = new Dictionary<string, object>();
    };

    abstract class PreferencesIO_Write : IPreferencesIO {
        public abstract void WriteEntry(string name, string value);

        public void Process(ref string value, string name) {
            if (value != null) WriteEntry(name,value);
        }
        public void Process(ref bool value, string name) {
            WriteEntry(name,value.ToString());
        }
        public void Process(ref ColorDesc value, string name) {
            if (value.IsEmpty) {
                WriteEntry(name,"");
            } else {
                WriteEntry(name,value.AsInt.ToString());
            }
        }
        public void Process(ref int value, string name) {
            WriteEntry(name,value.ToString());
        }
    }
    abstract class PreferencesIO_Read : IPreferencesIO {
        public abstract string ReadEntry(string name);

        public void Process(ref string value, string name) {
            string temp = ReadEntry(name);
            if (temp != null) value = temp;
        }
        public void Process(ref bool value, string name) {
            string temp = ReadEntry(name);
            if (temp != null) {
                try { value = Convert.ToBoolean(temp); } catch(FormatException) {}
            }
        }
        public void Process(ref ColorDesc value, string name) {
            string temp = ReadEntry(name);
            bool success = false;
            if (temp != null && temp != "") {
                try { value = ColorDesc.FromInt( Convert.ToInt32(temp) ); success = true; } catch(FormatException) {}
            }
            if (!success) value = ColorDesc.Empty;
        }

        public void Process(ref int value, string name) {
            string temp = ReadEntry(name);
            if (temp != null) {
                try {value = Convert.ToInt32(temp);} catch(FormatException) {}
            }
        }
    }

    class PreferencesIO_Read_XML : PreferencesIO_Read {
        public PreferencesIO_Read_XML(XmlNode node) {Node = node;}
        public override string ReadEntry(string name) {
            return XmlHelper.ReadStructField(Node,name);
        }
        XmlNode Node;
    }
    
    class PreferencesIO_Write_XML : PreferencesIO_Write {
        public PreferencesIO_Write_XML(XmlNode node) {
            XmlHelper.WipeStruct(node);
            Node = node;
        }
        public override void WriteEntry(string name, string value) {
            XmlHelper.WriteStructField(Node,name,value);
        }
        XmlNode Node;
    }

    public partial struct PreferencesData {
        public bool Valid {
            get {
                return NetUtils.PortRange.IsValidString(xferPortRange);
            }
        }

        void setDefaults() {
            defaultAwayMessages = "afk";
            defaultKickMessages = "";
            defaultPartMessages = "";
            defaultQuitMessages = "";
            fullName = ".";
            
            xferDefaultDirectory = "";
            xferDefaultDirectoryToken = "";
            xferPortRange = "1024-1030";
            xferAskForFileLocation = true;
            xferHideOutgoing = false;
            xferAAChannels = ""; xferAAUsers = "";
            ignore = halfIgnore = "";
            displayColors = true; displayFormatting = true;
            useLogging = false; loggingPath = ""; loggingPathToken = "";
            timestampFormat = "HH:mm";
            loggingTimestampFormat = "HH:mm:ss";
            splitLogsByDays = true;
            triggerWords = "";
            triggerWordsCaseSensitive = false;
            useIdent = true;
            identName = "user";
            splitLines = 400;
            unAwayOnActivity = false;
            awayOnMinimize = false;
            awayOnLock = true;
            useTriggerLog = false;
            useMessageLog = false;
            flashOnTrigger = true;
            useNickColors = false;
            outgoingMsgSecurity = false;
            defaultURLCopy = false;
            lineCountLimit = 1337;
            aliases = "";
            hideMessageTypes="";
            bsFilter = "";
            SetPlatformDefaults();
        }

        public static PreferencesData Defaults {
            get {
                PreferencesData data = new PreferencesData(); data.setDefaults(); return data;
            }
        }

        void _Process( IPreferencesIO io ) {
            io.Process(ref defaultQuitMessages,      "defaultQuitMessages");
            io.Process(ref defaultPartMessages,      "defaultPartMessages");
            io.Process(ref defaultAwayMessages,      "defaultAwayMessages");
            io.Process(ref defaultKickMessages,      "defaultKickMessages");
            io.Process(ref fullName,                 "fullName");
            io.Process(ref fontName,                 "fontName");
            io.Process(ref fontStyle,                "fontStyle");
            io.Process(ref fontSize,                 "fontSize");
            io.Process(ref listFontName,             "listFontName");
            io.Process(ref listFontStyle,            "listFontStyle");
            io.Process(ref listFontSize,             "listFontSize");
            io.Process(ref xferPortRange,            "xferPortRange");
            io.Process(ref xferAskForFileLocation,   "xferAskForFileLocation");
            io.Process(ref xferHideOutgoing,         "xferHideOutgoing");
            io.Process(ref xferDefaultDirectory,     "xferDefaultDirectory");
            io.Process(ref xferDefaultDirectoryToken, "xferDefaultDirectoryToken");
            io.Process(ref xferAAUsers,              "xferAAUsers");
            io.Process(ref xferAAChannels,           "xferAAChannels");
            io.Process(ref ignore,                   "ignore");
            io.Process(ref halfIgnore,               "halfIgnore");
            io.Process(ref displayColors,            "displayColors");
            io.Process(ref displayFormatting,        "displayFormatting");
            io.Process(ref beepOnTrigger,            "beepOnTrigger");
            io.Process(ref useLogging,               "useLogging");
            io.Process(ref loggingPath,              "loggingPath");
            io.Process(ref loggingPathToken,         "loggingPathToken");
            io.Process(ref timestampFormat,          "timestampFormat");
            io.Process(ref loggingTimestampFormat,   "loggingTimestampFormat");
            io.Process(ref splitLogsByDays,          "splitLogsByDays");
            io.Process(ref colorBackground,          "colorBackground");
            io.Process(ref colorText,                "colorText");
            io.Process(ref colorHighlight,           "colorHighlight");
            io.Process(ref colorChanHighlight1,      "colorChanHighlight1");
            io.Process(ref colorChanHighlight2,      "colorChanHighlight2");
            io.Process(ref colorChanHighlight3,      "colorChanHighlight3");
            io.Process(ref colorUserJoin,            "colorUserJoin");
            io.Process(ref colorUserPart,            "colorUserPart");
            io.Process(ref triggerWords,             "triggerWords");
            io.Process(ref triggerWordsCaseSensitive,"triggerWordsCaseSensitive");
            io.Process(ref useIdent,                 "useIdent");
            io.Process(ref identName,                "identName");
            io.Process(ref splitLines,               "splitLines");
            io.Process(ref unAwayOnActivity,         "unAwayOnActivity");
            io.Process(ref awayOnMinimize,           "awayOnMinimize");
            io.Process(ref awayOnLock,               "awayOnLock");
            io.Process(ref useTriggerLog,            "useTriggerLog");
            io.Process(ref useMessageLog,            "useMessageLog");
            io.Process(ref flashOnTrigger,           "flashOnTrigger");
            io.Process(ref useNickColors,            "useNickColors");
            io.Process(ref outgoingMsgSecurity,      "outgoingMsgSecurity");
            io.Process(ref defaultURLCopy,           "defaultURLCopy");
            io.Process(ref lineCountLimit,           "lineCountLimit");
            io.Process(ref aliases,                  "aliases");
            io.Process(ref hideMessageTypes,         "hideMessageTypes");
            io.Process(ref bsFilter,                 "bsFilter");
        }

        public void ToXML(XmlNode root) {
            XmlNode preferences = root.SelectSingleNode("Preferences");
            if (preferences == null) {
                preferences = root.OwnerDocument.CreateElement("Preferences");
                root.AppendChild(preferences);
            }
            _Process( new PreferencesIO_Write_XML(preferences) );
        }

        public static PreferencesData FromXML(XmlNode root) {
            try {
                PreferencesData temp = Defaults;
                XmlNode node = root.SelectSingleNode("Preferences");
                if (node != null) {
                    temp._Process(new PreferencesIO_Read_XML( node ) );
                }
                return temp;
            } catch(Exception e) {
                Debug.WriteLine(e.Message);
                return Defaults;
            }
        }

        /*public void WriteToRegistry() {
            using(RegistryKey key = ServerConfigManager.OpenRootKey()) {
                _Process( new PreferencesIO_Write_Registry(key) );
            }
        }*/
        public static string grabRandomMessage(string list) {
            if (list == null) return null;
            string[] messages = list.Split(new char[]{'\r','\n'},StringSplitOptions.RemoveEmptyEntries);
            if (messages.Length == 0) return null;
            Random rand = new Random();
            return messages[rand.Next(messages.Length)];
        }
        
        public string defaultQuitMessage() {return grabRandomMessage(defaultQuitMessages);}
        public string defaultAwayMessage() {return grabRandomMessage(defaultAwayMessages);}
        public string defaultPartMessage() {return grabRandomMessage(defaultPartMessages);}
        public string defaultKickMessage() {return grabRandomMessage(defaultKickMessages);}
        
        public static string[] splitMaskList(string list) {
            return list.Split(new char[]{' ',';',',','\r','\n'},StringSplitOptions.RemoveEmptyEntries);
        }

        public string[] triggerWordList {
            get { return splitMaskList(triggerWords); }
        }

        public static bool Equals(PreferencesData item1, PreferencesData item2) {
            var v1 = new PreferencesIO_BuildMap();
            var v2 = new PreferencesIO_BuildMap();
            item1._Process(v1); item2._Process(v2);
            return PreferencesIO_BuildMap.Equals( v1, v2 );
        }

        public override int GetHashCode() {
            return 0; // meaningless
        }
        public override bool Equals(object obj) {
            if (obj is PreferencesData) {
                return Equals(this, (PreferencesData)obj);
            } else {
                return false;
            }
        }
        public static bool operator==(PreferencesData item1, PreferencesData item2) {
            return Equals(item1,item2);
        }
        public static bool operator!=(PreferencesData item1, PreferencesData item2) {
            return !Equals(item1,item2);
        }
        public bool xferAAAvailable {
            get {
                if (xferDefaultDirectory.Length == 0) return false;
                return System.IO.Directory.Exists(xferDefaultDirectory);
            }
        }

        public string[] getXferAAUsers() {return splitMaskList(xferAAUsers);}
        public string[] getXferAAChannels() {return splitMaskList(xferAAChannels);}

        public string[] getIgnoreList() {return splitMaskList(ignore);}
        public string[] getHalfIgnoreList() {return splitMaskList(halfIgnore);}

        public string[] bsFilterTokens {
            get {
                return splitMaskList(bsFilter);
            }
        }


        public string defaultQuitMessages, defaultPartMessages, defaultAwayMessages, defaultKickMessages, fullName;
        public string fontName, fontStyle, fontSize;
        public string listFontName, listFontStyle, listFontSize;
        public string xferPortRange;
        public bool xferAskForFileLocation, xferHideOutgoing;
        public string xferDefaultDirectory, xferDefaultDirectoryToken;
        public string xferAAUsers, xferAAChannels;
        public string ignore, halfIgnore;
        public bool displayColors, displayFormatting;
        public bool beepOnTrigger;
        public bool useLogging;
        public string loggingPath, loggingPathToken;
        public string timestampFormat;
        public string loggingTimestampFormat;
        public bool splitLogsByDays;
        public ColorDesc colorText, colorBackground, colorHighlight, colorChanHighlight1, colorChanHighlight2, colorChanHighlight3, colorUserJoin, colorUserPart;
        public string triggerWords;
        public bool triggerWordsCaseSensitive;
        public bool useIdent;
        public string identName;
        public int splitLines;
        public bool unAwayOnActivity,awayOnMinimize,awayOnLock;
        public bool useTriggerLog, useMessageLog, flashOnTrigger;
        public bool useNickColors;
        public bool outgoingMsgSecurity;
        public bool defaultURLCopy;
        public int lineCountLimit;
        public string aliases;
        public string hideMessageTypes;
        public string bsFilter;

        public SortedSet<string> hideMessageTypeList {
            get {
                return new SortedSet<string>(splitMaskList(hideMessageTypes.ToUpper()));
            }
        }
    }

}