using System;
using System.Collections.Generic;

namespace NetUtils {

    public struct PortRange {
        public PortRange(UInt16 _Min, UInt16 _Max) {Min = _Min; Max = _Max;}
        public UInt16 Min, Max;


        public static LinkedList<PortRange> ParseString(string input) {
            return ParseString(input,true);
        }

        static bool IsParseException(Exception e) {
            return (e is FormatException) || (e is OverflowException);
        }

        public static LinkedList<PortRange> ParseString(string input, bool faultTolerant) {
            LinkedList<PortRange> list = new LinkedList<PortRange>();
            string[] strings = input.Split(new char[]{',',';',' '},StringSplitOptions.RemoveEmptyEntries);
            foreach(string entry in strings) {
                try {
                    string[] ex = entry.Split('-');
                    if (ex.Length == 2) {
                        list.AddLast(new PortRange(Convert.ToUInt16(ex[0]),Convert.ToUInt16(ex[1])));
                    } else if (ex.Length == 1) {
                        UInt16 port = Convert.ToUInt16(ex[0]);
                        list.AddLast(new PortRange(port,port));
                    } else {
                        throw new FormatException();
                    }
                } catch(Exception e) {
                    if (!IsParseException(e)) throw;
                    if (!faultTolerant) throw;
                }
            }
            return list;
        }


        public static bool IsValidString(string input) {
            try {
                ParseString(input,false);
                return true;
            } catch(Exception e) {
                if (IsParseException(e)) return false;
                throw;
            }
        }

        public static bool[] MakeTable(LinkedList<PortRange> list) {
            bool[] table = new bool[64*1024];
            foreach(PortRange entry in list) {
                for(UInt16 walk = entry.Min;;++walk) {
                    table[walk] = true;
                    if (walk == entry.Max) break;
                }
            }
            return table;
        }



    };


    public class AssignMorePorts : Exception {
        public AssignMorePorts() : base("Assign more ports!") {}
    };

    public class PortUsage : IDisposable {
        public PortUsage(LinkedList<PortRange> range) {
            m_port = Acquire(range);
            m_acquired = true;
        }

        void IDisposable.Dispose() {Close();}
        public void Close() {
            if (m_acquired) {
                m_acquired = false;
                Release(m_port);
            }
        }

        ~PortUsage() {Close();}

        public UInt16 Port {
            get {return m_port;}
        }

        UInt16 m_port;
        bool m_acquired;

        static UInt16 Acquire(LinkedList<PortRange> range) {
            lock(portUsageMap) {
                foreach(PortRange entry in range) {
                    for(UInt16 walk = entry.Min;;++walk) {
                        if (!portUsageMap[walk]) {
                            portUsageMap[walk] = true;
                            return walk;
                        }
                        if (walk == entry.Max) break;
                    }
                }
            }
            throw new AssignMorePorts();
        }

        static void Release(UInt16 port) {
            lock(portUsageMap) {
                portUsageMap[port] = false;
            }
        }

        static bool[] portUsageMap = new bool[64*1024];
    };
};