using System;
using System.Threading;
using System.Threading.Tasks;
using System.Text;


namespace monochrome {

    class NetStreamText : IDisposable {

        NetStream impl;

        public NetStreamText( Encoding _textEncoding, NetStream netStreamTakeOver = null )
        {
            textEncoding = _textEncoding;
            textEncodingIn = Encoding.GetEncoding( _textEncoding.CodePage, EncoderFallback.ExceptionFallback, DecoderFallback.ExceptionFallback );
//            textEncodingIn = (Encoding) _textEncoding.Clone();
//            textEncodingIn.DecoderFallback = DecoderFallback.ExceptionFallback;

            recvBuffer = new byte[0x10000];
            impl = netStreamTakeOver;
            if ( impl == null ) {
                impl = new NetStream();
            }
        }
        
        public async Task Accept(System.Net.IPAddress address, int port, CancellationToken token ) {
            await impl.Accept( address, port, token );
        }
        public async Task Open( NetStreamArg arg, CancellationToken token )
        {
            await impl.Open( arg, token );
        }
        public DateTime LastRead
        {
            get { lock(this) { return m_lastRead; } }
            set { lock(this) { m_lastRead = value; } }
        }

        public async Task CloseGracefully() {
            using( CancellationTokenSource cancel = new CancellationTokenSource( TimeSpan.FromSeconds(2))) {
                await ReadLinesLoop( (string s) => {}, cancel.Token );
            }
            Close();
        }

        public async Task SendLine( string Line )
        {
            await SendLine( Line, CancellationToken.None );
        }

        public async Task SendLine(string line, CancellationToken token ) {
            // Debug.WriteLine(" < " + line);
            var buffer = textEncoding.GetBytes(line + "\r\n"); 
            await impl.SendAsync( buffer, 0, buffer.Length, token );
        }


        public void Close() {
            impl.Close();
        }

        void IDisposable.Dispose() {
            Close();
        }


        public async Task<string> ReadLine(CancellationToken token) {
            string ret = null;
            await ReadLinesLoop( (string s) => { ret = s; }, token, 1);
            return ret;
        }

        public async Task ReadLinesLoop( Action<string> recv, CancellationToken token, int numRead = -1 )
        {
            int decrement = numRead;
            while ( decrement != 0 )
            {
                token.ThrowIfCancellationRequested();
                try
                {
                    if ( recvBuffer.Length == recvBufferUsed ) throw new OverflowException();
                    int delta = await impl.ReadAsync(recvBuffer,recvBufferUsed,recvBuffer.Length - recvBufferUsed, token);
                    if (delta <= 0) return;
                    recvBufferUsed += delta;
                    LastRead = DateTime.Now;
                } catch {
                    return;
                }
                ProcessIncoming( recv, token, ref decrement);
            }
        }

        System.Text.Encoding textEncoding, textEncodingIn;
        byte[] recvBuffer;
        int recvBufferUsed;

        string DecodeString(byte[] bytes, int index, int count) {
            try {
                return textEncodingIn.GetString(bytes,index,count);
            } catch(DecoderFallbackException) {
                return Encoding.GetEncoding(ServerParams.fallbackCharEncodingName).GetString(bytes,index,count);
            }
        }

        private void ProcessIncoming( Action<string> recv, CancellationToken abort, ref int decrement ) {
            var buffer = this.recvBuffer;
            var bufferUsed = this.recvBufferUsed;
            byte LF = 10, CR = 13; // can't just scan for \r\n due to various IRCD makers being incompetent clowns.
            int processed = 0;
            int walk = m_bufferParseOffset;
            while( walk < bufferUsed ) {
                int at = walk++;
                if (buffer[at] == LF) {
                    int point = at + 1;
                    int length = at - processed;
                    while (length > 0 && buffer[processed + length - 1] == CR) --length;
                    recv(DecodeString(buffer, processed, length));
                    processed = point;
                    if (abort.IsCancellationRequested) break;
                    if (decrement > 0) {
                        if (--decrement == 0) break;
                    }
                }
            }
            m_bufferParseOffset = walk - processed;
            RecvBufferAdvance( processed );

            abort.ThrowIfCancellationRequested();
        }

        void RecvBufferAdvance(int delta) {
            if (delta > recvBufferUsed) {
                throw new ArgumentOutOfRangeException();
            } else if (delta > 0) {
                int newUsed = recvBufferUsed - delta;
                for(int walk = 0; walk < newUsed; ++walk) recvBuffer[walk] = recvBuffer[walk + delta];
                recvBufferUsed = newUsed;
            }
        }

        int m_bufferParseOffset = 0;

        DateTime m_lastRead;
    };

};