using System;
using System.Collections.Generic;
using System.Collections;
using monochrome;

namespace IRCUtils {

    public class Modes {

        public struct Mode {
            public Mode(bool _sign, char _action, string _param,bool _paramIsNick) {sign = _sign; action = _action; param = _param; paramIsNick = _paramIsNick;}
            public char action;
            public string param;//param is null when the action doesn't require one
            public bool sign;
            public bool paramIsNick;
        };


        
        public static bool HideChannelMode(char action, ServerModeSpecs specs) {
            if (specs.ChanModes != null) {
                if (specs.ChanModes[0].IndexOf(action) >= 0) return true;
            }
            return specs.Prefixes.IsKnownMode(action);
        }

        static bool ModeNeedsParams(char action,bool sign, ServerModeSpecs specs) {
            bool dummy;
            return ModeNeedsParams(action,sign,specs,out dummy);
        }
        static bool ModeNeedsParams(char action,bool sign, ServerModeSpecs specs, out bool paramIsNick) {
            paramIsNick = false;
            if (specs.Prefixes.IsKnownMode(action)) {
                paramIsNick = true; return true;
            }
            if (specs.ChanModes != null) {
                if (specs.ChanModes[0].IndexOf(action) >= 0) return true;//user info mask
                if (specs.ChanModes[1].IndexOf(action) >= 0) return true;//always has a param
                if (specs.ChanModes[2].IndexOf(action) >= 0) return sign;//param only when setting
                if (specs.ChanModes[3].IndexOf(action) >= 0) return false;//no params
            }
            //should not get here unless the server is sending nonsense
            return false;
        }

        public static string MakeString(LinkedList<Mode> modes, ServerModeSpecs specs, bool channelTitle) {
            string ret = "";

            LinkedList<string> modeParams = new LinkedList<string>();
            {
                bool currentSign = false;
                bool signSet = false;
                foreach(Mode mode in modes) {
                    if (!channelTitle || !HideChannelMode(mode.action,specs)) {
                        if (!signSet || currentSign != mode.sign) {
                            ret += (mode.sign ? "+" : "-");
                            currentSign = mode.sign; signSet = true;
                        }
                        ret += mode.action;

                        if (ModeNeedsParams(mode.action,mode.sign,specs)) {
                            modeParams.AddLast(mode.param);
                        }
                    }
                }
            }
            foreach(string param in modeParams) {
                ret += " " + param;
            }
            return ret;
        }

        public static LinkedList<Mode> ParseEx(ArrayWrapper<string> source, int baseOffset, ServerModeSpecs modeSpecs) {
            int dataLen = source.Length;
            int dataBase = baseOffset + 1;
            LinkedList<Mode> results = new LinkedList<Mode>();
            if (dataLen >= dataBase) {
                int dataWalk = dataBase;
                string modes = source[baseOffset];
                bool sign = false;
                bool signKnown = false;
                foreach(char action in modes) {
                    switch(action) {
                        case '+': sign = true; signKnown = true; break;
                        case '-': sign = false; signKnown = true; break;
                        default:
                            if (!signKnown) throw new InvalidIRCEvent("Invalid MODE event - missing sign");
                            string modeParam = null;
                            bool paramIsNick = false;
                            if (ModeNeedsParams(action,sign,modeSpecs,out paramIsNick)) {
                                if (dataWalk >= dataLen) throw new InvalidIRCEvent("Invalid MODE event - not enough arguments");
                                modeParam = source[dataWalk++];
                            }
                            results.AddLast(new Mode(sign,action,modeParam,paramIsNick));
                            break;
                    }
                }
            }
            return results;
        }
        public static LinkedList<Mode> ParseEx(string[] source, int baseOffset,ServerModeSpecs modeSpecs) {
            return ParseEx(new ArrayWrapper<string>(source), baseOffset,modeSpecs);
        }
        public static LinkedList<Mode> Parse(string source,ServerModeSpecs modeSpecs) {
            return ParseEx( source.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries), 0, modeSpecs);
        }
        public static LinkedList<Mode> Parse(IRCEvent source,ServerModeSpecs modeSpecs) {
            if (source.What != "MODE") throw new InvalidIRCEvent("Expected a MODE event");
            return ParseEx(source.Data,1,modeSpecs);
        }
        public static string ModeTarget(IRCEvent source) {
            if (source.What != "MODE") throw new InvalidIRCEvent("Expected a MODE event");
            return source.Data[0];
        }
    };


    public class InvalidIRCEvent : Exception {
        public InvalidIRCEvent() : base("Invalid IRC Event") {}
        public InvalidIRCEvent(string what) : base(what) {}
    };
    public class InvalidNick : Exception {
        public InvalidNick() : base("Invalid Nick") {}
    }

    public class Nicks {
        public static bool MaskTest(string mask, string user) {
            if (mask.IndexOf('@') < 0) {
                if (mask.IndexOf('!') >= 0) return false;
                mask = mask + "!*@*";
            } else if (mask.IndexOf('!') < 0) {
                string[] meh = mask.Split('@');
                if (meh.Length != 2) return false;
                mask = meh[0] + "!*@" + meh[1];
            }
            return Misc.WildCardTest(mask,user);
        }

        public static bool IsChannel(string nick,string chanTypes) {
            if (nick.Length < 1) return false;
            return chanTypes.IndexOf(nick[0]) >= 0;
        }


        public static bool IsTriggerSeparator(char c) {
            if (c == '_') return false;
            return char.IsPunctuation(c) || char.IsSeparator(c);
        }

        public static bool IsTrigger(string triggerWord, string message,bool caseSensitive) {
            int substringBase = 0;
            int walk;
            for(walk = 0; walk < message.Length; ++walk) {
                char c = message[walk];
                if (IsTriggerSeparator(c)) {
                    if (substringBase < walk) {
                        string test = message.Substring(substringBase,walk - substringBase);
                        if (Misc.WildCardTest(triggerWord,test,caseSensitive)) return true;
                    }
                    substringBase = walk + 1;
                }
            }
            if (substringBase < walk) {
                string test = message.Substring(substringBase,walk - substringBase);
                if (Misc.WildCardTest(triggerWord,test,caseSensitive)) return true;
            }
            return false;
        }
        public static bool IsTrigger_BeginningOnly(string nick, string message) {
            int walk = 0;
            for(;;) {
                if (walk >= message.Length) break;
                char c = message[walk];
                if (char.IsPunctuation(c) || char.IsSeparator(c)) break;
                ++walk;
            }
            return nick == message.Substring(0,walk);
        }

        public static bool IsValid(string nick) {
            if (nick.Length == 0) return false;
            if (nick.IndexOfAny(new char[]{' ','.',':',';',',','*'}) >= 0) return false;
            return true;//TODO
        }
        public static string Validate(string nick) {
            nick = nick.Trim();
            if (!IsValid(nick)) throw new InvalidNick();
            return nick;
        }
        public static StringComparer Comparer = StringComparer.OrdinalIgnoreCase;
        public static bool Equals(string nick1,string nick2) {return Comparer.Equals(nick1,nick2);}
        
        public static string RandomDerive(string original, ref int status) {
            string temp = RandomDeriveUnchecked(original, ref status);
            if (temp.Length > 8) temp = temp.Substring(0,8);
            return temp;
        }
        public static string RandomDeriveUnchecked(string original, ref int status) {
            switch(status++) {
                case 0:
                    return "`" + original;
                case 1:
                    return original + "`";
                case 2:
                    return "``" + original;
                case 3:
                    return original + "``";
                case 4:
                    return "_" + original;
                case 5:
                    return original + "_";
                case 6:
                    return "[" + original + "]";
                default:
                    return status > 32 ? Random() : RandomEx();
            }
        }

        public static string RandomEx() {
            int length = monochrome.Globals.RandomNext(5,9);
            string ret = "";
            string matrix = "asdf";
            int offset = monochrome.Globals.RandomNext(matrix.Length);
            for(int walk = 0; walk < length; ++walk) {
                ret += matrix[(offset + walk) % matrix.Length];
            }
            return ret;
        }
        public static string Random() {
            string ret = new string[]{"meh","bah","doh","pah"}[monochrome.Globals.RandomNext(4)];
            for(int walk = 0; walk < 5; ++walk) {
                ret += monochrome.Globals.RandomNext(10);
            }
            return ret;
        }

        public static int StatusPrefixLength(string nick,string prefixList) {
            int walk = 0;
            while(walk < nick.Length && prefixList.IndexOf(nick[walk]) >= 0) ++walk;
            return walk;
        }

        public static string ExtractStatus(string nick,string prefixList) {
            return nick.Substring(0,StatusPrefixLength(nick,prefixList));
        }
        public static string Undecorate(string nick,string prefixList) {
            return nick.Substring(StatusPrefixLength(nick,prefixList));
        }
        public static string Extract(string userString) {
            if (userString.Length > 0) {
                if (userString[0] == '#') {
                    return userString;
                }
            }
            int cut = userString.IndexOfAny(new char[]{'@','!'});
            if (cut >= 0) return userString.Substring(0,cut);
            else return userString;
        }
        public static string ExtractHostmask(string userString) {
            int cut = userString.LastIndexOfAny(new char[]{'@','!'});
            if (cut >= 0) return userString.Substring(cut+1);
            else return "";
        }
        public static string ExtractUserHost(string userString) {
            int cut = userString.LastIndexOf('!');
            if (cut >= 0) return userString.Substring(cut+1);
            else return "";
        }
    };

    public class CTCP {
        /*
        public static string MakeDCCHost(System.Net.IPHostEntry host) {
            foreach(System.Net.IPAddress address in host.AddressList) {
                if (address.AddressFamily == System.Net.Sockets.AddressFamily.InterNetwork) {
                    //return address.ToString();
                    byte[] blah = address.GetAddressBytes();//ugly - todo: use address.ToString() output
                    if (blah.Length == 4) {
                        return (((UInt32) blah[3]) | ((UInt32) blah[2] << 8) | ((UInt32) blah[1] << 16) | ((UInt32) blah[0] << 24)).ToString();
                    }
                }
            }
            foreach(System.Net.IPAddress address in host.AddressList) {
                if (address.AddressFamily == System.Net.Sockets.AddressFamily.InterNetworkV6) {
                    return address.ToString();
                }
            }
            foreach(System.Net.IPAddress address in host.AddressList) {
                return address.ToString();
            }
            throw new Exception("Could not resolve own address");
        }
        */
        public static bool IsDCCHostNumber(string host) {
            foreach(char c in host) {
                if (!char.IsDigit(c)) return false;
            }
            return true;
        }

        public static string ParseDCCHost(string host) {
            if (IsDCCHostNumber(host)) {
                UInt32 raw = Convert.ToUInt32(host);
                string temp = (raw & 0xFF).ToString();
                raw >>= 8;
                temp = (raw & 0xFF).ToString() + "." + temp;
                raw >>= 8;
                temp = (raw & 0xFF).ToString() + "." + temp;
                raw >>= 8;
                temp = (raw & 0xFF).ToString() + "." + temp;
                return temp;
            } else {
                return host;
            }
        }

        public static string Enclose(string content) {
            char enclosure = (char)1;
            return enclosure + content + enclosure;
        }
        public static bool IsCTCP(string message) {
            if (message.Length < 2) return false;
            return message[0] == 1 && message[message.Length - 1] == 1;
        }
        public static string Extract(string message) {
            if (!IsCTCP(message)) throw new ArgumentException();
            return message.Substring(1,message.Length-2);
        }
    };

    public struct ChannelJoinEntry {
        public string Name, Key;
    }

    public class Commands {

        public static string EncloseFilename(string fn) {
            if (fn.IndexOf(' ') >= 0) fn = "\"" + fn + "\"";
            return fn;
        }

        public static ICollection<ChannelJoinEntry> SplitChannelJoinList(string inputList) {
            LinkedList<ChannelJoinEntry> output = new LinkedList<ChannelJoinEntry>();
            foreach(string walk in inputList.Split(',')) {
                string[] temp = walk.Split(new char[]{' '},2,StringSplitOptions.RemoveEmptyEntries);
                if (temp.Length > 0) {
                    ChannelJoinEntry entry = new ChannelJoinEntry();
                    entry.Name = temp[0];
                    if (temp.Length > 1) entry.Key = temp[1];
                    output.AddLast(entry);
                }
            }
            return output;
        }

        public static IRCCommand CTCP(string target,string type) {
            return Message(target,IRCUtils.CTCP.Enclose(type));
        }
        public static IRCCommand CTCP(string target,string type, string args) {
            return Message(target,IRCUtils.CTCP.Enclose(type + " " + args));
        }
        public static IRCCommand CTCPReply(string target,string type) {
            return Notice(target,IRCUtils.CTCP.Enclose(type));
        }
        public static IRCCommand CTCPReply(string target,string type, string args) {
            return Notice(target,IRCUtils.CTCP.Enclose(type + " " + args));
        }

        public static bool IsValidTarget(string target) {
            if ( target.Length == 0 ) return false;
            return target.IndexOfAny(new char[]{' '}) < 0;
        }
        public static string ValidateTarget(string target) {
            target = target.Trim();
            if (!IsValidTarget(target)) throw new InvalidIRCCommand();
            return target;
        }
        public static IRCCommand Message(string target, string content) {
            return new IRCCommand("PRIVMSG",new string[]{ValidateTarget(target),content});
        }
        public static IRCCommand Notice(string target, string content) {
            return new IRCCommand("NOTICE",new string[]{ValidateTarget(target),content});
        }
        public static IRCCommand Quit(string message) {
            if (message == null || message == "") {
                return new IRCCommand("QUIT");
            } else {
                return new IRCCommand("QUIT",new string[]{message});
            }
        }
    };
    public class Events
    {
        public static bool IsNickTemporarilyUnavailable(IRCEvent ev)
        {
            return ev.What == "437" && Lingustics.WordSequencePresent(ev.OptionalParam(2), "temporarily unavailable");
        }
    }
    public class Misc {
        static public bool IsThreeDigitCode(string msg) {
            if (msg.Length != 3) return false;
            for(int walk = 0; walk < 3; ++walk) {
                if (msg[walk] < '0' || msg[walk] > '9') return false;
            }
            return true;
        }
        public static bool WildCardTest(string mask, string test) {
            return WildCardTest(mask,test,false);
        }
        public static bool WildCardTestCaseSensitive(string mask, string test) {
            return WildCardTest(mask,test,true);
        }
        public static bool WildCardTest(string mask, string test, bool caseSensitive) {
            int walk = 0;
            for(;;) {
                if (mask.Length == walk) return test.Length == walk;
                char c = mask[walk];
                switch(c) {
                    case '?':
                        if (test.Length == walk) return false;
                        break;
                    case '*':
                        {
                            string maskRemaining = mask.Substring(walk+1);
                            for(;walk < test.Length; ++walk) {
                                if (WildCardTest(maskRemaining,test.Substring(walk),caseSensitive)) return true;                            
                            }
                            return WildCardTest(maskRemaining,"",caseSensitive);
                        }
                    default:
                        if (test.Length == walk) return false;

                        {
                            char t = test[walk];
                            if (caseSensitive) {
                                if (c != t) return false;
                            } else {
                                if (Char.ToUpperInvariant(c) != Char.ToUpperInvariant(t)) return false;
                            }
                        }
                        
                        break;                        
                }
                ++walk;
            }
        }

        public static string[] SplitWordWithQuotes(string text) {
            return SplitWordWithQuotes(text,int.MaxValue);
        }
        class StringSplitHelper {
            public StringSplitHelper(string[] output,int limit) {m_output = output; m_limit = limit;}

            public void Run(string text) {
                string current = "";
                bool inQuotes = false;
                for(int walk = 0; walk < text.Length; ++walk) {
                    char c = text[walk];
                    if (c == '\"') inQuotes = !inQuotes;
                    else if (!inQuotes && c == ' ') {
                        Output(current);
                        current = "";
                    } else {
                        current += c;
                    }
                }
                Output(current);
            }

            void Output(string what) {
                if (what.Length > 0) {
                    if (m_done == m_limit) {
                        if (m_done > 0 && m_output != null) {
                            m_output[m_done - 1] += " " + what;
                        }
                    } else {
                        if (m_output != null) m_output[m_done] = what;
                        m_done++;
                    }
                }
            }

            public int DoneCount() {return m_done;}
            
            int m_done, m_limit;
            string[] m_output;
        };
        static int SplitWordWithQuotes_Internal(string text, int limit,string[] output) {
            StringSplitHelper splitter = new StringSplitHelper(output,limit);
            splitter.Run(text);
            return splitter.DoneCount();
        }
        public static string[] SplitWordWithQuotes(string text, int limit) {
            string[] output = new string[SplitWordWithQuotes_Internal(text,limit,null)];
            SplitWordWithQuotes_Internal(text,limit,output);
            return output;
        }
        public static string ProcessOutgoing(string text) {
            return text.Replace("\t","  ").Replace((char)1,'_');
        }
        public static string ProcessIncoming(string text) {
            string ret = "";
            int added = 0;
            for(int walk = 0; walk < text.Length; ++walk) {
                int code = (int)text[walk];
                if (code < 32) {
                    int skip = 0;
                    if (code == 3) skip = 2;
                    else skip = 1;
                    ret += text.Substring(added, walk - added);
                    //ret += "[" + code.ToString() + "]";
                    
                    added = walk + skip;
                }
            }
            ret += text.Substring(added);
            return ret.Replace("\t","  ");
        }

        public static int ScanWord(string text,int offset) {
            int walk = offset;
            for(;;) {
                if (walk == text.Length) return walk - offset;
                if (text[walk] == ' ') return walk - offset;
                ++walk;
            }
        }
        public static int ScanSpace(string text,int offset) {
            int walk = offset;
            for(;;) {
                if (walk == text.Length) return walk - offset;
                if (text[walk] != ' ') return walk - offset;
                ++walk;
            }
        }
    };
    public struct ArrayWrapper<T> : IEnumerable<T> {
        public ArrayWrapper(T[] data) {m_data = data;}
        public T this[int i] {
            get { return m_data[i]; }
        }
        public int Length {
            get {return m_data.Length;}
        }
        IEnumerator<T> IEnumerable<T>.GetEnumerator() {
            IEnumerable<T> i = m_data;
            return i.GetEnumerator();
        }
        IEnumerator IEnumerable.GetEnumerator() {
            IEnumerable i = m_data;
            return i.GetEnumerator();
        }
        
        T[] m_data;
    };

    public class IRCEvent {
        IRCEvent() {}
        public IRCEvent(string _source, string _event, string[] _data) {
            m_source = _source; m_event = _event; m_data = (string[])_data.Clone();
        }

        public static IRCEvent BounceCommand(string ownNick,IRCCommand command) {
            if (command.quiet) return null;
            if (command.command == "PRIVMSG" || command.command == "NOTICE") {
                IRCEvent e = new IRCEvent(ownNick,command.command,command.parameters);
                e.m_bounced = true;
                return e;
            } else {
                return null;
            }
        }

        public static IRCEvent FromInput(string input) {
            IRCEvent e = new IRCEvent();
            e.fromInput(input);
            return e;
        }

        void fromInput(string input) {
            m_raw = input;
            if (input.Length > 0) {
                int walk = 0;
                if (input[0] == ':') {
                    walk++;
                    int delta = Misc.ScanWord(input,walk);
                    m_source = input.Substring(walk,delta);
                    walk += delta;
                    walk += Misc.ScanSpace(input,walk);
                } else {
                    m_source = "";
                }
                
                {
                    int delta = Misc.ScanWord(input,walk);
                    m_event = input.Substring(walk,delta);
                    walk += delta;
                    walk += Misc.ScanSpace(input,walk);
                }

                {
                    int count = 0;
                    int dataBase = walk;
                    for(;;) {
                        int delta = Misc.ScanWord(input,walk);
                        if (delta == 0) break;
                        count++;
                        if (input[walk] == ':') {
                            walk = input.Length;
                            break;
                        } else {
                            walk += delta;
                            walk += Misc.ScanSpace(input,walk);
                        }
                    }

                    m_data = new string[count];

                    count = 0;
                    walk = dataBase;
                    for(;;) {
                        int delta = Misc.ScanWord(input,walk);
                        if (delta == 0) break;
                        if (input[walk] == ':') {
                            m_data[count++] = input.Substring(walk+1);
                            walk = input.Length;
                            break;
                        } else {
                            m_data[count++] = input.Substring(walk,delta);
                            walk += delta;
                            walk += Misc.ScanSpace(input,walk);
                        }
                    }

                    if (count != Data.Length) throw new ArgumentException();
                }
            }
        }

        public string OptionalParam(int index) {
            if (index < m_data.Length) return m_data[index];
            else return null;
        }

        public bool IsThreeDigitCode {
            get { return Misc.IsThreeDigitCode(What); }
        }

        public string Source {
            get {return m_source;}
        }
        public string What {
            get {return m_event;}
        }
        public ArrayWrapper<string> Data {
            get {return new ArrayWrapper<string>(m_data);}
        }
        public string Raw {
            get {return m_raw;}
        }
            

        public bool Bounced {
            get {return m_bounced;}
        }
        
        public string StatusMessageEx(string separator) {
            string total = Data[1];
            for(int walk = 2; walk < Data.Length; ++walk) total += separator + Data[walk];
            return total;
        }

        public string StatusMessage {
            get {
                return StatusMessageEx(" ");
            }
        }

        public bool StatusMessageAvailable {
            get {
                return Data.Length > 1;
            }
        }
        public bool IsMsg {
            get {
                return m_event == "PRIVMSG" || m_event == "NOTICE";
            }
        }
        public bool IsMsgFrom( string nick ) {
            if (! IsMsg ) return false;
            return Nicks.Equals(nick, SourceNick);
        }
        public string SourceNick {
            get {
                return Nicks.Extract( m_source );
            }
        }
        string m_source, m_event;
        string[] m_data;
        bool m_bounced;
        string m_raw;
    };

    public class IRCCommand {
        public static void ValidateString(string str) {
            if (str.IndexOfAny(new char[]{'\r','\n'}) >= 0) {
                throw new InvalidIRCCommand();
            }
        }
        void Validate() {
            ValidateString(command);
            if (parameters != null) {
                foreach(string str in parameters) ValidateString(str);
            }
        }

        public bool IsMsg {
            get {
                return command == "PRIVMSG" || command == "NOTICE";
            }
        }
        // Valid for objects created with MakeAuthCommand()
        public string MsgTargetNick {
            get { 
                if (!IsMsg) return null;
                return parameters[0]; 
            }
        }
        public static IRCCommand MakeAuthCommand(string strUser) {
            string[] cmdParams = strUser.Split(new char[] { ' ' }, 3, StringSplitOptions.RemoveEmptyEntries);
            if (cmdParams.Length != 3) throw new ArgumentException();
            string cmd;
            string userCmd = cmdParams[0].ToUpperInvariant();
            if (userCmd == "/MSG" || userCmd == "/PRIVMSG") {
                cmd = "PRIVMSG";
            } else if (userCmd == "/NOTICE") {
                cmd = "NOTICE";
            } else throw new ArgumentException();
            string targetNick = cmdParams[1];
            if (!IRCUtils.Nicks.IsValid(targetNick)) throw new ArgumentException();
            return new IRCCommand(cmd, new string[] { targetNick, cmdParams[2] });
        }

        public IRCCommand(string _command,string _parameter) : this(_command,new string[]{_parameter}) {}
        
        public IRCCommand(string _command) : this(_command, new string[0]) {}
        public IRCCommand(string _command, string[] _parameters) {
            command = _command; parameters = _parameters;
            Validate();
        }

        public IRCCommand(string[] input) {
            if (input.Length < 1) throw new InvalidIRCCommand();
            command = input[0];
            if (input.Length > 1) {
                int total = input.Length - 1;
                parameters = new string[total];
                for(int walk = 0; walk < total; ++walk) parameters[walk] = input[walk+1];
            }
            Validate();
        }

        public string MakeString() {
            string output = command;
            if (parameters != null) {
                for(int walk = 0; walk < parameters.Length; ++walk) {
                    string param = parameters[walk];
                    if (param.IndexOf(' ') >= 0 || (param.Length > 0 && param[0] == ':')) {
                        if (walk + 1 == parameters.Length) {
                            output += " :" + param;
                        } else {
                            throw new InvalidIRCCommand();
                        }
                    } else {
                        output += " " + param;
                    }
                }
            }
            return output;
        }
        public bool Floodable {
            get {
                string commandEx = command.ToUpperInvariant();
                return commandEx == "PRIVMSG" || commandEx == "NOTICE" || commandEx == "MODE";
            }
        }
        // Indicates whether the server should be shortly replying with something to this command, used for stealth timeout detection
        public bool ExpectsServerReply {
            get {
                string commandEx = command.ToUpperInvariant();
                // Incomplete list but covers commonly fired ones
                return commandEx == "WHOIS" || commandEx == "WHO" || commandEx == "AWAY" || commandEx == "NICK" || commandEx == "JOIN" || command == "MODE";
            }
        }


        public string command;
        public string[] parameters;
        public bool quiet;
    };

    public class InvalidIRCCommand : Exception {
        public InvalidIRCCommand() : base("Invalid IRC Command") {}
        public InvalidIRCCommand(string what) : base("Invalid IRC Command: " + what) {}
    };


    public class AwayMessageFilter {
        public AwayMessageFilter() {
            m_data = new Dictionary<string,Entry>();
        }

        public bool Filter(string nick, string reason) {
            Entry entry;
            DateTime time = DateTime.UtcNow;
            TimeSpan filterDelay = TimeSpan.FromMinutes(20);
            if (m_data.TryGetValue(nick,out entry)) {
                bool retVal = reason != entry.m_reason || time - entry.m_timestamp > filterDelay;
                entry.m_reason = reason;
                entry.m_timestamp = time;
                return retVal;
            } else {
                m_data.Add(nick,new Entry(time,reason));
                return true;
            }
        }

        public void Flush(string nick) {
            m_data.Remove(nick);
        }


        class Entry {
            public Entry(DateTime p_timestamp, string p_reason) {m_timestamp = p_timestamp; m_reason = p_reason;}
            public DateTime m_timestamp;
            public string m_reason;
        }
        Dictionary<string, Entry> m_data;
    }

    public class IRCObjectList : IEnumerable<string> {
        
        IEnumerator IEnumerable.GetEnumerator() {return m_content.Keys.GetEnumerator();}
        IEnumerator<string> IEnumerable<string>.GetEnumerator() {return m_content.Keys.GetEnumerator();}

        public int Count { get { return m_content.Count; } }

        public IRCObjectList(IEnumerable<string> source) : this() {Add(source);}
        public IRCObjectList() {
            m_content = new Dictionary<string,object>(Nicks.Comparer);
        }
        public void Clear() {
            m_content.Clear();
        }
        public bool Contains(string name) {
            return m_content.ContainsKey(name);
        }
        public void Add(string name) {
            if (!m_content.ContainsKey(name)) m_content.Add(name,null);
        }
        public void Add(IEnumerable<string> source) {
            foreach(string entry in source) Add(entry);
        }
        public bool Remove(string name) {
            return m_content.Remove(name);
        }

        Dictionary<string,object> m_content;
    }

    public class PrefixMap {

        public static PrefixMap Default {
            get { return new PrefixMap("ov","@+"); }
        }

        public string ResortPrefixes(string input) {
            string output = "";
            foreach(char c in prefixes) {
                if (input.IndexOf(c) >= 0) output += c.ToString();
            }
            return output;
        }

        public static bool TryParse(string entry, out PrefixMap map) {
            map = null;
            if (entry.Length < 2 || entry.Length % 2 != 0) return false;
            int count = (entry.Length / 2) - 1;
            if (entry[0] != '(' || entry[1 + count] != ')') return false;
            string modes, prefixes;
            modes = entry.Substring(1,count);
            prefixes = entry.Substring(2+count,count);
            char[] forbidden = new char[]{'(',')',' '};//completeme?
            if (modes.IndexOfAny(forbidden) >= 0 || prefixes.IndexOfAny(forbidden) >= 0) return false;
            map = new PrefixMap(modes,prefixes);
            return true;
        }

        public bool ModeToPrefix(char mode, out char prefix) {
            int index = modes.IndexOf(mode);
            if (index < 0) {prefix = new char();return false;}
            prefix = prefixes[index];
            return true;
        }
        public bool PrefixToMode(char prefix, out char mode) {
            int index = prefixes.IndexOf(prefix);
            if (index < 0) {mode = new char();return false;}
            mode = modes[index];
            return true;
        }

        public bool IsKnownMode(char mode) {
            return modes.IndexOf(mode) >= 0;
        }

        PrefixMap(string _modes, string _prefixes) {
            if (_modes.Length != _prefixes.Length) throw new ArgumentException();
            modes = _modes; prefixes = _prefixes;
        }

        public string PrefixList { get { return prefixes;} }

        public string SimplifyPrefix( string prefix )
        {
            if (prefix.Length > 1) {
                foreach(char c in PrefixList ) {
                    if (prefix.IndexOf(c) >= 0) return c.ToString();
                }
                return prefix[0].ToString();
            } else {
                return prefix;
            }
        }


        string modes, prefixes;
    }

    public struct ServerModeSpecs {
        public PrefixMap Prefixes;
        public string[] ChanModes;

        public static ServerModeSpecs Defaults {
            get {
                ServerModeSpecs val;
                //freenode: CHANMODES=bdeIq,k,lfJD,cgijLmnPQrRstz
                val.ChanModes = new string[]{"b","k","l","inst"};
                val.Prefixes = PrefixMap.Default;
                return val;
            }
        }
    }

    public class TransferSpeedEstimator {
        public void Reset() {
            Entries.Clear();
        }
        public TransferSpeedEstimator() {
            Entries = new LinkedList<Entry>();
        }
        public void OnProgress(UInt64 newPosition) {
            DateTime currentTime = DateTime.UtcNow;
            DateTime cutoff = currentTime - TimeSpan.FromSeconds(10);
            while(Entries.Count > 2 && Entries.First.Value.Timestamp < cutoff) Entries.RemoveFirst();
            Entries.AddLast(new Entry(currentTime,newPosition));
        }
        public double Speed {
            get {
                if (Entries.Count < 2) return 0;
                double deltaTime = (Entries.Last.Value.Timestamp - Entries.First.Value.Timestamp).TotalSeconds;
                if (deltaTime > 0 && Entries.Last.Value.Position > Entries.First.Value.Position) {
                    return (double)(Entries.Last.Value.Position - Entries.First.Value.Position) / deltaTime;
                } else {
                    return 0;
                }
            }
        }

        public static string FormatSpeed(UInt64 deltaPosition, TimeSpan deltaTime) {
            double speed = 0;
            double deltaSeconds = deltaTime.TotalSeconds;
            if (deltaPosition > 0 && deltaSeconds > 0) {
                speed = (double)deltaPosition / deltaSeconds;
            }
            return FormatSpeed(speed);
        }

        public static string FormatSpeed(double speed) {
            return (speed / 1024).ToString("F") + "kB/s";
        }
        public string SpeedFormatted {
            get {
                return FormatSpeed(Speed);
            }
        }

        public static string FormatTime(UInt64 seconds) {
            return (seconds / 3600).ToString() + ":" + ((seconds/60)%60).ToString("D2") + ":" + (seconds % 60).ToString("D2");
        }

        public string EstimateRemaining(UInt64 TotalSize) {
            double curSpeed = Speed;
            if (curSpeed <= 0 || Entries.Count < 1) return "N/A";
            UInt64 remaining = TotalSize - Entries.Last.Value.Position;

            UInt64 seconds = (UInt64) Math.Round((double)remaining / curSpeed);
            return FormatTime(seconds);
        }

        public UInt64 BytesDone {
            get {
                if (Entries.Count == 0) return 0;
                return Entries.Last.Value.Position;
            }
        }

        public string FormatReport(UInt64 TotalSize) {
            return FormatReport(TotalSize,0);
        }

        public UInt32 PercentDone(UInt64 totalSize, UInt64 resumeOffset) {
            if (totalSize == UInt64.MaxValue) return UInt32.MaxValue;
            return checked((UInt32) ((BytesDone + resumeOffset) * 100 / totalSize) );
        }

        public string FormatReport(UInt64 TotalSize,UInt64 resumeOffset) {
            string message = BytesDone.ToString() + " bytes done";
            bool TotalSizeValid = TotalSize != UInt64.MaxValue && TotalSize > resumeOffset;
            if (TotalSizeValid) {
                message += " (" + PercentDone(TotalSize,resumeOffset) + "%)";
            }
            message += ", " + SpeedFormatted;
            if (TotalSizeValid) message += ", " + EstimateRemaining(TotalSize - resumeOffset) + " remaining";
            return message;
        }

        public static string FormatFinishedMessage(UInt64 sizeTransferred, TimeSpan timeElapsed) {
            return "Finished Successfully - " + /*TransferSpeedEstimator.*/FormatSpeed(sizeTransferred,timeElapsed) + " average";
        }

        struct Entry {
            public Entry(DateTime _Timestamp, UInt64 _Position) {Timestamp = _Timestamp; Position = _Position;}
            public DateTime Timestamp;
            public UInt64 Position;
        }

        LinkedList<Entry> Entries;
    }

    public struct BanDescription {
        public string Mask { get; set; }
        public string ByWhom { get; set; }
        public DateTime Date { get; set; }

        public string Added {
            get {
                string output = Nicks.Extract(this.ByWhom);
                string userHost = Nicks.ExtractUserHost(this.ByWhom);
                if (userHost.Length > 0) output += " (" + userHost + ")";
                if (this.Date != DateTime.MinValue) {
                    output += " on " + this.Date.ToLocalTime().ToString();
                }
                return output;
            }
        }
    };

    public class ServerErrorHelper {
        public ServerErrorHelper() {
            Backlog = new LinkedList<DateTime>();
        }

        public void OnError() {
            Backlog.AddLast(DateTime.UtcNow);
            while(Backlog.Count > 50) Backlog.RemoveFirst();
        }
        public bool ShouldAbort {
            get {
                if (Backlog.Count < 50) return false;
                return (Backlog.Last.Value - Backlog.First.Value) < TimeSpan.FromSeconds(60);
            }
        }
    
        LinkedList<DateTime> Backlog;
    }

    public class Time {
        public static DateTime IRCtoUTC(string value) {
            return IRCtoUTC( Convert.ToInt64(value) );
        }
        public static DateTime IRCtoUTC(Int64 value) {
            return (new DateTime(1970,1,1) + TimeSpan.FromSeconds(value));
        }
    }

    public class LineHistory {
        public LineHistory() {
            m_backLog = new LinkedList<string>();
            m_backLog.AddLast("");
        }
        public string Previous() {
            if (m_walk != null && m_walk.Previous != null) {
                m_walk = m_walk.Previous;
                return m_walk.Value;
            }
            return null;
        }
        public string Next() {
            if (m_walk != null && m_walk.Next != null) {
                m_walk = m_walk.Next;
                return m_walk.Value;
            }
            return null;
        }
        public void Add(string line) {
            if (line.Length > 0) {
                if (m_backLog.Count > 0) {
                    if (m_backLog.Last.Value == "") m_backLog.RemoveLast();
                }
                m_backLog.Remove(line);
                m_backLog.AddLast(line);
                m_walk = m_backLog.AddLast("");

                while(m_backLog.Count > 100) m_backLog.RemoveFirst();
            }
        }

        LinkedList<string> m_backLog;
        LinkedListNode<string> m_walk;
    }


    public class AliasList {
       
        // returns either a string or a ready-to-fire IRCCommand
        public object EvalAlias(string input, string context) {
            string inCmd = "", inParam = "";
            if (SplitCommandParam(input, ref inCmd, ref inParam)) {
                foreach(Alias v in m_aliases) {
                    object s = v.Eval(inCmd, inParam, context); if (s != null) return s;
                }
            }
            return null;
        }
        public AliasList(string config) {
            string[] lines = config.Split(new char[]{'\r','\n'},StringSplitOptions.RemoveEmptyEntries);
            m_aliases = new LinkedList<Alias> ();
            foreach(string l in lines) try { m_aliases.AddLast( new Alias(l) ); } catch(ParameterException) {}
        }

        private class ParameterException : Exception {
            public ParameterException() : base("Invalid parameters") {}
        };

        public static bool SplitCommandParam(string val, ref string command, ref string param) {
            string[] s = val.Split(new char[]{' '}, 2, StringSplitOptions.RemoveEmptyEntries);
            if (s.Length == 0) return false;
            command = s[0];
            if (s.Length > 1) param = s[1]; else param = "";
            return true;
        }

        private class Alias {
            static bool ParseParamRef(string str, ref int val) {
                if (str.Length == 0) return false;
                if (str[0] != '$') return false;
                int v;
                try {
                    v = Convert.ToInt32(str.Substring(1));
                } catch {return false;}
                if (v < 1) return false;
                val = v;
                return true;
            }
            public Alias(string line) {
                string subst1 = "", subst = "";
                if (!SplitCommandParam(line, ref m_commandIn, ref subst1)) throw new ParameterException();
                if (!SplitCommandParam(subst1, ref m_commandOut, ref subst)) throw new ParameterException();

                m_subst = FixEmptyArray(subst.Split(new char[]{' '}, StringSplitOptions.RemoveEmptyEntries));
                m_maxParam = 0;
                m_haveEllipsis = false; m_haveContext = false;
                for(int walk = 0; walk < m_subst.Length; ++walk) {
                    string t = m_subst[walk];
                    int v = 0;
                    if (t == "...") {
                        m_haveEllipsis = true;
                    } else if (t == "$c") {
                        m_haveContext = true;
                    } else if (ParseParamRef(t, ref v)) {
                        if (m_maxParam < v) m_maxParam = v;
                    }
                }
            }
            static string[] FixEmptyArray(string[] p) {
                if (p.Length == 1 && p[0] == "") return new string[0];
                return p;
            }
            public object Eval(string inCmd, string inParam, string context) {
                if (inCmd != m_commandIn) return null;
                if (m_haveContext && context == null) return null;
                string[] p;
                string ellipsisContent = "";
                if (m_haveEllipsis) {
                    p = FixEmptyArray(inParam.Split(new char[]{' '}, m_maxParam + 1));
                    if (p.Length <= m_maxParam) return null;
                    ellipsisContent = p[m_maxParam];
                } else {
                    p = FixEmptyArray(inParam.Split(new char[]{' '}, StringSplitOptions.RemoveEmptyEntries));
                    if (p.Length != m_maxParam) return null;
                }
                string[] outParams = new string[m_subst.Length];
                for(int walk = 0; walk < m_subst.Length; ++walk) {
                    string i = m_subst[walk];
                    string v;
                    if (i == "...") v = ellipsisContent;
                    else if (i == "$c") v = context;
                    else {
                        int idx = 0;
                        if (ParseParamRef(i, ref idx)) {
                            v = p[idx-1];
                        } else {
                            v = i;
                        }
                    }
                    outParams[walk] = v;
                }
                if (m_commandOut[0] == '/') {
                    return m_commandOut + " " + String.Join(" ", outParams);
                } else {
                    return new IRCCommand(m_commandOut, outParams);
                }
            }
            string m_commandIn, m_commandOut;
            string[] m_subst;
            int m_maxParam;
            bool m_haveEllipsis;
            bool m_haveContext;
        };

        LinkedList<Alias> m_aliases;
    };

    class Formatting
    {
        public static string formatDCCIP( string ipv4 ) {
            var components = ipv4.Split('.');
            if ( components.Length != 4 ) throw new ArgumentException();
            UInt32 final = 0;
            for ( int i = 0; i < 4; ++ i ) {
                final = final * 256 + UInt32.Parse( components[i] );
            }
            return final.ToString();
        }
        public static TextLine formatNickChange(string source, string newNick) {
            TextStyle style = TextStyle.Gray;
            return formatMiscAction(source, TextLine.Simple("is now known as ",style) + TextLine.UserNameEx(newNick,style),null);
        }
        public static TextLine formatMiscAction(string source,string content,string reason,TextStyle style) {
            return formatMiscAction(source,TextLine.FromIRC(content,style),reason,style);
        }
        public static TextLine formatMiscAction(string source,string content,string reason) {
            return formatMiscAction(source,content,reason,TextStyle.Gray);
        }
        public static TextLine formatMiscAction(string source,TextLine content,string reason) {
            return formatMiscAction(source,content,reason,TextStyle.Gray);
        }
        public static TextLine formatMiscAction(string source,TextLine content,string reason,TextStyle style) {
            TextStyle styleHighlight = TextStyle.NickEx(style);
            string sourceNick = Nicks.Extract(source);
            TextLine sourceEx = TextLine.Simple(Nicks.Extract(source),styleHighlight) /* TextLine.UserName(sourceNick,IsOwnNick(sourceNick)) */;
            string hostMask = Nicks.ExtractUserHost(source);
            if (hostMask != "") sourceEx += TextLine.Simple(" (",style) + TextLine.Simple(hostMask,styleHighlight) + TextLine.Simple( ")",style);
            TextLine message = TextLine.Simple("* ",style) + sourceEx + TextLine.Simple(" ",style) + content;
            if (reason != null) {
                if (reason.Length > 0) {
                    var styleNoMerge = style;
                    styleNoMerge.DoNotMerge = true;
                    message = message + TextLine.Simple(" (",style) + TextLine.FromIRC(reason, styleNoMerge)  + TextLine.Simple(")",style);
                }
            }            
            return message;
        }


        public static TextLine formatPrintMode(IRCEvent ev, ServerConnection OwningServer) {
            TextStyle style = TextStyle.Gray;
            TextStyle styleNick = TextStyle.NickEx(style);
            TextLine content = new TextLine();
            bool fallback = false;

            if (Nicks.IsChannel(Modes.ModeTarget(ev),OwningServer.InstanceData.ChanTypes)) {
                try {
                    TextLine paramsSuffix = new TextLine();
                    string modes = "";
                    bool currentSign = false;
                    bool currentSignSet = false;
                    foreach(Modes.Mode mode in Modes.Parse(ev,OwningServer.InstanceData.ModeSpecs)) {
                        if (!currentSignSet || currentSign != mode.sign) {
                            modes += mode.sign ? "+" : "-";
                            currentSign = mode.sign; currentSignSet = true;
                        }
                        modes += mode.action;
                        if (mode.param != null) {
                            paramsSuffix += TextLine.Simple(" ",style) + TextLine.Simple(mode.param,mode.paramIsNick ? styleNick : style);
                        }
                    }
                    content = TextLine.Simple(modes,style) + paramsSuffix;
                } catch {
                    fallback = true;
                }
            } else {
                fallback = true;
            }
            if (fallback) {
                string msg = "";
                for(int walk = 1; walk < ev.Data.Length; ++walk) {
                    if (walk > 1) msg += " ";
                    msg += ev.Data[walk];
                }
                content = TextLine.Simple(msg,style);
            }
            
            return TextLine.Simple("Mode: [",style) + content + TextLine.Simple("] by ",style) + TextLine.Simple(Nicks.Extract(ev.Source),styleNick);
        }
    }

};
