﻿using System;
using IRCUtils;
using System.Threading.Tasks;
using System.Collections.Generic;

namespace monochrome {

    namespace IRCScripts {
        class Auth {
            public abstract class Handler {
                public Handler(IScriptContext ctx) { m_ctx = ctx; }

                public abstract Task<bool> KillGhost(string nick, string password);
                public abstract Task<bool> ReleaseNick(string nick, string password);
                public abstract Task Identify(string account, string password);
                public abstract Task<bool> InviteSelf(string channel);

                protected IScriptContext m_ctx;
            };
            class HandlerNickServ : Handler {
                public HandlerNickServ(IScriptContext ctx) : base(ctx) {}

                public override async Task<bool> KillGhost(string nick, string password) {
                    using (var events = m_ctx.MakeEventStream()) {
                        m_ctx.Command(new IRCCommand("NICKSERV", new string[] { "GHOST", nick, password }));
                        await events.WaitFor(new EventFilter((new ServicesCommands.NickservReplies()).EventFilter_GhostResponse));
                        return true;
                    }
                }
                public override async Task<bool> ReleaseNick(string nick, string password) {
                    using (var events = m_ctx.MakeEventStream()) {
                        m_ctx.Command(new IRCCommand("NICKSERV", new string[] { "RELEASE", nick, password }));
                        await events.WaitFor(new EventFilter((new ServicesCommands.NickservReplies()).EventFilter_GhostResponse));
                        return true;
                    }
                }
                public override async Task Identify(string account, string password) {
                    using (var events = m_ctx.MakeEventStream()) {
                        string[] args;
                        if ( account != "" ) {
                            args = new string[] { "IDENTIFY", account, password };
                        } else {
                            args = new string[] { "IDENTIFY", password };
                        }
                        m_ctx.Command(new IRCCommand("NICKSERV", args ));
                        await events.WaitFor(new EventFilter((new ServicesCommands.NickservReplies()).EventFilter_IdentifyResponse));
                    }
                }
                public override async Task<bool> InviteSelf(string channel)
                {
                    using (var events = m_ctx.MakeEventStream()) {
                        m_ctx.Command(new IRCCommand("CHANSERV", new string[] { "INVITE", channel }));
                        return await ChannelJoinCommands.WaitForInviteReply(events, channel, "ChanServ");
                    }
                }
            };
            class HandlerQuakeNet : Handler {
                public HandlerQuakeNet(IScriptContext ctx) : base(ctx) {}

                public override async Task Identify(string account, string password)
                {
                    if ( account == "" ) account = m_ctx.GetOwnNick();
                    using (var events = m_ctx.MakeEventStream()) {
                        QCmd("AUTH " + account + " " + password);
                        await events.WaitFor(new EventFilter((new ServicesCommands.QReplies()).EventFilter_IdentifyResponse));
                    }
                }
                public override Task<bool> KillGhost(string nick, string password) {
                    return Task.FromResult(false);
                }
                public override Task<bool> ReleaseNick(string nick, string password)
                {
                    return Task.FromResult(false);
                }
                public override Task<bool> InviteSelf(string channel)
                {
                    QCmd("INVITE " + channel);
                    return Task.FromResult(false);
                }

                void QCmd(string param) {
                    IRCCommand cmd = Commands.Message("Q@CServe.quakenet.org", param);
                    cmd.quiet = true;
                    m_ctx.CommandFloodable( cmd );
                }
            };
            static bool MatchNetworkName(string host, string suffix) {
                return host.EndsWith(suffix, StringComparison.OrdinalIgnoreCase);
            }
            public static Handler CreateHandler(IScriptContext ctx) {
                string host = ctx.ServerHostName();
                if (MatchNetworkName(host, ".quakenet.org")) {
                    return new HandlerQuakeNet(ctx);
                } else {
                    return new HandlerNickServ(ctx);
                }
            }

            public class SASL { 
                public static async Task Run(IScriptContext ctx, string nick, string password)
                {
                    using( var events = ctx.MakeEventStream())
                    {
                        var timeout = TimeSpan.FromSeconds(20);
                        ctx.Command(new IRCCommand(new string[] {"AUTHENTICATE", "PLAIN" }));
                        IRCEvent evt = await events.WaitForReply("AUTHENTICATE", timeout);
                        if ( evt == null ) throw new Exception("No AUTHENTICATE response");
                        if ( evt.Data.Length != 1 || evt.Data[0] != "+" ) new Exception("Unexpected AUTHENTICATE response");


                        {
                            List<byte> list = new List<byte>();

                            list.AddRange(System.Text.Encoding.UTF8.GetBytes(nick));
                            list.Add(0);
                            list.AddRange(System.Text.Encoding.UTF8.GetBytes(nick));
                            list.Add(0);
                            list.AddRange(System.Text.Encoding.UTF8.GetBytes(password));

                            string send = System.Convert.ToBase64String( list.ToArray() );
                            ctx.Command(new IRCCommand( "AUTHENTICATE", send));
                        }
                        
                        evt = await events.WaitForReply("900", timeout);
                        if ( evt == null ) throw new Exception("SASL authentication failure");
                        evt = await events.WaitForReply("903", timeout);
                        if (evt == null) throw new Exception("SASL authentication failure");
                        ctx.Command( new IRCCommand( "CAP END" ));
                    }
                }
            };

        };



        class ServicesCommands {
            static bool IsServiceMsg(IRCEvent ev,string service) {
                return (ev.What == "PRIVMSG" || ev.What == "NOTICE") && Nicks.Equals(Nicks.Extract(ev.Source),service);
            }
            public class ServicesCommandOK {};
            public class ServicesCommandFailure {};

            public static object BotInviteFilter(string chan, IRCEvent ev, string inviter) {
                if (ev.What == "401" && ev.Data.Length >= 2 && Nicks.Equals(ev.Data[1], inviter)) {
                    return new ServicesCommandFailure();
                } else if (ev.What == "INVITE" && Nicks.Equals(ev.Data[1],chan)) {
                    return new ServicesCommandOK();
                } else if (IsServiceMsg(ev,inviter)) {
                    string[] words = Lingustics.StringToWords(ev.Data[1]);
                    if (Lingustics.WordsPresentAny(words,new string[]{"identify","insufficient","register"}) || Lingustics.WordsPresent(words,new string[]{"access","level","required"})) return new ServicesCommandFailure();
                    return null;
                } else {
                    return null;
                }
            }

            public class EventFilter_BotInvite {
                public EventFilter_BotInvite(string _chan, string _inviter) {chan = _chan; inviter = _inviter; }
                public object Proc(IRCEvent ev) {
                    return BotInviteFilter(chan,ev,inviter);
                }

                string chan, inviter;
            };


            public class Replies {
                public Replies(string source) {m_source = source;}

                public object EventFilter_IdentifyResponse(IRCEvent ev) {
                    if (IsServiceMsg(ev,m_source)) {
                        string content = ev.Data[1];
                        if (Lingustics.IsMsgPasswordIncorrect(content)) throw new PasswordIncorrect();
                        else if (Lingustics.IsMsgNickNotRegistered(content)) throw new NickNotRegistered();
                        else if (Lingustics.IsMsgSuccessfullyIdentified(content)) return new ServicesCommandOK();
                        else return null;
                    } else {
                        return null;
                    }
                }
                public object EventFilter_GhostResponse(IRCEvent ev) {
                    if (IsServiceMsg(ev,m_source)) {
                        string content = ev.Data[1];
                        if (Lingustics.IsMsgPasswordIncorrect(content)) throw new PasswordIncorrect();
                        else if (Lingustics.IsMsgNickNotRegistered(content)) throw new NickNotRegistered();
                        return new ServicesCommandOK();
                    } else {
                        return null;
                    }
                }

                string m_source;
            };
            public class NickservReplies : Replies {
                public NickservReplies() : base("NICKSERV") {}
            };
            public class QReplies : Replies {
                public QReplies() : base("Q") {}
            };
        }
    }
}
