using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using IRCUtils;
using System.Threading;
using monochrome;

namespace monochrome {
    public delegate object EventFilter(IRCEvent e);

    class EventFilters {
        public static object Dummy(IRCEvent e) { return null; }
    };

    class EventStream : IDisposable {
        public EventStream() : this(CancellationToken.None) {
        }
        public EventStream( CancellationToken aborter ) {
            m_aborter = aborter;
            m_semaphore = new SemaphoreSlim(0);
            m_events = new LinkedList<IRCEvent>();
        }
        public static TimeSpan ReplyTimeOut = TimeSpan.FromSeconds(20);
        public static TimeSpan ReplyTimeOutEx(int commandCount) {
            return TimeSpan.FromTicks(ReplyTimeOut.Ticks * (1 + commandCount / 2));
        }

        public async Task<IRCEvent> WaitForReply( String what, TimeSpan timeout )
        {
            Func<IRCEvent, object> f = (IRCEvent evt) => { return evt.What == what ? evt : null; };
            return await WaitFor(f, timeout) as IRCEvent;
        }

        public async Task<object> WaitFor(EventFilter filter, TimeSpan timeout) {
            Func<IRCEvent, object> f = (IRCEvent ev) => filter(ev);
            return await WaitFor(f, timeout);
        }
        public async Task<object> WaitFor(EventFilter filter) {
            return await WaitFor(filter, ReplyTimeOut);
        }
        public async Task<object> WaitFor(Func<IRCEvent, object> filter) {
            return await WaitFor(filter, ReplyTimeOut);
        }
        public async Task<object> WaitFor(Func<IRCEvent, object> filter, TimeSpan timeout) {
            DateTime entry = DateTime.Now;
            for (;;) {
                TimeSpan slept = DateTime.Now - entry;
                if ( slept >= timeout ) throw new TimeoutException();
                if (await m_semaphore.WaitAsync(timeout - slept, m_aborter)) {
                    if (m_events.Count == 0) throw new IRCScripts.ScriptAborted();
                    IRCEvent e = m_events.First.Value;
                    m_events.RemoveFirst();
                    var state = filter(e);
                    if ( state != null ) return state;
                }
            }
        }

        public void Add(IRCEvent evt) {
            m_events.AddLast( evt );
            m_semaphore.Release();
        }

        public void Cancel() {
            m_events.Clear();
            m_semaphore.Release();
        }

        protected virtual void Disposing() {
            m_semaphore.Dispose();
        }
        void IDisposable.Dispose() {
            Disposing();
        }

        private LinkedList<IRCEvent> m_events;
        private SemaphoreSlim m_semaphore;
        private CancellationToken m_aborter;
    }


    namespace IRCScripts {

        interface IScriptContext {
            
            CancellationToken GetAborter();

            void Command(IRCCommand cmd);
            void CommandFloodable(IRCCommand cmd);

            EventStream MakeEventStream();

            void Done();
            void DoneQuiet();
            void Failure(Exception e);
            string GetOwnNick();
            void PrintOutput(string message);
            string ServerHostName();
        };

        class ScriptAborted : Exception {
            public ScriptAborted() : base("Script Aborted") { }
            public ScriptAborted(string msg) : base(msg) { }
        };

        class CantAcquireNick : Exception {
            public CantAcquireNick() : base("Can't Acquire Nick") { }
            public CantAcquireNick(string msg) : base(msg) { }
        };
        class PasswordIncorrect : Exception {
            public PasswordIncorrect() : base("Password Incorrect") { }
            public PasswordIncorrect(string msg) : base(msg) { }
        };
        class NickNotRegistered : Exception {
            public NickNotRegistered() : base("Nick Not Registered") { }
            public NickNotRegistered(string msg) : base(msg) { }
        };



        class ServerStartupScript {
            public struct Param {
                public string[] NickList;
                public ChannelJoinEntry[] ChanList;
                public string NickPassword, NickAccount;
                public string UserModes;
                public bool AllowRandomNick;
                public Dictionary<string, string> InviteHandlers;
                public string[] ExtraStartupCommands;
                public bool ShouldAuthorize;
            };


            public ServerStartupScript(IScriptContext _context, Param param, IRCObjectList _droppedChannels) {
                droppedChannels = new IRCObjectList(_droppedChannels);
                context = _context;
                m_param = param;
                m_authHandler = Auth.CreateHandler(context);

                _ = Worker();
            }



            class NickAcquired { };
            class NickInUse { };
            class Host_Successful { };
            class NickTemporarilyUnavailable : NickInUse { };

            object EventFilter_NickChangeResponse(IRCEvent ev) {
                if (ev.What == "NICK") {
                    if (!acceptableNick(ev.Data[0])) throw new InvalidIRCEvent("Error acquiring nickname");
                    return new NickAcquired();
                } else if (ev.What == "433") {
                    return new NickInUse();
                } else if (IRCUtils.Events.IsNickTemporarilyUnavailable(ev)) {
                    return new NickTemporarilyUnavailable();
                } else {
                    return null;
                }
            }



            bool acceptableNick(string nick) {
                foreach (string walk in m_param.NickList) {
                    if (Nicks.Equals(walk, nick)) return true;
                }
                return false;
            }

            async Task<object> AcquireNick() {
                if (acceptableNick(context.GetOwnNick())) return new NickAcquired();
                using (var events = context.MakeEventStream()) {
                    object firstResponse = null;
                    bool firstNick = true;
                    foreach (string nick in m_param.NickList) {
                        context.Command(new IRCCommand("NICK", nick));
                        object response = await events.WaitFor(new EventFilter(EventFilter_NickChangeResponse));
                        if (response is NickAcquired) return response;
                        if (firstNick) {
                            firstResponse = response;
                            firstNick = false;
                        }
                    }
                    return firstResponse;
                }
            }

            async Task<bool> KillGhost(string nick) {
                return await m_authHandler.KillGhost(nick, m_param.NickPassword);
            }
            async Task<bool> KillGhost() {
                return await KillGhost(m_param.NickList[0]);
            }
            async Task<bool> ReleaseNick(string nick) {
                return await m_authHandler.ReleaseNick(nick, m_param.NickPassword);
            }
            async Task<bool> ReleaseNick() {
                return await ReleaseNick(m_param.NickList[0]);
            }


            async Task Identify() {
                await m_authHandler.Identify(m_param.NickAccount, m_param.NickPassword);
            }

            bool havePassword { get { return m_param.NickPassword != ""; } }

            async Task JoinChannel(string chan, string key) {
                string inviteHandler; m_param.InviteHandlers.TryGetValue(chan, out inviteHandler);
                object state = await ChannelJoinCommands.TryJoinEx(context, chan, key, havePassword ? m_authHandler : null, inviteHandler);
                if (state is ChannelJoinCommands.JoinFailure) {
                    this.context.PrintOutput("Could not autojoin channel: " + chan + ", reason: " + state.ToString());
                }
            }

            async Task HandleNick() {
                for (;;) {
                    object status = await AcquireNick();
                    if (status is NickAcquired) break;
                    if (havePassword) {
                        if (status is NickTemporarilyUnavailable) {
                            if (await ReleaseNick()) continue;
                        } else if (status is NickInUse) {
                            if (await KillGhost()) continue;
                        }

                    }
                    if (m_param.AllowRandomNick) {
                        break;
                    } else {
                        throw new CantAcquireNick();
                    }
                }

                if (havePassword && m_param.ShouldAuthorize) {
                    await Identify();
                }
            }

            class EventFilter_MultiJoin {
                public EventFilter_MultiJoin(IRCObjectList toJoin, IRCObjectList toInvite) {
                    m_remaining = new IRCObjectList();
                    foreach (string chan in toJoin) m_remaining.Add(chan);
                    m_toInvite = toInvite;
                }

                public object Proc(IRCEvent ev) {
                    LinkedList<string> toRemove = new LinkedList<string>();
                    foreach (string chan in m_remaining) {
                        object status = ChannelJoinCommands.StatusHelper(chan, ev);
                        if (status != null) {
                            toRemove.AddLast(chan);
                            if (m_toInvite != null && status is ChannelJoinCommands.JoinFailure_InviteOnly) {
                                m_toInvite.Add(chan);
                            }
                        }
                    }
                    foreach (string chan in toRemove) m_remaining.Remove(chan);

                    return m_remaining.Count > 0 ? null : new ChannelJoinCommands.JoinSuccess();
                }
                IRCObjectList m_remaining, m_toInvite;
            }

            async Task HandleAutojoin() {
                IRCObjectList toJoin = new IRCObjectList();
                toJoin.Add(droppedChannels);
                Dictionary<string, string> channelKeys = new Dictionary<string, string>(Nicks.Comparer);
                foreach (ChannelJoinEntry entry in m_param.ChanList) {
                    toJoin.Add(entry.Name);
                    if (entry.Key != null) {
                        if (!channelKeys.ContainsKey(entry.Name)) channelKeys.Add(entry.Name, entry.Key);
                    }
                }


                foreach (string channel in toJoin) {
                    string key;
                    if (!channelKeys.TryGetValue(channel, out key)) key = null;
                    await JoinChannel(channel, key);
                }


                // Brave version attempting to join all channels simultaneously, needs thorough forking of used script context to propagate events right, currently does not work
                // Besides it could trigger joinflood protection on some nets so better don't
                // Changes needed to make this work: Fork() context used for JoinChannel(), Fork() context used for auth handler

                /*
                var tasks = new List<Task>();
                foreach(string channel in toJoin) {
                    string key;
                    if (!channelKeys.TryGetValue(channel,out key)) key = null;
                    tasks.Add( JoinChannel(channel,key) );
                }
                Exception status = null;
                foreach( Task t in tasks )
                {
                    try
                    {
                        await t;
                    } catch(Exception e)
                    {
                        if ( status == null ) status = e;
                    }
                }
                if ( status != null ) throw status;
                */
            }

            async Task HandleHost() {
                using (var events = context.MakeEventStream()) {
                    context.Command(new IRCCommand("USERHOST", context.GetOwnNick()));

                    Func<IRCEvent, object> filter = (IRCEvent ev) => (ev.What == "302") ? new Host_Successful() : null;

                    await events.WaitFor( filter );
                }
            }

            void HandleModes() {
                if (m_param.UserModes != "") {
                    context.Command(new IRCCommand("MODE", new string[] { context.GetOwnNick(), "+" + m_param.UserModes }));
                }
            }
            async Task Worker() {
                try {
                    await HandleHost();
                    await HandleNick();
                    HandleModes();
                    await HandleExtras();
                    await HandleAutojoin();
                    context.Done();
                } catch (Exception e) {
                    context.Failure(e);
                }
            }
            async Task HandleExtras()
            {
                const int defWaitMS = 500;
                string waitPrefix = "wait ";
                bool needWait = false;

                foreach(string strUser in m_param.ExtraStartupCommands)
                {
                    if ( strUser.StartsWith(waitPrefix) ) {
                        int waitLenMS = 0;
                        try {
                            waitLenMS = Convert.ToInt32( strUser.Substring( waitPrefix.Length ) );
                        } catch { }
                        if ( waitLenMS > 0 ) {
                            await Task.Delay(waitLenMS, context.GetAborter());
                        }
                        needWait = false;
                    } else {
                        if ( needWait ) {
                            await Task.Delay(defWaitMS, context.GetAborter());
                        }
                        using( var stream = context.MakeEventStream() ) {
                            IRCCommand cmd = IRCCommand.MakeAuthCommand( strUser );
                            context.CommandFloodable( cmd );
                            string bot = cmd.MsgTargetNick;
                            if ( bot != null ) {
                                Func<IRCEvent, object> f = (IRCEvent ev) => { return ev.IsMsgFrom(bot) ? "ok" : null; };
                                _ = await stream.WaitFor( f );
                            }
                        }
                        needWait = true;
                    }
                }
                if ( needWait ) {
                    await Task.Delay(defWaitMS, context.GetAborter());
                }
            }

            IScriptContext context;
            Param m_param;
            IRCObjectList droppedChannels;
            Auth.Handler m_authHandler;
        };

        class ChannelJoinScript {
            public ChannelJoinScript(IScriptContext _context, string _channel, string _key, string _inviteOverride) {
                context = _context;
                channel = _channel;
                inviteOverride = _inviteOverride;
                key = _key;
                m_auth = Auth.CreateHandler(context);


                var stfu = Worker();
            }


            async Task Worker() {
                try {
                    object state = await ChannelJoinCommands.TryJoinEx(context, channel, key, m_auth, inviteOverride);
                    if (state is ChannelJoinCommands.JoinFailure) {
                        context.PrintOutput("Could not join channel: " + channel + ", reason: " + state.ToString());
                    }
                    context.Done();
                } catch (Exception e) {
                    context.Failure(e);
                }
            }
            IScriptContext context; string channel, key, inviteOverride;
            Auth.Handler m_auth;
        }

        class GhostScript {
            public GhostScript(IScriptContext _context, string _victim, string _password) {
                m_context = _context;
                m_victim = _victim;
                m_password = _password;
                m_auth = Auth.CreateHandler(m_context);

                var stfu = Worker();
            }

            async Task Worker() {
                try {
                    if (!await m_auth.KillGhost(m_victim, m_password)) throw new Exception("GHOST command not supported on this IRC network.");
                } catch (Exception e) {
                    m_context.Failure(e);
                }
            }

            Auth.Handler m_auth;
            IScriptContext m_context;
            string m_victim, m_password;
        }

        class ChannelJoinCommands {
            public class JoinFailure {
                public override string ToString() { return "Generic Error"; }
            };
            public class JoinFailure_UnexpectedError : JoinFailure {
                public JoinFailure_UnexpectedError(string reason) { m_reason = reason; }
                public override string ToString() { return m_reason; }
                string m_reason;
            };
            public class JoinFailure_InviteOnly : JoinFailure {
                public override string ToString() { return "Channel is Invite-Only"; }
            };
            public class JoinFailure_Banned : JoinFailure {
                public override string ToString() { return "You are Banned"; }
            };
            public class JoinFailure_NeedKey : JoinFailure {
                public override string ToString() { return "Channel Requires a Key"; }
            };
            public class JoinFailure_ChannelFull : JoinFailure {
                public override string ToString() { return "Channel is Full"; }
            };
            public class JoinSuccess {
                public override string ToString() { return "Success"; }
            };
            public class JoinSuccess_Forwarded : JoinSuccess {
                public JoinSuccess_Forwarded(string target) { ForwardedTo = target; }
                public override string ToString() { return "Forwarded to Another Channel: " + ForwardedTo; }
                string ForwardedTo;
            };


            public static object StatusHelper(string chan, IRCEvent ev) {
                if (ev.IsThreeDigitCode) {
                    if (ev.Data.Length >= 2 && Nicks.Equals(ev.Data[1], chan)) {
                        switch (Convert.ToInt32(ev.What)) {
                            case 470:
                                return new JoinSuccess_Forwarded(ev.Data[2]);
                            case 471:
                                return new JoinFailure_ChannelFull();
                            case 473:
                                return new JoinFailure_InviteOnly();
                            case 474:
                                return new JoinFailure_Banned();
                            case 475:
                                return new JoinFailure_NeedKey();
                            default:
                                return null;
                        }
                    } else {
                        return null;
                    }
                } else if (ev.What == "JOIN" && Nicks.Equals(ev.Data[0], chan)) {
                    return new JoinSuccess();
                } else {
                    return null;
                }
            }

            public class EventFilter_Join {
                public EventFilter_Join(string _chan) { chan = _chan; }
                public object Proc(IRCEvent ev) {
                    return StatusHelper(chan, ev);
                }
                string chan;
            };

            public static async Task<object> TryJoin(IScriptContext context, string chan, string key) {
                using (var events = context.MakeEventStream()) {
                    IRCCommand cmd;
                    if (key != null) {
                        cmd = new IRCCommand("JOIN", new string[] { chan, key });
                    } else {
                        cmd = new IRCCommand("JOIN", chan);
                    }
                    context.Command(cmd);
                    return await events.WaitFor(new EventFilter((new EventFilter_Join(chan)).Proc));
                }
            }

            public static async Task<bool> WaitForInviteReply(EventStream stream, string channel, string inviter) {
                var status = await stream.WaitFor(new EventFilter((new ServicesCommands.EventFilter_BotInvite(channel, inviter)).Proc));
                return status is ServicesCommands.ServicesCommandOK;
            }

            public static async Task<bool> TryInviteHandler(IScriptContext context, string channel, string inviteHandler) {
                try {
                    using( var events = context.MakeEventStream() ) {
                        IRCCommand cmd = IRCCommand.MakeAuthCommand(inviteHandler);
                        context.CommandFloodable( cmd );
                        return await WaitForInviteReply(events, channel, cmd.MsgTargetNick);
                    }
                } catch (ArgumentException) {
                    context.PrintOutput("Invite handlers must be \"/MSG <nick> <message>\" or \"/NOTICE <nick> <message>\".");
                    return false;
                }
            }

            public static async Task<object> TryJoinEx(IScriptContext context, string channel, string key, Auth.Handler auth, string inviteHandler) {
                try {
                    object status = await TryJoin(context, channel, key);
                    if (status is JoinFailure_InviteOnly) {
                        bool retry = false;
                        if (inviteHandler != null) {
                            retry = await TryInviteHandler(context, channel, inviteHandler);
                        } else if (auth != null) {
                            retry = await auth.InviteSelf(channel);
                        }
                        if (retry) status = await TryJoin(context, channel, key);
                    }
                    return status;
                } catch (Exception e) {
                    return new JoinFailure_UnexpectedError(e.Message);
                }
            }
        }


        class HelloScript {
            public static void Run(IScriptContext context, string victim) {
                try {
                    foreach (string line in goatse) {
                        context.CommandFloodable(Commands.Notice(victim, line));
                    }

                } catch (Exception e) {
                    context.Failure(e);
                    return;
                }
                context.DoneQuiet();
            }


            static string[] goatse = {
"* g o a t s e x * g o a t s e x * g o a t s e x *",
"g                                               g",
"o /     \\             \\            /    \\       o",
"a|       |             \\          |      |      a",
"t|       `.             |         |       :     t",
"s`        |             |        \\|       |     s",
"e \\       | /       /  \\\\\\   --__ \\\\       :    e",
"x  \\      \\/   _--~~          ~--__| \\     |    x",
"*   \\      \\_-~                    ~-_\\    |    *",
"g    \\_     \\        _.--------.______\\|   |    g",
"o      \\     \\______// _ ___ _ (_(__>  \\   |    o",
"a       \\   .  C ___)  ______ (_(____>  |  /    a",
"t       /\\ |   C ____)/      \\ (_____>  |_/     t",
"s      / /\\|   C_____)       |  (___>   /  \\    s",
"e     |   (   _C_____)\\______/  // _/ /     \\   e",
"x     |    \\  |__   \\\\_________// (__/       |  x",
"*    | \\    \\____)   `----   --'             |  *",
"g    |  \\_          ___\\       /_          _/ | g",
"o   |              /    |     |  \\            | o",
"a   |             |    /       \\  \\           | a",
"t   |          / /    |         |  \\           |t",
"s   |         / /      \\__/\\___/    |          |s",
"e  |           /        |    |       |         |e",
"x  |          |         |    |       |         |x",
"* g o a t s e x * g o a t s e x * g o a t s e x *",
        };
        };


    };
}