﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Threading.Tasks;
using System.Threading;
using NetUtils;
using IRCUtils;

namespace monochrome
{

    public interface IDCCFileReader : IDisposable {
        Task Seek(ulong position);
        Task<uint> Read( byte[] buffer );
    }

    public interface IDCCSendDelegate
    {
        Task<IDCCFileReader> OpenFile();
        void OnFinished( Exception e );
        String StatusText { get; set; }
        String MainText { get; set; }
        void SetProgress( UInt64 bytesTransferred );
        void SetPercentDone( UInt32 percentDone );
        void ToggleRunning( bool bRunning );
    }

    public class DCCSend : IDisposable {
        public DCCSend(ServerConnection connection, DCCDescription desc, IDCCSendDelegate delegate_, UInt64 fileSize)
        {
            m_description = desc;
            m_connection = connection;
            m_resume = new ResumeStatus();
            m_abort = new CancellationTokenSource();
            m_delegate = delegate_;
            m_fileSize = fileSize;

            m_delegate.MainText = "Sending File: " + desc.FileName;
            m_delegate.StatusText = "Waiting";

            connection.DCCSendAdd( this );

            StartTransfer();

            TimerLoop();
        }
        private async void TimerLoop() {
            for ( ;; ) {
                await Task.Delay(333);
                if ( m_delegate == null ) return; // are we disposed?

                TimerTickWorker();
            }
        }
        private void TimerTickWorker()
        {
            UInt32 percentage = UInt32.MaxValue;
            if (Running) {
                UInt64 bytesDone = Progress;
                SpeedEstimator.OnProgress(bytesDone);
                UInt64 xferBase = TransferOffset;
                if (bytesDone > 0) {
                    m_delegate.StatusText = SpeedEstimator.FormatReport(m_fileSize,xferBase);
                }
                if (m_fileSize > 0 && m_fileSize != UInt64.MaxValue)
                {
                    m_delegate.SetProgress( bytesDone + xferBase );
//                    progressBar.Value = Math.Min( progressBar.Maximum, (int) ( (UInt64)progressBar.Maximum * (bytesDone + xferBase) / m_fileSize ) );
                }
                if (SpeedEstimator.BytesDone > 0) {
                    percentage = SpeedEstimator.PercentDone(m_fileSize,xferBase);
                }
            }
            m_delegate.SetPercentDone( percentage );
        }

        public void Dispose()
        {
            Abort();
            m_connection.DCCSendRemove( this );
            m_delegate = null;
        }

        async void ThreadMain(PortUsage port, IPAddress listenAddress)
        {
            var abort = m_abort.Token;
            Exception status = null;
            UInt64 done = 0;
            var started = DateTime.UtcNow;
            try
            {
                done = await Worker( port, listenAddress, abort );
            }
            catch (Exception e) { status = e; }

            if (abort.IsCancellationRequested) return;

            var elapsed = DateTime.UtcNow - started;
            OnFinished( status, done, elapsed );
        }
        async Task<UInt64> AckLoop( NetStream stream, UInt64 transferOffset, UInt64 fileSize, CancellationToken abort ) {
            byte[] temp = new byte[4];
            int got = 0;
            UInt64 bytesAcknowledged = 0;
            UInt64 readUntil = checked(fileSize - transferOffset);
            while( bytesAcknowledged < readUntil ) { 
                var delta = await stream.ReadAsync(temp, got, temp.Length-got, abort );
                if ( delta <= 0 ) break;
                got += delta;
                if ( got == 4 ) {
                    got = 0;
                    UInt64 blah = (((UInt32)temp[0]) << 24) | (((UInt32)temp[1]) << 16) | (((UInt32)temp[2]) << 8) | ((UInt32)temp[3]);
                    UInt64 last = bytesAcknowledged + transferOffset;
                    if (blah < last) {
                        blah += last & 0xFFFFFFFF00000000;
                        if (blah < last) blah += 0x100000000;
                        if (blah < last) throw new InvalidDataException();
                    }
                    bytesAcknowledged = checked(blah - transferOffset);
                }
            }
            return bytesAcknowledged;
        }

        async Task<UInt64> Worker( PortUsage port, IPAddress listenAddress, CancellationToken aborter ) {
            using (var stream = new NetStream() ) {
                using (port) {
                    await stream.Accept(listenAddress, port.Port, aborter );
                }
                using(var file = await m_delegate.OpenFile() ) {
                    UInt64 transferOffset = m_resume.GetTransferOffset();
                    await file.Seek(transferOffset);
                    bool processAck = Description.Mode == DCCMode.Send;
                    UInt64 bytesSent = 0;
                    byte[] buffer = new byte[64 * 1024];
                    int bufferUsed = 0;

                    Task<UInt64> taskAck = null;
                    if ( processAck ) taskAck = AckLoop( stream, transferOffset, m_fileSize, aborter );

                    for (;;) {
                        if (aborter.IsCancellationRequested) return bytesSent;


                        bufferUsed = (int) await file.Read(buffer);
                        if (bufferUsed <= 0) break;

                        await stream.SendAsync( buffer, 0, bufferUsed, aborter );

                        bytesSent += (uint) bufferUsed;

                        Progress = bytesSent;
                    }
                    await taskAck;

                    return bytesSent;
                }
            }
        }

        void OnFinished(Exception e, UInt64 bytesDone, TimeSpan elapsed) {
            Running = false;
            if ( Aborting ) return;

            string label;
            if (e == null) {
                label = IRCUtils.TransferSpeedEstimator.FormatFinishedMessage(bytesDone,elapsed);
                m_delegate.SetProgress( UInt64.MaxValue );
            } else {
                label = "Failure: " + e.Message;
            }
            m_delegate.StatusText = label;
            m_delegate.SetPercentDone( UInt32.MaxValue );

            m_delegate.OnFinished( e );
        }

        bool Aborting {
            get {
                return m_abort.IsCancellationRequested;
            }
        }
        public void Abort()
        {
            m_abort.Cancel();
        }

        public void Resend() {
            m_resume.Reset();
            m_delegate.SetPercentDone( 0 );
            m_delegate.SetProgress( 0 );
            StartTransfer();
        }

        public UInt64 TransferOffset
        {
            get { return m_resume.PeekTransferOffset(); }
        }

        public bool TryResume(UInt64 resumeOffset) {
            if (resumeOffset > m_fileSize) return false;
            if (m_resume.SetTransferOffset(resumeOffset)) {
                return true;
            } else {
                return false;
            }
        }

        static IPAddress DetermineListenAddress(string ipString) {
            IPAddress ownAddress;
            if (IPAddress.TryParse(IRCUtils.CTCP.ParseDCCHost(ipString),out ownAddress)) {
                switch(ownAddress.AddressFamily) {
                    case AddressFamily.InterNetwork:
                        return IPAddress.Any;
                    case AddressFamily.InterNetworkV6:
                        return IPAddress.IPv6Any;
                    default:
                        return IPAddress.Any;
                }
            } else {
                return IPAddress.Any;
            }
        }

        class ListenThreadParam {
            public ListenThreadParam(PortUsage port, IPAddress listenAddress) {Port = port; ListenAddress = listenAddress;}
            public PortUsage Port;
            public IPAddress ListenAddress;
        };

        public void StartTransfer() {
            PortUsage port = null;
            try {
                m_abort = new CancellationTokenSource();
                SpeedEstimator.Reset();

                {
                    LinkedList<NetUtils.PortRange> portRanges = NetUtils.PortRange.ParseString(PreferencesManager.Current.xferPortRange);
                    port = new NetUtils.PortUsage(portRanges);
                }
                m_description.Port = port.Port;
                

                string ipString = m_connection.DCCIPString;
                IPAddress listenAddress = DetermineListenAddress(ipString);

                switch(Description.Mode) {
                    case DCCMode.Send:
                        m_connection.addCommand(IRCUtils.Commands.CTCP(Description.Nick,"DCC SEND " + IRCUtils.Commands.EncloseFilename(Description.FileName) + " " + ipString + " " + Description.Port + " " + m_fileSize));
                        break;
                    case DCCMode.TSend:
                        m_connection.addCommand(IRCUtils.Commands.CTCP(Description.Nick,"DCC TSEND " + IRCUtils.Commands.EncloseFilename(Description.FileName) + " " + ipString + " " + Description.Port + " " + m_fileSize));
                        break;
                    default:
                        throw new ArgumentException();
                }
                Running = true;
                ThreadMain( port, listenAddress );
            } catch {
                if (port != null) port.Close();
                throw;
            }
        }

        public DCCDescription Description {
            get { return m_description; }
        }
        ResumeStatus m_resume;
        CancellationTokenSource m_abort;
        DCCDescription m_description;
        IDCCSendDelegate m_delegate;
        UInt64 m_fileSize;
        ServerConnection m_connection;
        private TransferSpeedEstimator SpeedEstimator = new TransferSpeedEstimator();
        private UInt64 Progress = 0;
        private bool m_running = false;

        bool Running
        {

            get { return m_running; }
            set
            {
                m_running = value;
                if ( m_delegate != null ) m_delegate.ToggleRunning( value );
            }
        }

        class ResumeStatus {
            public ResumeStatus() {
                m_alreadyRunning = false;
                m_resumeOffset = 0;
            }

            public void Reset() {
                lock(this) {
                    m_alreadyRunning = false;
                    m_resumeOffset = 0;
                }
            }

            public UInt64 GetTransferOffset() {
                lock(this) {
                    m_alreadyRunning = true;
                    return m_resumeOffset;
                }
            }
            public UInt64 PeekTransferOffset() {
                lock(this) {
                    return m_resumeOffset;
                }
            }

            public bool SetTransferOffset(UInt64 offset) {
                lock(this) {
                    if (m_alreadyRunning) return false;
                    m_resumeOffset = offset;
                    return true;
                }
            }
                
            bool m_alreadyRunning;
            UInt64 m_resumeOffset;
        };
    };
}
