﻿using System;
using System.IO;
using System.Threading.Tasks;
using System.Threading;
using IRCUtils;

namespace monochrome
{

    public interface IDCCFileWriter : IDisposable {
        Task Seek(ulong position);
        Task SetLength( ulong length );
        Task<ulong> GetLength();
        Task Write(byte[] buffer, uint length);
    }


    public interface IDCCReceiveDelegate
    {
        UInt64 FileSize { get; }
        Task<IDCCFileWriter> CreateFile();
        Task<IDCCFileWriter> AppendFile();
        void SetProgress( UInt64 bytesTransferred );
        void SetPercentDone( UInt32 percentDone );
        String MainText { get; set; }
        String StatusText { get; set; }
        String SaveDialogFileName { get; set; }
        void OnFinished( Exception e );
        void ToggleRunning( bool bRunning );
        bool CanAcceptResume { get; }
    }

    public class DCCReceive : IDisposable
    {
        private CancellationTokenSource m_abort;
        private DCCDescription m_desc;
        private IDCCReceiveDelegate m_delegate;
        private ServerConnection m_connection;
        private bool m_running;
        public DCCDescription Description { get { return m_desc; } }
        private ResumeRequest m_resume = null;
        private UInt64 m_resumeOffset = 0;
        private UInt64 Progress = 0;
        private UInt64 m_fileSize;
        private TransferSpeedEstimator SpeedEstimator = new TransferSpeedEstimator();

        public bool Running
        {
            get { return m_running; }
            set
            {
                m_running = value;
                m_delegate.ToggleRunning( value );
            }
        }

        public DCCReceive( ServerConnection _conn, DCCDescription _desc, IDCCReceiveDelegate _delegate, PreferencesData _prefs )
        {
            m_connection = _conn;
            m_desc = _desc;
            m_delegate = _delegate;
            m_fileSize = _delegate.FileSize;

            m_connection.DCCReceiveAdd( this );

            m_delegate.MainText = "Receiving File: " + m_desc.FileName;
            m_delegate.StatusText = "Pending";

            if (_prefs.xferDefaultDirectory != "") {
                m_delegate.SaveDialogFileName = Path.Combine(_prefs.xferDefaultDirectory, m_desc.FileName);
            } else {
                m_delegate.SaveDialogFileName = m_desc.FileName;
            }

            TimerLoop();
        }
        private async void TimerLoop() {
            for (;;) {
                await Task.Delay(333);
                if (m_delegate == null) return; // are we disposed?
                TimerTickWorker();
            }
        }
        private void TimerTickWorker() {

            UInt32 percentage = UInt32.MaxValue;
            if (m_abort == null || m_abort.IsCancellationRequested) {
                //should not happen / nothing to do here
            } if (!Running) {
                if (m_resume != null) {
                    if (DateTime.UtcNow - m_resume.Timestamp > TimeSpan.FromSeconds(30)) {
                        m_resume = null;
                        m_delegate.StatusText = "Resume request timed out.";
                    }
                }
            } else {
                UInt64 bytesDone = Progress;
                SpeedEstimator.OnProgress(bytesDone);
                if (bytesDone > 0) {
                    m_delegate.StatusText = SpeedEstimator.FormatReport(m_fileSize,m_resumeOffset);
                }
                if (m_fileSize != UInt64.MaxValue) {
                    m_delegate.SetProgress( bytesDone + m_resumeOffset );
                }
                percentage = SpeedEstimator.PercentDone(m_fileSize,m_resumeOffset);
            }
            m_delegate.SetPercentDone(percentage);
        }

        public void Dispose() {
            m_connection.DCCReceiveRemove( this );
            if (m_abort != null) {
                m_abort.Cancel();
            }

            m_delegate = null;
        }
        public async void Start() {
            Running = true;

            SpeedEstimator.Reset();

            m_delegate.StatusText = "Connecting to: " + Description.Host + " : " + Description.Port.ToString();

            var resumeOffset = m_resumeOffset;
            var aborter = m_abort; // detach any previous thread
            var desc = m_desc; // desc varies with resume, use a copy of current to pass to thread
            if (aborter != null) {
                aborter.Cancel();
            }
            aborter = m_abort = new CancellationTokenSource();
            var startTime = DateTime.UtcNow;
            Exception status = null;
            UInt64 bytesTransferred = 0;
            try {
                bytesTransferred = await WorkerProc( resumeOffset, aborter.Token, desc );
            } catch( Exception e ) {
                status = e;
            }
            var elapsed = DateTime.UtcNow - startTime;
            if ( aborter.IsCancellationRequested ) return;
            await WorkerFinished( status, bytesTransferred, elapsed );
        }

        private async Task WorkerFinished( Exception e, UInt64 bytesTranferred, TimeSpan elapsed )
        {
            Running = false;
            string label;
            if (e == null) {
                label = IRCUtils.TransferSpeedEstimator.FormatFinishedMessage(bytesTranferred, elapsed );
                m_delegate.SetProgress( UInt64.MaxValue );
            } else {
                label = "Failure: " + e.Message;
            }
            m_delegate.StatusText = label;
            m_delegate.SetPercentDone( UInt32.MaxValue );
            m_delegate.OnFinished( e );

            await Task.Yield();

            if (ResumeReady) {
                StartResumedTransfer();
            }
        }

        private async Task<UInt64> WorkerProc( UInt64 _resumeOffset, CancellationToken aborter, DCCDescription desc ) {
            var resumeOffset = _resumeOffset;

            using( NetStream socket = new NetStream() ) {
                await socket.Open(
                    new NetStreamArg { remoteHost = desc.Host, remotePort = desc.Port }, 
                    CancellationToken.None 
                    );
                UInt64 bytesProcessed = 0;
                // using(FileStream file = resumeOffset > 0 ? File.Open(targetPath,FileMode.Append) : File.Create(targetPath)) {
                using(var file = resumeOffset > 0 ? await m_delegate.AppendFile() : await m_delegate.CreateFile() ) {
                    if (resumeOffset > 0) {
                        if (resumeOffset > await file.GetLength() ) {
                            throw new Exception("Resume offset out of range");
                        }
                        await file.SetLength(resumeOffset);
                        await file.Seek(resumeOffset);
                    }
                        
                    byte[] buffer = new byte[1024*1024];
                    byte[] ackTemp = new byte[4];
                        
                    bool bNeedAck = false;
                    bool done = false;
                    bool bProcessAck = (desc.Mode == DCCMode.Send);
                    Task<int> task_read = null;
                    while (!done || bNeedAck || task_read != null) {
                        if ( aborter.IsCancellationRequested ) return bytesProcessed;

                        if ( task_read == null && !done ) {
                            task_read = socket.ReadAsync(buffer, 0, buffer.Length, CancellationToken.None);
                            await Task.Yield(); // give it a chance to finish synchronously before we proceed to spam ACKs
                        }

                        if (task_read != null && (task_read.IsCompleted || !bNeedAck))
                        {
                            int received = await task_read; task_read = null;
                            if ( received <= 0 ) {
                                done = true;
                            } else {
                                await file.Write(buffer,(uint) received);
                                if (bProcessAck) bNeedAck = true;
                                bytesProcessed += (uint)received;
                                Progress = bytesProcessed;
                            }
                        } else if ( bNeedAck ) {

                            UInt64 ackPos = checked(bytesProcessed + (UInt64)resumeOffset);
                            byte[] temp = ackTemp;
                            temp[3] = (byte)(ackPos & 0xFF);
                            temp[2] = (byte)((ackPos >> 8) & 0xFF);
                            temp[1] = (byte)((ackPos >> 16) & 0xFF);
                            temp[0] = (byte)((ackPos >> 24) & 0xFF);

                            await socket.SendAsync( temp, 0, temp.Length, CancellationToken.None );
                            bNeedAck = false;
                        }
                    }
                }
                if (m_fileSize != UInt64.MaxValue && bytesProcessed + (UInt64)resumeOffset < m_fileSize) {
                    throw new Exception("Transfer aborted on remote side");
                }
                return bytesProcessed;
            }
        }
        public void InitiateResume(UInt64 requestResumePos) {
            InitiateResume(Description,requestResumePos);
        }

        public void InitiateResume(DCCDescription desc, UInt64 resumeRequestPos) {
            m_resume = new ResumeRequest(desc,UInt64.MaxValue /* gets filled when we get ACCEPT back */);

            m_connection.addCommand( IRCUtils.Commands.CTCP(desc.Nick,"DCC","RESUME " + IRCUtils.Commands.EncloseFilename(desc.FileName) + " " + desc.Port + " " + resumeRequestPos));
            m_delegate.StatusText = "Attempting to resume at " + resumeRequestPos + " bytes...";
        }

        bool ResumeReady {
            get {
                if (m_resume == null) return false;
                if (m_resume.Offset == UInt64.MaxValue) return false;
                return true;
            }
        }
        public bool TryInitiateResume(DCCDescription desc)
        {
            if (! m_delegate.CanAcceptResume ) return false;
            InitiateResume(desc,Progress + m_resumeOffset);
            return true;
        }
        void StartResumedTransfer() {
            m_desc = m_resume.Description;
            m_resumeOffset = m_resume.Offset;
            m_resume = null;
            Start();
        }

        public bool ResumeAccepted(UInt64 resumeOffset) {
            if (m_resume == null) return false;
            if (m_resume.Offset != UInt64.MaxValue) return false;
            m_resume.Offset = resumeOffset;
            if (!Running) StartResumedTransfer();
            return true;
        }

        class ResumeRequest {
            public ResumeRequest(DCCDescription desc, UInt64 offset) {
                Description = desc; Offset = offset;
                Timestamp = DateTime.UtcNow;
            }
            public DCCDescription Description;
            public UInt64 Offset;
            public DateTime Timestamp;
        };
    }
}
