﻿using System;

namespace monochrome
{
    // Partial because additional methods are added per-platform
    public partial struct ColorDesc
    {
        public byte Alpha8, Red8, Green8, Blue8;

        public double Alpha
        {
            get { return (double) Alpha8 / (double) byte.MaxValue; }
            set { Alpha8 = (byte) Math.Round( value * byte.MaxValue); }
        }
        public double Red
        {
            get { return (double) Red8 / (double) byte.MaxValue; }
            set { Red8 = (byte) Math.Round( value * byte.MaxValue); }
        }
        public double Green
        {
            get { return (double) Green8 / (double) byte.MaxValue; }
            set { Green8 = (byte) Math.Round( value * byte.MaxValue); }
        }
        public double Blue
        {
            get { return (double) Blue8 / (double) byte.MaxValue; }
            set { Blue8 = (byte) Math.Round( value * byte.MaxValue); }
        }

        public bool IsDark {
            get { return Brightness < 0.5; }
        }

        public double Brightness {
            get { return ( Red + Green + Blue ) / 3; }
        }

        public RGB RGB {
            get
            {
                RGB ret = new RGB();
                ret.Red = Red;
                ret.Green = Green;
                ret.Blue = Blue;
                return ret;
            }

            set
            {
                Red = value.Red;
                Green = value.Green;
                Blue = value.Blue;
            }
        }

        public static ColorDesc FromRGB( RGB rgb )
        {
            ColorDesc ret = new ColorDesc();
            ret.RGB = rgb;
            ret.Alpha8 = byte.MaxValue;
            return ret;
        }
        
        public static ColorDesc Empty {
            get {
                ColorDesc ret = new ColorDesc();
                ret.RGB = RGB.Black;
                ret.Alpha = 0;
                return ret;
            }
        }
        private static byte ImportHex(string s, int at)
        {
            int i = int.Parse(s.Substring(at,2), System.Globalization.NumberStyles.HexNumber);
            if ( i < 0 || i > 255 ) throw new ArgumentException();
            return (byte) i;
        }
        public static ColorDesc FromString( string s )
        {
            if (s.Length > 0 && s[0] == '#' )
            {
                s = s.Substring(1);
            }
            ColorDesc ret = new ColorDesc();
            ret.Alpha = 1;
            if ( s.Length == 8 )
            {
                ret.Alpha8 = ImportHex(s, 0);
                s = s.Substring(2);
            }
            if ( s.Length != 6 ) throw new ArgumentException();
            ret.Red8 = ImportHex( s, 0 );
            ret.Green8 = ImportHex( s, 2 );
            ret.Blue8 = ImportHex( s, 4 );
            return ret;
        }
        public static ColorDesc FromARGB( int r, int g, int b )
        {
            var ret = new ColorDesc();
            ret.Red8 = (byte) r;
            ret.Green8 = (byte) g;
            ret.Blue8 = (byte) b;
            ret.Alpha8 = byte.MaxValue;
            return ret;
        }
        public static ColorDesc FromARGB( int a, int r, int g, int b )
        {
            var ret = new ColorDesc();
            ret.Red8 = (byte) r;
            ret.Green8 = (byte) g;
            ret.Blue8 = (byte) b;
            ret.Alpha8 = (byte) a;
            return ret;
        }
        public bool IsEmpty
        {
            get { return Alpha == 0; }
        }
        public int AsInt
        {
            get
            {
                int ret = (int) Blue8;
                ret += (int) Green8 << 8;
                ret += (int) Red8 << 16;
                ret += (int) Alpha8 << 24;
                return ret;
            }
            set
            {
                int t = value;
                Blue8 = (byte) ( t & 0xFF ); t >>= 8;
                Green8 = (byte) ( t & 0xFF ); t >>= 8;
                Red8 = (byte) ( t & 0xFF ); t >>= 8;
                Alpha8 = (byte) ( t & 0xFF );
            }
        }
        public static ColorDesc FromInt( int i  )
        {
            var asdf = new ColorDesc();
            asdf.AsInt = i; return asdf;
        }

        static bool Equals( ColorDesc d1, ColorDesc d2 ) {
            return d1.AsInt == d2.AsInt;
        }
        public override int GetHashCode() {
            return AsInt;
        }
        public override bool Equals(object obj) {
            if (obj is ColorDesc) {
                return Equals(this, (ColorDesc)obj);
            } else {
                return false;
            }
        }
        public static bool operator==(ColorDesc item1, ColorDesc item2) { return Equals(item1, item2); }
        public static bool operator!=(ColorDesc item1, ColorDesc item2) { return Equals(item1, item2); }

        public static ColorDesc Blend(ColorDesc color1, ColorDesc color2, double position) {
            double aCombined = color1.Alpha + color2.Alpha;
            if (aCombined == 0) return Empty;
            aCombined = 1.0 / aCombined;
            var ret = new ColorDesc();
            ret.Red = (color1.Red * color1.Alpha + color2.Red * color2.Alpha) * aCombined;
            ret.Green = (color1.Green * color1.Alpha + color2.Green * color2.Alpha) * aCombined;
            ret.Blue = (color1.Blue * color1.Alpha + color2.Blue * color2.Alpha) * aCombined;
            ret.Alpha = (color1.Alpha + color2.Alpha) / 2;
            return ret;
        }
    }

    public struct HSL {
        public double Hue, Saturation, Luminance;//all in 0..1 range
    };
    public struct RGB {
        public double Red, Green, Blue;//all in 0..1 range

        public bool IsDark {
            get {
                return Red + Green + Blue < 0.5;
            }
        }
        public static RGB Gray(double v)
        {
            RGB ret; ret.Red = v; ret.Green = v; ret.Blue = v; return ret;
        }
        public static RGB Black
        {
            get { return Gray(0); }
        }
        public static RGB White
        {
            get { return Gray(1); }
        }
        
    };

    public partial class ColorUtils {
        public static RGB HSLtoRGB(HSL val) {
            RGB ret;
	        if (val.Saturation == 0)
	        {
		        // achromatic (grey)
		        ret.Red = ret.Green = ret.Blue = val.Luminance;
		        return ret;
	        }

	        double h = val.Hue * 6; // sector 0 to 5
	        int i = (int)Math.Floor(h);
	        double f = h - i; // factorial part of h

	        // p: primary color
	        // q: secondary color
	        double p, q;
	        if (val.Luminance < 0.5)
	        {
		        p = (1 + val.Saturation) * val.Luminance;
		        q = (1 - val.Saturation) * val.Luminance;
	        }
	        else
	        {
		        p = val.Luminance + (1 - val.Luminance) * val.Saturation;
		        q = val.Saturation * ((val.Luminance - 0.5) * 2) + (1 - val.Saturation) * val.Luminance;
	        }

	        switch (i)
	        {
	        case 0: // red -> yellow
		        ret.Red = p;
		        ret.Green = f * p + (1 - f) * q;
		        ret.Blue = q;
		        break;
	        case 1: // yellow -> green
		        ret.Red = (1 - f) * p + f * q;
		        ret.Green = p;
		        ret.Blue = q;
		        break;
	        case 2: // green -> cyan
		        ret.Red = q;
		        ret.Green = p;
		        ret.Blue = f * p + (1 - f) * q;
		        break;
	        case 3: // cyan -> blue
		        ret.Red = q;
		        ret.Green = (1 - f) * p + f * q;
		        ret.Blue = p;
		        break;
	        case 4: // blue -> magenta
		        ret.Red = f * p + (1 - f) * q;
		        ret.Green = q;
		        ret.Blue = p;
		        break;
	        default: // magenta -> red
		        ret.Red = p;
		        ret.Green = q;
		        ret.Blue = (1 - f) * p + f * q;
		        break;
	        }
            return ret;
        }

        public static RGB RandomColorRGB(int seed,double saturation, double luminance) {
            HSL temp;
            temp.Saturation = saturation;
            temp.Luminance = luminance;
            temp.Hue = (double)(Math.Abs(seed) % 256) / (double)256;
            return HSLtoRGB(temp);
        }
    }
}
