﻿using System;
using System.ComponentModel;
using IRCUtils;
using System.Collections.Generic;

namespace monochrome {

    public delegate void ChannelUserPrefixChangedHandler(string nick, string newPrefix);

    public sealed class ChannelStatusRecord {
        public ChannelStatusRecord(string prefix = "") { Prefix = prefix; OnActivity(); }
        public string Prefix;
        public DateTime Activity;
        public void OnActivity() {
            Activity = DateTime.UtcNow;
        }
    };
    public sealed class ChannelStatus : INotifyPropertyChanged {
        public ChannelStatus(ServerConnection connection, string channelName) {
            m_connection = new WeakReference<ServerConnection>(connection);
            m_channelName = channelName;
        }
        private string m_channelName;
        public string ChannelName { get { return m_channelName; } }

        public event PropertyChangedEventHandler PropertyChanged;
        private void NotifyPropertyChanged(String propertyName = "") {
            if (PropertyChanged != null) {
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        public event ChannelUserPrefixChangedHandler UserPrefixChanged;
        private void NotifyPrefixChanged(string nick, string prefix) {
            if ( UserPrefixChanged != null ) UserPrefixChanged( nick, prefix );
        }
        

        private string m_channelMode = "";
        public string ChannelMode {
            get { return m_channelMode; }
            set {
                bool couldSetTopic = CanSetTopic;
                m_channelMode = value;
                NotifyPropertyChanged("ChannelMode");
                if (couldSetTopic != CanSetTopic) NotifyPropertyChanged("CanSetTopic");
            }
        }

        private string m_userMode = "";
        public string UserMode {
            get { return m_userMode; }
            set {
                if (m_userMode == value) return;
                bool hadOp = this.HaveOp;
                bool couldSetTopic = this.CanSetTopic;
                m_userMode = value;
                NotifyPropertyChanged("UserMode");
                if (CanSetTopic != couldSetTopic) NotifyPropertyChanged("CanSetTopic");
                if (HaveOp != hadOp) NotifyPropertyChanged("HaveOp");
            }
        }

        public bool HaveOp {
            get {
                return UserMode.IndexOfAny(new char[] { '@', '%' }) >= 0;
            }
        }

        private string m_topic = "";
        public string Topic {
            get { return m_topic; }
            set {
                if (m_topic == value) return;
                m_topic = value; NotifyPropertyChanged("Topic");
            }
        }
        public bool CanSetTopic {
            get {
                return HaveOp || !ChannelModeSet('t');
            }
        }

        public bool ChannelModeSet(char mode) {
            try {
                foreach (Modes.Mode walk in Modes.Parse(ChannelMode, Connection.InstanceData.ModeSpecs)) {
                    if (walk.action == mode) return walk.sign;
                }
                return false;
            } catch { return false; }
        }

        private static Dictionary<string, ChannelStatusRecord> NewStatusMap() {
            return new Dictionary<string, ChannelStatusRecord>(Nicks.Comparer);
        }
        private Dictionary<string, ChannelStatusRecord> m_statusByNick = NewStatusMap();

        public DateTime LastActivityOfNick( string nick ) {
            ChannelStatusRecord ret;
            if (! m_statusByNick.TryGetValue( nick, out ret ) ) return DateTime.MinValue;
            return ret.Activity;
        }

        public string PrefixOfNick( string nick ) {
            ChannelStatusRecord ret;
            if ( ! m_statusByNick.TryGetValue(nick, out ret ) ) return "";
            return ret.Prefix;
        }

        public void SetNameList(IEnumerable<string> namesWithPrefixes) {
            var newStatus = NewStatusMap();

            if ( namesWithPrefixes != null ) {
                var prefixList = this.PrefixList;
                foreach (string nameWithPrefix in namesWithPrefixes) {
                    string nick = Nicks.Undecorate(nameWithPrefix, prefixList);
                    string prefix = Nicks.ExtractStatus(nameWithPrefix, prefixList);
                    ChannelStatusRecord rec;
                    if ( ! m_statusByNick.TryGetValue( nick, out rec ) ) {
                        rec = new ChannelStatusRecord();
                    }
                    rec.Prefix = prefix;

                    try {
                        newStatus.Add(nick, rec);
                    } catch {
                        // Do not DoS ourselves if the IRC server is giving us nonsense
                    }
                }
            }
            m_statusByNick = newStatus;
            NotifyPropertyChanged( "UserNicks" );
            
            UserMode = PrefixOfNick( Connection.ownNick );
        }
        public void AddNick( string nick, string status = "" ) {
            status = Connection.InstanceData.ModeSpecs.Prefixes.ResortPrefixes(status);
            m_statusByNick.Remove(nick); // sanity
            m_statusByNick.Add(nick, new ChannelStatusRecord(status));
            if (Nicks.Equals(nick, Connection.ownNick)) {
                this.UserMode = status;
            }
        }
        public bool HaveNick( string nick ) {
            return m_statusByNick.ContainsKey( nick );
        }
        public bool RemoveNick( string nick ) {
            return m_statusByNick.Remove( nick );
        }
        public bool IsOn {
            get {
                return HaveNick( Connection.ownNick );
            }
        }

        ChannelStatusRecord makeRecord( string nick ) {
            ChannelStatusRecord rec;
            if (!m_statusByNick.TryGetValue( nick, out rec ) ) {
                rec = new ChannelStatusRecord();
                m_statusByNick.Add( nick, rec );
            }
            return rec;
        }

        void togglePrefix(string who, char prefix, bool sign) {
            var nick = Nicks.Extract(who);
            var rec = makeRecord( nick );

            string status = rec.Prefix;

            if (sign) {
                if (status.IndexOf(prefix) < 0) status += prefix;
                else return;
            } else {
                int index = status.IndexOf(prefix);
                if (index >= 0) status = status.Remove(index, 1);
                else return;
            }
            status = Connection.InstanceData.ModeSpecs.Prefixes.ResortPrefixes(status);

            rec.Prefix = status;

            if (Nicks.Equals(nick, Connection.ownNick)) {
                this.UserMode = status;
            }

            NotifyPrefixChanged(nick, status);
        }
        public void HandleActivity( string nick ) {
            ChannelStatusRecord rec;
            if ( m_statusByNick.TryGetValue( nick, out rec ) ) {
                rec.OnActivity();
            }
        }

        public void HandleMode(IRCEvent ev, ServerModeSpecs modeSpecs) {
            LinkedList<Modes.Mode> modes = Modes.Parse(ev, modeSpecs);
            LinkedList<Modes.Mode> channelModes;
            try {
                channelModes = Modes.Parse(this.ChannelMode, modeSpecs);
            } catch (Exception e) {
                Connection.printContextOutput(ChannelName, "Could not parse channel modes: " + e.Message);
                return;
            }
            bool channelModesChanged = false;
            foreach (Modes.Mode mode in modes) {
/*
                if (mode.param != null) {
                    BanListWindow wnd;
                    if (m_listWindows.TryGetValue(mode.action, out wnd)) {
                        if (mode.sign) {
                            BanDescription desc = new BanDescription();
                            desc.Mask = mode.param;
                            desc.ByWhom = ev.Source;
                            desc.Date = DateTime.UtcNow;
                            wnd.OnEntryAdded(desc);
                        } else {
                            wnd.OnEntryRemoved(mode.param);
                        }
                    }
                }
                */
                char prefix;
                if (modeSpecs.Prefixes.ModeToPrefix(mode.action, out prefix)) {
                    togglePrefix(mode.param, prefix, mode.sign);
                } else {
                    channelModesChanged = true;
                    {
                        bool found = false;
                        for (LinkedListNode<Modes.Mode> walk = channelModes.First; walk != null;) {
                            LinkedListNode<Modes.Mode> next = walk.Next;
                            if (walk.Value.action == mode.action) {
                                found = true;
                                if (mode.sign != walk.Value.sign) {
                                    channelModes.Remove(walk);
                                }
                            }
                            walk = next;
                        }
                        if (!found && mode.sign) {
                            channelModes.AddLast(mode);
                        }
                    }
                }
            }

            if (channelModesChanged) {
                ChannelMode = Modes.MakeString(channelModes, modeSpecs, true);
            }
        }

        public void NickChanged( string oldNick, string newNick ) {
            ChannelStatusRecord rec;
            if ( m_statusByNick.TryGetValue( oldNick, out rec) ) {
                m_statusByNick.Remove( oldNick );
                try {
                    m_statusByNick.Add(newNick, rec);
                } catch { }
            }
        }
        public void Unsubscribe() {
            ChannelMode = "";
            UserMode = "";
            SetNameList(null);
        }
        public IEnumerable<string> UserNicks {
            get { return m_statusByNick.Keys; }
        }

        public ServerConnection Connection {
            get {
                ServerConnection c;
                if (!m_connection.TryGetTarget(out c)) throw new InvalidOperationException();
                return c;
            }
        }
        PrefixMap Prefixes {
            get { return Connection.InstanceData.ModeSpecs.Prefixes; }
        }
        string PrefixList {
            get { return Prefixes.PrefixList; }
        }

        private WeakReference<ServerConnection> m_connection;
    }

}