﻿using System.Collections.Generic;

namespace monochrome
{

    public interface IAutoCompleteSource {
        LinkedList<string> GetACStrings();
        string GetACChanTypes();
    }

    public interface IAutoCompleteEditBox {
        void SetText(string newText, int newSelOffset);
        int GetSelectionEnd();
        string GetText();
    }

    public class AutoCompleteContext {
        public AutoCompleteContext(LinkedList<string> _values) {
            values = _values;
        }

        static bool Compare(string input, string test) {
            int len = input.Length;
            if (len < 1 || len > test.Length) return false;
            return IRCUtils.Nicks.Equals(input, test.Substring(0,len));
        }

        LinkedListNode<string> CircularNext(LinkedListNode<string> walk,bool reverse) {
            if (reverse) {
                LinkedListNode<string> next = walk.Previous;
                if (next == null) next = values.Last;
                return next;
            } else {
                LinkedListNode<string> next = walk.Next;
                if (next == null) next = values.First;
                return next;
            }
        }

        public string Next(string input,bool reverse) {
            if (walk == null) {
                if (reverse) {
                    for(walk = values.Last; walk != null; walk = walk.Previous) {
                        if (Compare(input,walk.Value)) return walk.Value;
                    }
                } else {
                    for(walk = values.First; walk != null; walk = walk.Next) {
                        if (Compare(input,walk.Value)) return walk.Value;
                    }
                }
                return null;
            } else {
                LinkedListNode<string> before = walk;
                walk = CircularNext(walk,reverse);
                for( ;; ) {
                    if (Compare(input,walk.Value)) return walk.Value;
                    walk = CircularNext(walk,reverse);
                    if (walk == before) return null;
                }
            }
        }

        LinkedList<string> values;
        LinkedListNode<string> walk;
    };



    public class AutoCompleteHandler {
        public AutoCompleteHandler(IAutoCompleteSource source, IAutoCompleteEditBox editBox) {
            EditBox = editBox;
            Source = source;
        }

        public void OnTriggerAutoComplete(bool bShift) {
            if (Source != null) Run(bShift);
        }
        // Doesn't really care if it gets called after or before the actual text changes
        // But it's essential that it gets called synchronously from within our SetText() call - or not called at all if we set the text
        // Must *NOT* be called as a result of our SetText() but past SetText() duration.
        public void OnTextChanging() {
            if (!m_ownUpdate) Reset();
        }
        public void OnEnterLeave() {
            Reset();
        }

        static bool isSpacing(char c) {
            switch (c) {
                case ' ':
                case ',':
                case '.':
                case ';':
                case ':':
                    return true;
                default:
                    return false;
            }
        }

        private void Run(bool reverse) {
            if (autoComplete == null) {
                if (Source == null) return;
                var lst = Source.GetACStrings();
                if ( lst == null ) return;
                autoComplete = new AutoCompleteContext(lst);
            }

            int origSelEnd = EditBox.GetSelectionEnd();
            if (autoCompleteOffset < 0) autoCompleteOffset = /*editBox.SelectionStart + editBox.SelectionLength;*/ origSelEnd;
            if (autoCompleteOffset < 0) return;

            int walk = autoCompleteOffset;
            string text = /*editBox.Text;*/ EditBox.GetText();
            if (walk > text.Length) walk = text.Length;
            for (;;) {
                if (walk == 0) break;
                if (isSpacing(text[walk - 1])) break;
                --walk;
            }
            int len = 0;
            for (;;) {
                if (walk + len >= text.Length || walk + len >= autoCompleteOffset) break;
                if (isSpacing(text[walk + len])) break;
                len++;
            }
            string remaining;
            if (autoCompleteOffsetEnd < autoCompleteOffset) {
                int remBase = walk + len;
                for (;;) {
                    if (remBase >= text.Length) break;
                    if (!isSpacing(text[remBase])) break;
                    remBase++;
                }
                remaining = text.Substring(remBase);
            } else if (autoCompleteOffsetEnd < text.Length ) {
                remaining = text.Substring(autoCompleteOffsetEnd);
            } else {
                remaining = "";
            }
            string acWhat = text.Substring(walk, len);
            string replacement = autoComplete.Next(acWhat, reverse);
            if (replacement != null) {
                string suffix = (walk == 0 && !IRCUtils.Nicks.IsChannel(replacement, Source.GetACChanTypes() )) ? ": " : "";
                string newText = text.Substring(0, walk) + replacement + suffix;
                int newSelOffset = autoCompleteOffsetEnd = newText.Length;
                newText += remaining;
                try {
                    m_ownUpdate = true;
                    EditBox.SetText( newText, newSelOffset );
                } finally { m_ownUpdate = false; }
            }
        }

        public void Reset() { autoCompleteOffset = autoCompleteOffsetEnd = -1; autoComplete = null; }
        public void Reset(IAutoCompleteSource source) {
            Source = source;
            Reset();
        }
        IAutoCompleteSource Source;
        AutoCompleteContext autoComplete;
        IAutoCompleteEditBox EditBox;
        int autoCompleteOffset, autoCompleteOffsetEnd;
        bool m_ownUpdate;
    };

}
