﻿using System;
using System.IO;
using System.Threading;
using Windows.Networking;
using Windows.Networking.Sockets;
using System.Threading.Tasks;
using Windows.Storage.Streams;
using System.Runtime.InteropServices.WindowsRuntime;

namespace monochrome
{
    class NetStream : IDisposable
    {
        StreamSocket socket;

        public void Close() {
            if ( socket != null )
            {
                try {
                    socket.Dispose();
                    socket = null;
                } catch { }
            }
        }

        public NetStream( StreamSocket initSocket = null ) { socket = initSocket; }

        public async Task Accept( System.Net.IPAddress address, int port, CancellationToken token ) {
            using (StreamSocketListener listener = new StreamSocketListener())
            using (var signal = new SemaphoreSlim(0, 1)) {

                listener.ConnectionReceived += (StreamSocketListener sender, StreamSocketListenerConnectionReceivedEventArgs args) => {
                    this.socket = args.Socket;
                    signal.Release();
                };

                if (address != System.Net.IPAddress.Any) {
                    var host = new HostName(address.ToString());
                    await listener.BindEndpointAsync(host, port.ToString());
                } else {
                    string s = port.ToString();
                    await listener.BindServiceNameAsync(s);
                }

                await signal.WaitAsync(token);
            }
        }

        public async Task Open( string remoteHost, int remotePort, bool useSSL, System.Threading.CancellationToken token )
        {
            var host = new HostName( remoteHost );
            socket = new StreamSocket();
            var protLevel = SocketProtectionLevel.PlainSocket;
            if ( useSSL ) protLevel = SocketProtectionLevel.Tls12;
            await socket.ConnectAsync( host, remotePort.ToString(), protLevel );
        }

        public async Task SendAsync(byte[] buffer,int offset, int count, System.Threading.CancellationToken token) {
            var b = WindowsRuntimeBufferExtensions.AsBuffer(buffer, offset, count);
            await socket.OutputStream.WriteAsync( b );
        }
        public async Task<int> ReadAsync(byte[] buffer, int offset, int count, System.Threading.CancellationToken token) {
            var b = WindowsRuntimeBufferExtensions.AsBuffer(buffer, offset, count);
            var returned = await socket.InputStream.ReadAsync(b, (uint) count, InputStreamOptions.Partial );
            if ( returned != b ) {
                returned.CopyTo( b );
            }
            return (int) returned.Length;
        }
        public void Send(byte[] buffer,int offset, int count) {
            var b = WindowsRuntimeBufferExtensions.AsBuffer(buffer, offset, count);
            var task = socket.OutputStream.WriteAsync( b );
            task.AsTask().Wait();
        }
        public int Read(byte[] buffer, int offset, int count) {
            var b = WindowsRuntimeBufferExtensions.AsBuffer(buffer, offset, count);
            var task = socket.InputStream.ReadAsync(b, (uint) count, InputStreamOptions.Partial );
            task.AsTask().Wait();
            var returned = task.GetResults();
            if ( returned != b ) {
                returned.CopyTo( b );
            }
            return (int) returned.Length;
        }

        void IDisposable.Dispose()
        {
            Close();
        }


        public static async Task<string> ResolveHostName(string inHostName)
        {
            if ( TextUtils.IsStringIPv4( inHostName ) ) return inHostName;
            var results = await StreamSocket.GetEndpointPairsAsync(new HostName(inHostName), "0" );
            string ret = "";
            foreach( EndpointPair entry in results )
            {
                if ( ret.Length > 0 ) ret += ", ";
                ret += entry.RemoteHostName.ToString();
            }
            return ret;
        }

        public async static Task<string> ResolveHostNameForDCC( string inHostName )
        {
            string name = await ResolveHostName( inHostName );
            if ( TextUtils.IsStringIPv4( name ) ) {
                name = IRCUtils.Formatting.formatDCCIP( name );
            }
            return name;
        }
    }
}
