﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices.WindowsRuntime;
using Windows.Foundation;
using Windows.Foundation.Collections;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Controls.Primitives;
using Windows.UI.Xaml.Data;
using Windows.UI.Xaml.Input;
using Windows.UI.Xaml.Media;
using Windows.UI.Xaml.Navigation;
using monochrome;
using System.Collections.ObjectModel;
using System.ComponentModel;
using Windows.UI.Popups;

namespace Monochrome_UWP
{

    public class PageIndexEntry : INotifyPropertyChanged {


        public FlyoutBase ContextMenu {
            get {
                if ( m_data is PreferencesState || m_data is ServerSetupState ) return null;

                var iws = m_data as IRCWindowState;
                if ( iws != null ) {
                    return iws.ContextMenu;
                }
                var ds = m_data as IDisposable;
                if ( ds != null ) {
                    var menu = new MenuFlyout();
                    var itemClose = new MenuFlyoutItem();
                    itemClose.Text = "Close";
                    itemClose.Click += (System.Object sender, RoutedEventArgs e) => { ds.Dispose(); };
                    menu.Items.Add( itemClose );
                    return menu;
                }
                return null;
            }
        }

        public event PropertyChangedEventHandler PropertyChanged;

        public void PreferencesChanged() {
            NotifyPropertyChanged();
        }
        void NotifyPropertyChanged(String propertyName = "") {
            if (PropertyChanged != null) {
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        public PageIndexEntry( string name, Type type, object data ) {
            m_name = name;
            m_type = type;
            m_data = data;
        }
        public override string ToString() {
            return Name;
        }

        public string Name {
            get { return m_name; }
        }
        public Type Type {
            get { return m_type; }
        }
        public object Data {
            get { return m_data; }
        }

        public Brush ColorBrush {
            get {
                return new SolidColorBrush( Color );
            }
        }
        public Windows.UI.Color Color {
            get {
                switch(m_highlight) {
                    default:
                    case 0:
                        return PreferencesManager.Current.ColorText;
                    case 1:
                        return PreferencesManager.Current.ColorChanHighlight1;
                    case 2:
                        return PreferencesManager.Current.ColorChanHighlight2;
                    case 3:
                        return PreferencesManager.Current.ColorChanHighlight3;
                }
            }
        }

        public int Highlight {
            set {
                m_highlight = value;
                NotifyPropertyChanged("Highlight");
                NotifyPropertyChanged("Color");
                NotifyPropertyChanged("ColorBrush");
            }
            get {
                return m_highlight;
            }
        }

        private Type m_type;
        private string m_name;
        private object m_data;
        private int m_highlight = 0;
    }

    public sealed partial class MainPage : Page, INotifyPropertyChanged {

        public event PropertyChangedEventHandler PropertyChanged;

        void NotifyPropertyChanged(String propertyName = "") {
            if (PropertyChanged != null) {
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        ObservableCollection<PageIndexEntry> leftPaneItems;
        int leftPaneIndex;

        public ObservableCollection<PageIndexEntry> LeftPaneItems {
            get { return leftPaneItems; }
        }
        public int LeftPaneIndex {
            get { return leftPaneIndex; }
            set { leftPaneIndex = value; NotifyPropertyChanged("LeftPaneIndex"); }
        }
        public bool PreferencesVisible {
            get {
                int idx = this.LeftPaneIndex;
                if (idx < 0) return false;
                var page = this.LeftPaneItems[idx];
                return (page.Data is PreferencesState);
            }
        }
        public MainPage() {
            this.InitializeComponent();

            PreferencesManager.Notify += PreferencesManager_Notify;

            leftPaneItems = new ObservableCollection<PageIndexEntry> { };

            OpenServerSetup();
            OpenPreferences();

            AutoConnect();


        }

        private void PreferencesManager_Notify(PreferencesData newData) {
            foreach( var e in LeftPaneItems ) {
                e.PreferencesChanged();
            }
        }

        private void App_LeavingBackground(object sender, Windows.ApplicationModel.LeavingBackgroundEventArgs e) {
            ServerConnection.IsAway = ManualAway;
        }

        private void App_EnterBackground(object sender, Windows.ApplicationModel.EnteredBackgroundEventArgs e) {
            ServerConnection.IsAway = true;
        }

        private void AutoConnect() {
            var shift = Window.Current.CoreWindow.GetKeyState(Windows.System.VirtualKey.Shift);
            if (shift.HasFlag(Windows.UI.Core.CoreVirtualKeyStates.Down)) {
                return;
            }
            var ac = ServerConfigManager.GrabAutoConnectEntries();
            foreach (var entry in ac) {
                Connect(entry.Value, entry.Key);
            }
        }
        public void SetPageHighlight(object data, int flashLevel) {
            for (int i = 0; i < leftPaneItems.Count; ++i) {
                var entry = leftPaneItems[i];
                if (entry.Data == data) {
                    if (i != leftPaneIndex) {
                        entry.Highlight = flashLevel;
                    }
                    return;
                }
            }
        }
        public int IndexOfPage(object data) {
            for (int i = 0; i < leftPaneItems.Count; ++i) {
                var e = leftPaneItems[i];
                if (e.Data == data) {
                    return i;
                }
            }
            return -1;
        }

        public bool SwitchToPageIndex(int index) {
            if (index < 0) return false;
            int walk = index;
            for (int i = 0; i < LeftPaneItems.Count; ++i) {
                var entry = LeftPaneItems[i];
                if (entry.Data is IMessageWindow) {
                    if (walk == 0) {
                        this.LeftPaneIndex = i;
                        return true;
                    }
                    --walk;
                }
            }
            return false;
        }
        public void SwitchPageDelta(int delta) {
            int total = this.LeftPaneItems.Count;
            if (total <= 0) return;
            if (delta > 0) delta %= total;
            else delta = -((-delta) % total);
            int walk = this.LeftPaneIndex;
            walk += delta;
            while (walk < 0) walk += total;
            walk %= total;
            this.LeftPaneIndex = walk;
        }
        public void SwitchToPage(object data) {
            for (int i = 0; i < leftPaneItems.Count; ++i) {
                var e = leftPaneItems[i];
                if (e.Data == data) {
                    LeftPaneIndex = i; return;
                }
            }
        }
        public void AddPage(string name, Type type, object data) {
            leftPaneItems.Add(new PageIndexEntry(name, type, data));
        }
        public void RemovePage(object data) {
            for (int i = 0; i < leftPaneItems.Count; ++i) {
                var e = leftPaneItems[i];
                if (e.Data == data) {
                    int LPIwas = leftPaneIndex;
                    leftPaneItems.RemoveAt(i);
                    if ( leftPaneIndex < 0 && leftPaneItems.Count > 0 ) {
                        if ( LPIwas + 1 >= leftPaneItems.Count ) {
                            LeftPaneIndex = leftPaneItems.Count - 1;
                        } else {
                            LeftPaneIndex = LPIwas;
                        }
                    }
                    
                    return;
                }
            }
        }
        public void Connect(ServerParams arg, string netName) {
            new ServerWindowState(this, arg, netName);
        }

        void OpenServerSetup() {
            new ServerSetupState(this);
        }
        void OpenPreferences() {
            new PreferencesState(this);
        }

        private void HandleSwitchPage() {
            if (leftPaneIndex < 0) return; // FIX ME happens when closing windows
            
            
            var entry = leftPaneItems[leftPaneIndex];
            Debug.WriteLine("HandleSwitchPage: " + leftPaneIndex + " " + entry.Data.ToString());
            entry.Highlight = 0;
            var t = entry.Type;
            if (t != null) {
                MainFrame.Navigate(t, entry.Data);
                // FIX ME default-focus
            }
        }

        private void ListBox_SelectionChanged(object sender, SelectionChangedEventArgs e) {
            HandleSwitchPage();
        }

        private void MenuClose_Click(object sender, RoutedEventArgs e) {
            var fe = sender as FrameworkElement;
            if (fe == null) return;
            var entry = fe.DataContext as PageIndexEntry;
            if (entry == null) return;
            var obj = entry.Data as IDisposable;
            if (obj != null) {
                obj.Dispose();
            }
        }

        private void Grid_KeyDown(object sender, KeyRoutedEventArgs e) {
            if (HandleGlobalKeyDown(e.Key)) e.Handled = true;
        }

        protected override void OnNavigatedTo(NavigationEventArgs e) {
            base.OnNavigatedTo(e);
            Window.Current.CoreWindow.KeyDown += CoreWindow_KeyDown;

            Application.Current.EnteredBackground += App_EnterBackground;
            Application.Current.LeavingBackground += App_LeavingBackground;
        }

        static bool IsModifierPressed(Windows.System.VirtualKey k) {
            var state = Window.Current.CoreWindow.GetKeyState(k);
            return state.HasFlag(Windows.UI.Core.CoreVirtualKeyStates.Down);
        }
        private bool HandleGlobalKeyDown(Windows.System.VirtualKey key) {
            // This is called from both corewindow keydown and root grid keydown, apparently necessary
            if (key >= Windows.System.VirtualKey.Number0 && key <= Windows.System.VirtualKey.Number9) {
                int index = (int)(key - Windows.System.VirtualKey.Number0);
                if (index == 0) index = 10;
                else index--;
                if (IsModifierPressed(Windows.System.VirtualKey.Menu)) {
                    return SwitchToPageIndex(index);
                }
            } else if (key == Windows.System.VirtualKey.PageUp || key == Windows.System.VirtualKey.PageDown) {
                if (IsModifierPressed(Windows.System.VirtualKey.Control)) {
                    SwitchPageDelta(key == Windows.System.VirtualKey.PageUp ? -1 : 1);
                    return true;
                }
            }
            return false;
        }
        private void CoreWindow_KeyDown(Windows.UI.Core.CoreWindow sender, Windows.UI.Core.KeyEventArgs args) {
            if (HandleGlobalKeyDown(args.VirtualKey)) {
                args.Handled = true;
            }
        }

        protected override void OnNavigatedFrom(NavigationEventArgs e) {
            base.OnNavigatedFrom(e);
            Window.Current.CoreWindow.KeyDown -= CoreWindow_KeyDown;

            Application.Current.EnteredBackground -= App_EnterBackground;
            Application.Current.LeavingBackground -= App_LeavingBackground;

        }

        public bool ManualAway { get; set; }

        
        TriggerLogState m_triggerLog = null, m_messageLog = null;

        public void TriggerLogClosed( TriggerLogState log ) {
            if ( m_triggerLog == log ) m_triggerLog = null;
            else if ( m_messageLog == log ) m_messageLog = null;
        }

        private void LogEvent( string netName, string context, TextLine line, ref TriggerLogState log, string logLabel, bool highlight ) {
            if (log == null) {
                log = new TriggerLogState(this, logLabel);
            }
            log.AddLine(TextLine.ContextPrefix(netName) + TextLine.ContextPrefix(context) + line);
        }

        public void LogActivity( string netName, string context, TextLine line, int level) {
            PreferencesData prefs = PreferencesManager.Current;
            if (prefs.useTriggerLog && level >= 3) LogEvent(netName, context, line, ref m_triggerLog, "Trigger Log", false);
            if (prefs.useMessageLog && level >= 2) LogEvent(netName, context, line, ref m_messageLog, "Message Log", level >= 3);
        }
    }
}
