﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Windows.Storage;
using Windows.Storage.AccessCache;
using System.Runtime.InteropServices.WindowsRuntime;

namespace monochrome
{
    class Logging {

        static SemaphoreSlim loggingTrigger = new SemaphoreSlim(0);
        static LinkedList<Func<Task> > loggingActions = new LinkedList<Func<Task> >();

        static bool loggingLoopRunning = false;

        static async void loggingLoop() {
            for (;;) {
                await loggingTrigger.WaitAsync();
                var action = loggingActions.First.Value;
                loggingActions.RemoveFirst();

                try {
                    await action();
                } catch {}
            }
        }

        static Dictionary<string, StorageFolder> g_folderCache = new Dictionary<string, StorageFolder>();
        static Dictionary<string, StorageFile> g_fileCache = new Dictionary<string, StorageFile>();

        private static string fixInvalidCharacters(string filename) {
            char[] invalid = Path.GetInvalidFileNameChars();
            for (int walk = 0; ;) {
                int index = filename.IndexOfAny(invalid, walk);
                if (index < 0) break;
                filename = filename.Substring(0, index) + '_' + filename.Substring(index + 1);
                walk = index + 1;
            }
            return filename;
        }

        public class LoggingTargetInfo {
            public StorageFile File;
            public StorageFolder Folder;
        };
        // returns path of used log file
        public static void WriteLine(string server, string context, string line, Action<object> loggingTargetTalkBack) {
            PreferencesData prefs = PreferencesManager.Current;

            if (prefs.useLogging && prefs.loggingPath != "" ) {

                if (!loggingLoopRunning) {
                    loggingLoopRunning = true;
                    loggingLoop();
                }

                string basePathToken = prefs.loggingPathToken;
                string basePath = prefs.loggingPath;
                string subPath = fixInvalidCharacters( server );

                DateTime time = DateTime.Now;
                string filename = ((context == null) ? "[server]" : context);
                if (prefs.splitLogsByDays) filename += ", " + time.ToString("yyyy-MM-dd");
                filename += ".txt";
                filename = fixInvalidCharacters( filename );

                string retLogFilePath = Path.Combine(basePath, subPath, filename);

                string lineWriteMe = time.ToString(prefs.loggingTimestampFormat) + " " + line + "\r\n";

                loggingActions.AddLast( async () => {

                    StorageFolder subFolder;
                    if (!g_folderCache.TryGetValue(Path.Combine(basePath, subPath), out subFolder)) {
                        StorageFolder baseFolder;
                        if (!g_folderCache.TryGetValue(basePath, out baseFolder)) {
                            baseFolder = await StorageApplicationPermissions.FutureAccessList.GetFolderAsync(basePathToken);
                        }
                        subFolder = await baseFolder.CreateFolderAsync(subPath, CreationCollisionOption.OpenIfExists);
                    }

                    StorageFile file;
                    if (! g_fileCache.TryGetValue( Path.Combine(basePath, subPath, filename ), out file ) ) {
                        file = await subFolder.CreateFileAsync(filename, CreationCollisionOption.OpenIfExists);
                    }
                    if (loggingTargetTalkBack != null) {
                        var obj = new LoggingTargetInfo();
                        obj.File = file;
                        obj.Folder = subFolder;
                        loggingTargetTalkBack( obj );
                    }
                    using (var stream = await file.OpenAsync(FileAccessMode.ReadWrite)) {
                        stream.Seek(stream.Size);

                        var bytes = System.Text.Encoding.UTF8.GetBytes(lineWriteMe);
                        
                        await stream.WriteAsync( bytes.AsBuffer() );
                    }
                });
                loggingTrigger.Release();
            }
        }
    }
}
