﻿using System;
using System.Collections.Generic;
using Windows.Foundation;
using Windows.Foundation.Collections;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Controls.Primitives;
using Windows.UI.Xaml.Data;
using Windows.UI.Xaml.Input;
using Windows.UI.Xaml.Navigation;
using monochrome;
using System.Threading.Tasks;
using System.ComponentModel;
using System.Collections.ObjectModel;

namespace Monochrome_UWP {

    public sealed partial class IRCWindow : Page {

        private ChannelViewHooks m_cvHooks;
        public ChannelViewHooks CV { get { return m_cvHooks; } }

        public IRCWindow() {
            this.InitializeComponent();
            this.Loaded += IRCWindow_Loaded;
            m_cvHooks = new Monochrome_UWP.ChannelViewHooks( ChannelView, ChannelViewScroll );
        }
        private void IRCWindow_Loaded(object sender, RoutedEventArgs e) {
            CV.ScrollToBottom();
        }

        protected override void OnNavigatedTo(NavigationEventArgs e) {
            base.OnNavigatedTo(e);
            State = (IRCWindowState) e.Parameter;
            State.Frontend = this;
            
            CV.Initialize( State.ViewContent );

            InputBox.ACSource = State.ACSource;
        }

        protected override void OnNavigatingFrom(NavigatingCancelEventArgs e) {
            base.OnNavigatingFrom(e);
            if ( State != null ) State.Frontend = null;
        }

        public IRCWindowState State { get; set; }

        private void TopicBox_EscPressed(object sender) {
            State.UserRevertTopic();
        }
        private void TopicBox_EnterPressed(object sender) {
            State.UserSetTopic( TopicBox.Text );
        }
        private void InputBox_EnterPressed(object sender) {
            if ( ! CV.IsAtBottom ) {
                CV.ScrollToBottomNowAndThen(100); return;
            }


            {
                var content = State.InputBoxContent.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);
                if (!BSFilter.Test( content ) ) return;
            }

            InputBox.AddHistoryItem();
            State.DoSendInput();
        }

        private void ListBox_DoubleTapped(object sender, DoubleTappedRoutedEventArgs e) {
            FrameworkElement fe = e.OriginalSource as FrameworkElement;
            if ( fe != null ) {
                State.UserDoubleTapped( fe.DataContext );
            }
        }

        public void OnClear() {
            ChannelView.Blocks.Clear();
        }

        private void InputBox_HeightWillGrow(object sender) {
            CV.ScrollToBottomNowAndThen( 100 );
        }

        private void UserList_ContextCommand(object sender, RoutedEventArgs e) {
            FrameworkElement fe = e.OriginalSource as FrameworkElement;
            if (fe != null) {
                State.UserContextCommand(fe.DataContext, (string) fe.Tag);
            }
        }
    }



    public class IRCWindowState : IMessageWindow, IDisposable, INotifyPropertyChanged {

        private class ACSourceImpl : IAutoCompleteSource {
            private WeakReference<IRCWindowState> m_owner;

            public ACSourceImpl( IRCWindowState owner ) {
                m_owner = new WeakReference<IRCWindowState>(owner);
            }

            LinkedList<string> IAutoCompleteSource.GetACStrings() {
                IRCWindowState s;
                if (!m_owner.TryGetTarget(out s)) return null;
                return s.Connection.GetACStringsFor( s.Context );
            }
            string IAutoCompleteSource.GetACChanTypes() {
                IRCWindowState s;
                if (!m_owner.TryGetTarget(out s)) return "";
                return s.Connection.GetACChanTypes();
            }
        }

        private int m_lineCountLimit;

        public bool UserListVisible { get; set; }
        public bool TopicVisible { get; set; }
        private bool m_topicReadOnly = false;
        public bool TopicReadOnly { get { return m_topicReadOnly; } set { m_topicReadOnly = value; NotifyPropertyChanged("TopicReadOnly"); } }
        private string m_topic = "";
        public string Topic { get { return m_topic; } set { m_topic = value; NotifyPropertyChanged("Topic"); } }

        private bool m_opCommandsEnabled;
        public bool OpCommandsEnabled {
            get { return m_opCommandsEnabled; }
            set { m_opCommandsEnabled = value; NotifyPropertyChanged("OpCommandsEnabled"); }
        }

        public IAutoCompleteSource ACSource {
            get { return new ACSourceImpl(this); }
        }

        public ObservableCollection<object> UserList { get; set; } = new ObservableCollection<object>();

        public event PropertyChangedEventHandler PropertyChanged;
        protected void NotifyPropertyChanged(String propertyName = "") {
            if ( PropertyChanged != null ) {
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        public IRCWindowState(MainPage _main, string _context) {
            m_ownerMainPage = new WeakReference<MainPage>(_main);
            m_context = _context;
            _main.AddPage( _context, typeof(IRCWindow), this);
            PreferencesManager.Notify += PreferencesManager_Notify;
            PreferencesManager_Notify( PreferencesManager.Current );

            MenuItemShowLogs = MakeMenuCmd( "Browse logs", () => { ShowLogs(); } );
            MenuItemShowLogs.IsEnabled = false;
        }

        private void PreferencesManager_Notify(PreferencesData newData) {
            m_lineCountLimit = newData.lineCountLimit;
            applyLineCountLimit();
        }

        private void applyLineCountLimit() {
            int limit = Math.Max(m_lineCountLimit, 10);
            
            if ((Int64)ViewContent.Count * 3 / 4 > limit) {
                int numGone = 0;
                while(ViewContent.Count > limit) { ViewContent.RemoveFirst(); ++numGone; }
                if (numGone > 0 && Frontend != null) Frontend.CV.RemoveFirstLines(numGone);
            }
        }

        public virtual void Dispose() {
            PreferencesManager.Notify -= PreferencesManager_Notify;
            MainPage m;
            if (m_ownerMainPage.TryGetTarget( out m ) ) {
                m.RemovePage( this );
            }
        }

        void IMessageWindow.SwitchTo() {
            MainPage m;
            if (m_ownerMainPage.TryGetTarget(out m)) {
                m.SwitchToPage( this );
            }
        }
        public void PrintOutputV2(TextLine line) {
            ViewContent.AddLast( line );
            
            NotifyPropertyChanged("ViewContent");
            if ( Frontend != null ) {
                Frontend.CV.AddLine( line );
            }
            applyLineCountLimit();
        }
        
        void IMessageWindow.Clear() {
            ViewContent.Clear();
            if ( Frontend != null ) Frontend.OnClear();
        }
        void IMessageWindow.Close() {
            Dispose();
        }

        void IMessageWindow.SetLogTarget(object logTarget) {
            m_logTarget = logTarget as Logging.LoggingTargetInfo;
            NotifyPropertyChanged("CanShowLogs");
            MenuItemShowLogs.IsEnabled = CanShowLogs;
        }
        void IMessageWindow.Flash(int flashLevel) {
            OwnerMainPage.SetPageHighlight( this, flashLevel );
        }
        void IMessageWindow.HandleDisconnection() {

        }

        public virtual void /*IMessageWindow.*/SetUserAway(string nick, bool isAway, string reasonOptional) {

        }
        public virtual void /*IMessageWindow.*/HandleNickChange(string oldNick, string newNick, bool clash) {

        }
        public virtual void /*IMessageWindow.*/UserHostHint(string nick, string userHost) {

        }

        protected void SendInput(string input) {
            var content = input.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);
            foreach (string line in content) {
                RunCommand(IRCUtils.Misc.ProcessOutgoing(line));
            }
        }
        public void DoSendInput() {
            SendInput( this.InputBoxContent );
            InputBoxContent = "";
            this.NotifyPropertyChanged("InputBoxContent");
        }
        public virtual void UserRevertTopic() {

        }
        public virtual void UserSetTopic( string newTopic ) {

        }
        public virtual void RunCommand( string command ) {

        }
        public virtual void UserContextCommand(object user, string what ) {

        }
        public virtual void UserDoubleTapped( object user ) {
            UserContextCommand( user, "/query" );
        }

        public string Context {
            get { return m_context; }
            set { m_context = value; NotifyPropertyChanged("Context"); }
        }

        public MainPage OwnerMainPage {
            get {
                MainPage mp;
                if (!m_ownerMainPage.TryGetTarget(out mp)) throw new InvalidOperationException();
                return mp;
            }
        }

        private WeakReference<MainPage> m_ownerMainPage;
        private string m_context;

        public LinkedList<TextLine> ViewContent = new LinkedList<TextLine>();

        public string InputBoxContent {  get; set; } = "";

        public IRCWindow Frontend { get; set; }

        private WeakReference<ServerConnection> m_connection;
        public ServerConnection Connection {
            get {
                ServerConnection c;
                if (!m_connection.TryGetTarget(out c)) throw new InvalidOperationException();
                return c;
            }
            set {
                m_connection = new WeakReference<ServerConnection>(value);
                NotifyPropertyChanged("Connection");
            }
        }
        private Logging.LoggingTargetInfo m_logTarget;

        public bool CanShowLogs {
            get { return m_logTarget != null; }
        }

        public async void ShowLogs() {
            try {
                if ( m_logTarget == null ) return;
                Windows.Storage.StorageFolder folder = m_logTarget.Folder;
                Windows.Storage.StorageFile file = m_logTarget.File;
                var opt = new Windows.System.FolderLauncherOptions();
                opt.ItemsToSelect.Add(file);

                await Windows.System.Launcher.LaunchFolderAsync(folder, opt);
            } catch { }
        }
        public MenuFlyoutItem MenuItemShowLogs;

        protected static MenuFlyoutItem MakeMenuCmd(string title, Action action) {
            var item = new MenuFlyoutItem();
            item.Text = title;
            item.Click += (System.Object sender, RoutedEventArgs e) => { action(); };
            return item;
        }

        public virtual void ContextMenuHook(MenuFlyout menu) { }
        public virtual MenuFlyout ContextMenu {
            get {
                var menu = new MenuFlyout();
                menu.Items.Add(MenuItemShowLogs);
                ContextMenuHook(menu);
                menu.Items.Add( new MenuFlyoutSeparator() );
                menu.Items.Add(MakeMenuCmd("Close", () => { Dispose(); } ) );
                return menu;
            }
        }
    }
}
