﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices.WindowsRuntime;
using Windows.Foundation;
using Windows.Foundation.Collections;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Controls.Primitives;
using Windows.UI.Xaml.Data;
using Windows.UI.Xaml.Input;
using Windows.UI.Xaml.Media;
using Windows.UI.Xaml.Navigation;
using monochrome;
using System.ComponentModel;
using System.Threading.Tasks;
using Windows.Storage.Streams;
using Windows.Storage.Pickers;
using Windows.Storage;

namespace Monochrome_UWP {

    class FileWriterImpl : IDCCFileWriter {

        public FileWriterImpl( IRandomAccessStream stream ) {
            m_stream = stream;
        }

        async Task IDCCFileWriter.Seek(ulong position) {
            m_stream.Seek( position );
        }
        async Task IDCCFileWriter.SetLength(ulong length) {
            m_stream.Size = length;
        }
        async Task<ulong> IDCCFileWriter.GetLength() {
            return m_stream.Size;
        }

        async Task IDCCFileWriter.Write(byte[] buffer, uint length) {
            if ( m_buffer == null || m_buffer.Capacity < length ) {
                m_buffer = new Windows.Storage.Streams.Buffer(length);
            }
            m_buffer.AsStream().Write(buffer, 0, (int) length);
            m_buffer.Length = length;
            await m_stream.WriteAsync( m_buffer );
        }

        void IDisposable.Dispose() {
            m_stream.Dispose();
        }

        Windows.Storage.Streams.Buffer m_buffer;
        IRandomAccessStream m_stream;
    };

    public sealed partial class DCCReceiveWindow : Page {
        public DCCReceiveWindow() {
            this.InitializeComponent();
        }
        protected override void OnNavigatedTo(NavigationEventArgs e) {
            base.OnNavigatedTo(e);
            State = (DCCReceiveState)e.Parameter;
        }
        protected override void OnNavigatedFrom(NavigationEventArgs e) {
            base.OnNavigatedFrom(e);
        }

        public DCCReceiveState State { get; set; }

        private void ButtonClose(object sender, RoutedEventArgs e) {
            IDisposable v = State;
            v.Dispose();
        }

        private async void ButtonBrowse(object sender, RoutedEventArgs e) {
            var f = State.OutputFile;
            if (f != null) {
                try {
                    var folder = await f.GetParentAsync();
                    var opts = new Windows.System.FolderLauncherOptions();
                    opts.ItemsToSelect.Add( f );
                    await Windows.System.Launcher.LaunchFolderAsync(folder,opts);
                } catch { }
            }
        }
        private async void ButtonOpen(object sender, RoutedEventArgs e) {
            var f = State.OutputFile;
            if (f != null) {
                try {
                    await Windows.System.Launcher.LaunchFileAsync(f);
                } catch { }
            }
        }
        private void ButtonAccept(object sender, RoutedEventArgs e) {
            State.OnAccept();
        }
        private void ButtonResume(object sender, RoutedEventArgs e) {
            State.OnResume();
        }
    }

    public sealed class DCCReceiveState : IDCCReceiveDelegate, IDisposable, INotifyPropertyChanged {

        public event PropertyChangedEventHandler PropertyChanged;
        public void NotifyPropertyChanged(String propertyName = "") {
            if (PropertyChanged != null) {
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        WeakReference<MainPage> m_ownerMainPage;
        DCCReceive m_receive;
        
        PreferencesData m_prefs;

        StorageFile m_targetFile = null;

        public DCCReceiveState(MainPage ownerMainPage, ServerConnection connection, DCCDescription desc, UInt64 _fileSize, PreferencesData _prefs, bool _autoAcceptTrigger ) {
            m_prefs = _prefs;
            m_ownerMainPage = new WeakReference<MainPage>(ownerMainPage);
            ownerMainPage.AddPage("Receiving file", typeof(DCCReceiveWindow), this);

            UserName = desc.Nick;
            FileName = desc.FileName;
            FileSize = _fileSize;

            m_receive = new DCCReceive( connection, desc, this, _prefs);

            if (_autoAcceptTrigger) {
                OnAccept(true);
            }
        }

        public async void OnResume() {
            if (m_receive.Running) return;
            var target = await GrabResumeLocation();
            if ( target != null ) {
                Windows.Storage.FileProperties.BasicProperties props;
                try {
                    props = await target.GetBasicPropertiesAsync();
                } catch {
                    return;
                }
                Accepted = true;
                NotifyPropertyChanged("Accepted");

                m_targetFile = target;

                m_receive.InitiateResume( props.Size );;
            }
        }
        async Task<StorageFile> GrabResumeLocation() {
            try {
                FileOpenPicker picker = new FileOpenPicker();
                picker.FileTypeFilter.Add("*");
                picker.SuggestedStartLocation = PickerLocationId.Downloads;
                return await picker.PickSingleFileAsync();
            } catch {
                return null;
            }
        }
        public async void OnAccept(bool wasAA = false) {
            if ( m_receive.Running ) return;
            var target = await GrabTargetLocation(wasAA);
            if ( target != null ) {
                Accepted = true;
                NotifyPropertyChanged("Accepted");
                m_targetFile = target; m_receive.Start();
            }
        }
        async Task<StorageFile> GrabTargetLocation(bool wasAA) {
            if (!m_prefs.xferAskForFileLocation) {
                string outDir = m_prefs.xferDefaultDirectory;
                if (outDir.Length > 0 && Directory.Exists(outDir)) {
                    string path;
                    path = Path.Combine(outDir, FileName);
                    if (!File.Exists(path)) {
                        return await StorageFile.GetFileFromPathAsync( path );
                    }
                    int ext = 1;
                    while (ext < 100) {
                        path = Path.Combine(outDir, Path.GetFileNameWithoutExtension(FileName) + "-" + ext.ToString() + Path.GetExtension(FileName));
                        if (!File.Exists(path)) {
                            return await StorageFile.GetFileFromPathAsync( path );
                        }
                        ++ext;
                    }
                }
            }

            if (wasAA) {
                return null;
            }


            try {
                var picker = new FileSavePicker();
                picker.SuggestedStartLocation = PickerLocationId.Downloads;
                picker.SuggestedFileName = this.FileName;
                var ext = System.IO.Path.GetExtension(this.FileName);
                picker.FileTypeChoices.Add( ext + " files", new List<string>() { ext } );
                return await picker.PickSaveFileAsync();
            } catch {
                return null;
            }
        }

        void IDisposable.Dispose() {
            MainPage mp;
            if (m_ownerMainPage.TryGetTarget(out mp)) {
                mp.RemovePage(this);
            }
            if (m_receive != null) {
                m_receive.Dispose();
                m_receive = null;
            }
        }


        async Task<IDCCFileWriter> IDCCReceiveDelegate.CreateFile() {
            var stream = await m_targetFile.OpenAsync(FileAccessMode.ReadWrite);
            stream.Size = 0;
            return new FileWriterImpl( stream );
        }
        async Task<IDCCFileWriter> IDCCReceiveDelegate.AppendFile() {
            var stream = await m_targetFile.OpenAsync(FileAccessMode.ReadWrite);
            stream.Seek( stream.Size );
            return new FileWriterImpl(stream);
        }

        private bool m_completed = false;
        public bool Completed { get {return m_completed; } }
        async void IDCCReceiveDelegate.OnFinished(Exception e) {
            Running = false;
            if (e == null) {
                m_completed = true; NotifyPropertyChanged("Completed");

                try {
                    if (m_targetFile != null) {
                        var folder = await m_targetFile.GetParentAsync();
                        if ( folder != null ) {
                            CanBrowse = true; NotifyPropertyChanged( "CanBrowse" );
                        }
                    }
                } catch { }
            }
        }

        public string UserName { get; set; } = "";
        public string FileName { get; set; } = "";
        public UInt64 FileSize { get; set; } = 0;

        private string m_statusText = "", m_mainText = "";
        private UInt64 m_progress = 0;
        private UInt32 m_percentDone = 0;

        public string StatusText {
            get { return m_statusText; }
            set {
                if (m_statusText != value) {
                    m_statusText = value; NotifyPropertyChanged("StatusText");
                }
            }
        }
        public string MainText {
            get { return m_mainText; }
            set {
                if (m_mainText != value) {
                    m_mainText = value; NotifyPropertyChanged("MainText");
                }
            }
        }
        private bool m_running = false;
        public string CloseButtonText {
            get {
                return m_running ? "Cancel" : "Close";
            }
        }
        public bool Running {
            get {
                return m_running;
            }
            set {
                if (value != m_running) {
                    m_running = value;
                    NotifyPropertyChanged("Running");
                    NotifyPropertyChanged("CloseButtonText");

                    if ( value ) {
                        CanAccept = false;
                        CanResume = false;
                        NotifyPropertyChanged("CanAccept");
                        NotifyPropertyChanged("CanResume");
                    }
                }
            }
        }

        public UInt64 Progress { get { return m_progress; } }
        public UInt32 PercentDone { get { return m_percentDone; } }

        void IDCCReceiveDelegate.SetProgress(UInt64 bytesTransferred) {
            m_progress = bytesTransferred;
            if (m_progress > FileSize) m_progress = FileSize;
            NotifyPropertyChanged("Progress");
        }
        void IDCCReceiveDelegate.SetPercentDone(UInt32 percentDone) {
            m_percentDone = percentDone;
            NotifyPropertyChanged("PercentDone");
        }
        void IDCCReceiveDelegate.ToggleRunning(bool bRunning) {
            Running = bRunning;
        }
        public String SaveDialogFileName { get; set; }
        public bool CanAcceptResume { get { return Accepted && AutoAcceptChecked; } }
        public bool AutoAcceptChecked { get; set; }
        public bool Accepted { get; set; }
        public bool CanBrowse { get; set; } = false;
        public bool CanAccept { get; set; } = true;
        public bool CanResume { get; set; } = true;

        public StorageFile OutputFile {  get { return m_targetFile; } }
    }
}
