﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using monochrome;
using IRCUtils;
using System.ComponentModel;
using Windows.UI.Xaml.Media;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml;

namespace Monochrome_UWP {

    class ChannelUserData : INotifyPropertyChanged {
        public ChannelUserData( ChannelWindowState s ) {
            m_channelState = new WeakReference<ChannelWindowState>(s);
        }

        public event PropertyChangedEventHandler PropertyChanged;
        protected void NotifyPropertyChanged(String propertyName = "") {
            if (PropertyChanged != null) {
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }
        public string MenuHeader {
            get {
                string ret = Nick;
                string uh = UserHost;
                if ( uh.Length > 0 ) {
                    ret += " (" + uh + ")";
                }
                return ret;
            }
        }

        public Brush TextColor {
            get {
                if (m_isAway) {
                    return new SolidColorBrush( Windows.UI.Colors.Gray );
                }
                return PreferencesForXAML.Instance.BrushColorText;
            }
        }
        public string Nick { get; set; } = "";
        public string Prefix { get; set; } = "";
        public string NickPrefixed { get { return Prefix + Nick; } }
        public string UserHost { get; set; }  = "";
        public DateTime LastActivity;
        private bool m_isAway = false;
        public bool IsAway {
            get {
                return m_isAway;
            }
            set {
                if ( m_isAway == value ) return;
                m_isAway = value;
                NotifyPropertyChanged("IsAway");
                NotifyPropertyChanged("TextColor");
            }
        }
        public void HostChanged( string host ) {
            if ( host != this.UserHost ) {
                this.UserHost = host;
                NotifyPropertyChanged( "UserHost" );
                NotifyPropertyChanged("MenuHeader" );
            }
        }
        public void PrefixChanged( string prefix ) {
            Prefix = prefix;
            NotifyPropertyChanged( "Prefix" );
            NotifyPropertyChanged( "NickPrefixed" );
        }
        public void NickChanged( string newNick ) {
            Nick = newNick;
            NotifyPropertyChanged( "Nick" );
            NotifyPropertyChanged( "NickPrefixed" );
            NotifyPropertyChanged("MenuHeader");
        }
        public string SortValue {
            get {
                string ret = "";
                if ( Prefix.Contains( "@" ) ) ret = "1";
                else if ( Prefix.Contains( "+" ) ) ret = "2";
                else if ( Prefix.Length > 0 ) ret = "4 " + Prefix;
                else ret += "4";
                ret += " ";
                ret += Nick;
                return ret;
            }
        }

        private WeakReference<ChannelWindowState> m_channelState;
        public ChannelWindowState ChannelState {
            get {
                ChannelWindowState s;
                if (!m_channelState.TryGetTarget(out s)) throw new InvalidOperationException();
                return s;
            }
        }
    };

    public sealed class ChannelWindowState : IRCWindowState, IChannelWindow {

        private Dictionary<string, ChannelUserData> m_nickMap = new Dictionary<string, ChannelUserData>();
        private ChannelStatus m_status;
        public ChannelWindowState(ServerWindowState _owner, ChannelStatus _status) : base(_owner.OwnerMainPage, _status.ChannelName) {
            Connection = _owner.Connection;
            m_status = _status;
            UserListVisible = true;
            TopicVisible = true;

            m_status.PropertyChanged += StatusPropertyChanged;
            m_status.UserPrefixChanged += StatusUserPrefxChanged;
            StatusPropertyChangedEx(null);
        }

        public override void Dispose() {
            m_status.PropertyChanged -= StatusPropertyChanged;
            m_status.UserPrefixChanged -= StatusUserPrefxChanged;

            Connection.WindowClosed(Context, this);

            foreach( var list in new LinkedList<BanListState>( m_listWindows.Values ) ) {
                try { list.Dispose(); } catch { }
            }
            base.Dispose();
        }

        private void StatusUserPrefxChanged(string nick, string newPrefix) {
            var obj = resolveNick(nick);
            if (obj != null) {
                obj.PrefixChanged(newPrefix);
                applySort();
            }
        }

        private void StatusPropertyChangedEx(string name) {
            bool didSetTopic = false;
            if (name == null || name == "Topic") {
                this.Topic = m_status.Topic;
                didSetTopic = true;
            }
            if (name == null || name == "CanSetTopic") {
                bool readOnly = !m_status.CanSetTopic;
                TopicReadOnly = readOnly;
                if (readOnly && !didSetTopic) Topic = m_status.Topic;
            }
            if (name == null || name == "HaveOp") {
                OpCommandsEnabled = m_status.HaveOp;
            }
            if (name == null || name == "UserMode") {

            }
            if (name == null || name == "UserNicks") {
                NameListChanged();
            }
        }

        private void StatusPropertyChanged(object sender, PropertyChangedEventArgs e) {
            StatusPropertyChangedEx(e.PropertyName);
        }

        public override void RunCommand(string command) {
            Connection.runChannelCommand(command, this.Context, this);
        }
        public override void UserContextCommand(object user_, string what) {
            var user = user_ as ChannelUserData;
            if (user != null) {
                string strCTCP = "CTCP ";
                if (what.Length > 0 && what[0] == '/') {
                    RunCommand(what + " " + user.Nick);
                } else if (what.StartsWith(strCTCP)) {
                    string arg = what.Substring(strCTCP.Length);
                    RunCommand("/CTCP " + user.Nick + " " + arg);
                }
            }
        }

        public override void UserRevertTopic() {
            Topic = m_status.Topic;
        }
        public override void UserSetTopic(string newTopic) {
            Connection.addCommand(new IRCCommand("TOPIC", new string[] { this.Context, newTopic }));
        }

        private void NameListChanged() {
            var hosts = new Dictionary<string, string>(Nicks.Comparer);
            foreach (ChannelUserData data in UserList) {
                hosts.Add(data.Nick, data.UserHost);
            }
            clearNameList();
            string prefixList = PrefixList;
            foreach (string nick in m_status.UserNicks) {
                string userHost;
                if (!hosts.TryGetValue(nick, out userHost)) userHost = "";

                addNick(nick, m_status.PrefixOfNick(nick), userHost);
            }
            applySort();
        }
        void addNick(string name, string prefix, string userHost) {
            var data = new ChannelUserData(this);
            data.LastActivity = DateTime.UtcNow;
            data.UserHost = userHost;
            data.Nick = name;
            data.Prefix = prefix;
            UserList.Add(data);
            m_nickMap.Remove(name);//for safety
            m_nickMap.Add(name, data);
        }
        void clearNameList() {
            UserList.Clear();
            m_nickMap.Clear();
        }
        void applySort() {
            var sortMe = this.UserList;
            var sorted = sortMe.OrderBy(x => ((ChannelUserData)x).SortValue).ToList();
            for (int i = 0; i < sorted.Count(); i++)
                sortMe.Move(sortMe.IndexOf(sorted[i]), i);
        }
        void IChannelWindow.UserJoined(string userString, bool isSelf) {
            if (!isSelf) {
                string name = Nicks.Extract(userString);
                addNick(name, "", Nicks.ExtractUserHost(userString));
                applySort();
            }
        }
        void IChannelWindow.UserParted(string source, bool isSelf) {
            removeNick(Nicks.Extract(source));
        }
        void IChannelWindow.UserQuit(string source) {
            removeNick(Nicks.Extract(source));
        }
        bool removeNick(string nick) {
            var item = resolveNick(nick);
            if (item != null) {
                m_nickMap.Remove(nick);
                UserList.Remove(item);
                return true;
            } else {
                return false;
            }
        }
        ChannelUserData resolveNickEx(string who) {
            return resolveNick(Nicks.Extract(who));
        }
        ChannelUserData resolveNick(string nick) {
            ChannelUserData item;
            if (!m_nickMap.TryGetValue(nick, out item)) item = null;
            return item;
        }
        bool IChannelWindow.HasNick(string nick) {
            return resolveNick(nick) != null;
        }

        private Dictionary<char,BanListState> m_listWindows = new Dictionary<char, BanListState>();

        void IChannelWindow.OnBanList(char type, IEnumerable<BanDescription> items) {

            BanListState wnd;
            if ( !m_listWindows.TryGetValue( type, out wnd )) {
                wnd = new BanListState( this, type, items );
                m_listWindows.Add( type, wnd );
            } else {
                wnd.UpdateBanList( items );
            }
        }
        public void RemoveList( char type ) {
            m_listWindows.Remove( type );
        }
        void IChannelWindow.HandleMode(IRCEvent ev, ServerModeSpecs modeSpecs) {
            LinkedList<Modes.Mode> modes = Modes.Parse(ev, modeSpecs);
            foreach (Modes.Mode mode in modes) {

                if (mode.param != null) {
                    BanListState wnd;
                    if ( m_listWindows.TryGetValue( mode.action, out wnd ) ) {
                        if (mode.sign) {
                            BanDescription desc = new BanDescription();
                            desc.Mask = mode.param;
                            desc.ByWhom = ev.Source;
                            desc.Date = DateTime.UtcNow;
                            wnd.OnEntryAdded(desc);
                        } else {
                            wnd.OnEntryRemoved(mode.param);
                        }
                    }
                }
            }

        }
        public override void SetUserAway(string nick, bool isAway, string reasonOptional) {
            base.SetUserAway(nick, isAway, reasonOptional);
            var ni = resolveNick(nick);
            if (ni != null) ni.IsAway = isAway;
        }

        public override void /*IMessageWindow.*/HandleNickChange(string oldNick, string newNick, bool clash) {
            base.HandleNickChange(oldNick, newNick, clash);
            var data = resolveNickEx(oldNick);
            if (data != null) {
                data.NickChanged(newNick);
                m_nickMap.Remove(oldNick); m_nickMap.Remove(newNick);
                m_nickMap.Add(newNick, data);
                applySort();
            }
        }

        public override void /*IMessageWindow.*/UserHostHint(string nick, string userHost) {
            base.UserHostHint(nick, userHost);
            var data = this.resolveNickEx(nick);
            if (data != null) data.HostChanged(userHost);
        }

        PrefixMap Prefixes {
            get { return Connection.InstanceData.ModeSpecs.Prefixes; }
        }
        string PrefixList {
            get { return Prefixes.PrefixList; }
        }

        public override void ContextMenuHook(MenuFlyout menu) {
            menu.Items.Add(MakeMenuCmd("Show ban list", () => { Connection.beginShowList(Context, 'b'); } ));
            menu.Items.Add(MakeMenuCmd("Show ban exemption list", () => { Connection.beginShowList(Context, 'e'); }));
        }
    }
}
