﻿using monochrome;
using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Windows.UI;
using Windows.UI.Text;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Documents;
using Windows.UI.Xaml.Media;

namespace Monochrome_UWP {
    public class ChannelViewTools {

        static Color ColorBlend(Color c1, Color c2, double v) {
            return ColorDesc.Blend(ColorDesc.FromPlatformColor(c1), ColorDesc.FromPlatformColor(c2), v).PlatformColor;
        }

        static Color _resolveColor(TextStyle style, Color ForeColor, Color BackColor) {

            switch (style.Type) {
                case TextStyleType.Hyperlink: {
                        bool bDark = ColorDesc.FromPlatformColor(BackColor).IsDark;
                        return bDark ? Colors.AliceBlue : Colors.DarkBlue;
                    }
                case TextStyleType.UserName: {
                        PreferencesData prefs = PreferencesManager.Current;
                        if (prefs.useNickColors) {
                            bool bDark = ColorDesc.FromPlatformColor(BackColor).IsDark;
                            var c = ColorUtils.RandomColorRGB(style.TypeParam, style.Highlight ? 0.8 : 0.6, bDark ? 0.7 : 0.3);
                            return ColorDesc.FromRGB(c).PlatformColor;
                        } else {
                            return prefs.ColorHighlight;
                        }
                    }
                case TextStyleType.CustomColor:
                    return style.CustomColor.PlatformColor;
                default:
                    return ForeColor;
                case TextStyleType.Plain:
                    return style.Highlight ? PreferencesManager.Current.ColorHighlight : ForeColor;
                case TextStyleType.Gray:
                    return style.Highlight ? ColorBlend(SystemColors.GrayText, ForeColor, 0.4) : SystemColors.GrayText;
                case TextStyleType.UserJoin: {
                        var c = PreferencesManager.Current.ColorUserJoin;
                        return style.Highlight ? ColorBlend(c, ForeColor, 0.4) : c;
                    }
                case TextStyleType.UserPart: {
                        var c = PreferencesManager.Current.ColorUserPart;
                        return style.Highlight ? ColorBlend(c, ForeColor, 0.4) : c;
                    }
            }
        }
        static Inline TextToInline(TextLineElem elem) {
            var r = new Run();
            r.Text = elem.Text;
            if (elem.Style.Italic) r.FontStyle |= FontStyle.Italic;
            if (elem.Style.Bold) r.FontWeight = FontWeights.Bold;

            var prefs = PreferencesManager.Current;
            
            var foreground = prefs.ColorText;
            var background = prefs.ColorBackground;

            r.Foreground = new SolidColorBrush(_resolveColor(elem.Style, foreground, background));

            if (elem.Style.Type == TextStyleType.Hyperlink) {
                var hl = new Hyperlink();
                string linkContent = elem.Text;
                hl.Click += async (Hyperlink, HyperlinkClickEventArgs) => {
                    try {
                        await Windows.System.Launcher.LaunchUriAsync(new Uri(linkContent));
                    } catch { }
                };
                hl.Inlines.Add(r);
                return hl;
            }

            return r;
        }
        static Paragraph TextToParagraph(TextLine tx) {
            Paragraph p = new Paragraph();

            foreach (var elem in tx.Elements) {
                var r = TextToInline(elem);
                p.Inlines.Add(r);
            }

            return p;
        }

        public enum SetTextOptions {
            None = 0,
            RetainSelection = 1 << 0,
            ScrollToBottom = 1 << 1,
        };

        public static void AddLine(RichTextBlock richTextBlock, TextLine line, SetTextOptions options) {
            var s1 = richTextBlock.SelectionStart;
            var s2 = richTextBlock.SelectionEnd;

            richTextBlock.Blocks.Add(TextToParagraph(line));

            if ((options & SetTextOptions.RetainSelection) != 0 && s1 != null && s2 != null) {
                Func<Task> temp = async () => { await Task.Delay(50); richTextBlock.Select(s1, s2); };
                temp();
            }
        }
    }

    public sealed class ChannelViewHooks {
        public ChannelViewHooks(RichTextBlock cv, ScrollViewer cvs) {
            ChannelView = cv;
            ChannelViewScroll = cvs;

            cvs.ViewChanged += this.ChannelViewScroll_ViewChanged;
            cvs.ViewChanging += this.ChannelViewScroll_ViewChanging;
        }

        public void ScrollToBottomNowAndThen(int delayMS) {
            ScrollToBottom();
            // Because sometimes once is not good enough
            Func<Task> temp = async () => { await Task.Delay(delayMS); ScrollToBottom(); };
            temp();
        }

        public bool IsAtBottom {
            get {
                var sv = ChannelViewScroll;
                //                Debug.WriteLine("IsAtBottom :: " + sv.ScrollableHeight + " : " + sv.VerticalOffset);
                return Math.Abs(sv.ScrollableHeight - sv.VerticalOffset) < 0.5;
            }
        }

        public void ScrollToBottom() {
            try {
                m_stickToBottom = true;
                var sv = ChannelViewScroll;
                var pos = sv.ScrollableHeight;
                // Debug.WriteLine("ScrollToBottom : " + sv.VerticalOffset + " => " + pos);
                sv.ChangeView(0, pos, 1);
            } catch { }
        }
        public void Initialize(IEnumerable<TextLine> lines) {
            ChannelView.Blocks.Clear();
            foreach (TextLine tx in lines) {
                AddLine(tx);
            }
            ScrollToBottom();
        }

        public void RemoveFirstLines(int num) {
            var blocks = ChannelView.Blocks;
            if (num >= blocks.Count) blocks.Clear();
            else for (int i = 0; i < num; ++i) {
                    blocks.RemoveAt(0);
                }
        }

        public void AddLine(TextLine tx) {
            tx = tx.DetectHyperlinks();
            // Debug.WriteLine( " > AddLine");
            var sticky = m_stickToBottom || !m_scrollInProgress;
            var options = ChannelViewTools.SetTextOptions.RetainSelection;
            if (sticky) options |= ChannelViewTools.SetTextOptions.ScrollToBottom;
            ChannelViewTools.AddLine(this.ChannelView, tx, options);
            if (sticky) {
                // Debug.WriteLine("AddLine => ScrollToBottom");
                ScrollToBottomNowAndThen(50);
            }
            // Debug.WriteLine(" < AddLine");
        }
        public void Clear() {
            ChannelView.Blocks.Clear();
        }

        private void ChannelViewScroll_ViewChanged(object sender, ScrollViewerViewChangedEventArgs e) {
            var sv = ChannelViewScroll;
            // Debug.WriteLine("ViewChanged: " + sv.VerticalOffset + " / " + sv.ScrollableHeight + ", intermediate: " + e.IsIntermediate);
            m_scrollInProgress = e.IsIntermediate;
            if (!m_scrollInProgress) {
                if (IsAtBottom) {
                    m_stickToBottom = true;
                    // Debug.WriteLine("ViewChanged setting stick-to-bottom");
                }
            }
        }

        private void ChannelViewScroll_ViewChanging(object sender, ScrollViewerViewChangingEventArgs e) {
            m_scrollInProgress = true;

            // Debug.WriteLine("ViewChanging");

            // var finalOffset = e.FinalView.VerticalOffset;
            if (m_stickToBottom) {
                var nextOffset = e.NextView.VerticalOffset;
                var currentOffset = ChannelViewScroll.VerticalOffset;
                //            Debug.WriteLine("ViewChanging: " + currentOffset + " " + nextOffset + " " + finalOffset);

                if (nextOffset < currentOffset) {
                    // up
                    m_stickToBottom = false;
                    // Debug.WriteLine("ViewChanging resetting stick-to-bottom");
                }
            }

            /*
            if (m_stickToBottom) {
                m_stickToBottom = false;
                var sv = ChannelViewScroll;
                Debug.WriteLine("ViewChanging clearing stick-to-bottom: " + sv.VerticalOffset + " / " + sv.ScrollableHeight);
            }*/


        }
        bool m_scrollInProgress = false;
        bool m_stickToBottom = false;

        RichTextBlock ChannelView;
        ScrollViewer ChannelViewScroll;
    }
}